const fs = require('fs-extra');
const path = require('path');
const os = require('os');
const { createArchive } = require('../src/archive');
const { restoreArchive, fixPaths } = require('../src/restore');

async function runTest() {
  const testRoot = path.join(__dirname, '../test-run/path-healing-test');
  const sourceDir = path.join(testRoot, 'source');
  const restoreDir = path.join(testRoot, 'restored');
  const archivePath = path.join(testRoot, 'test.oca');
  const password = 'test-password';

  console.log('🚀 Starting Path Healing Test...');

  // 1. Prepare Mock Source
  await fs.emptyDir(testRoot);
  await fs.ensureDir(path.join(sourceDir, '.openclaw'));
  
  const mockConfig = {
    agents: {
      defaults: {
        workspace: "/Users/olduser/clawd",
        logs: "/Users/olduser/clawd/logs"
      }
    },
    skills: {
      custom: "/Users/olduser/my-skills"
    }
  };
  await fs.writeJson(path.join(sourceDir, '.openclaw/openclaw.json'), mockConfig);
  await fs.ensureDir(path.join(sourceDir, 'clawd'));
  await fs.writeFile(path.join(sourceDir, 'clawd/MEMORY.md'), '# Test');

  // 2. Create Archive
  // We need to temporarily mock os.homedir() for the archive to think it's 'olduser'
  // But since we can't easily mock os.homedir in a child process without complex tools,
  // we will manually edit the manifest after archive creation to simulate a foreign archive.
  console.log('📦 Creating archive...');
  await createArchive([sourceDir, path.join(sourceDir, '.openclaw')], archivePath, password);

  // 3. Restore and Fix
  console.log('🔓 Restoring archive...');
  await fs.ensureDir(restoreDir);
  await restoreArchive(archivePath, restoreDir, password);

  // Manually inject a foreign 'home' into manifest.json to trigger healing
  const manifestPath = path.join(restoreDir, 'manifest.json');
  const manifest = await fs.readJson(manifestPath);
  manifest.home = "/Users/olduser"; 
  await fs.writeJson(manifestPath, manifest);

  console.log('🔧 Running fixPaths...');
  await fixPaths(restoreDir);

  // 4. Verify
  const fixedConfig = await fs.readJson(path.join(restoreDir, '.openclaw/openclaw.json'));
  const currentHome = os.homedir();
  
  console.log('\nVerification:');
  console.log(`Current Home: ${currentHome}`);
  console.log(`Fixed Workspace: ${fixedConfig.agents.defaults.workspace}`);
  
  const isHealed = fixedConfig.agents.defaults.workspace.includes(currentHome) && 
                   fixedConfig.skills.custom.includes(currentHome);

  if (isHealed) {
    console.log('\n✅ TEST PASSED: All paths healed successfully.');
  } else {
    console.log('\n❌ TEST FAILED: Paths were not correctly healed.');
    console.log(JSON.stringify(fixedConfig, null, 2));
    process.exit(1);
  }
}

runTest().catch(err => {
  console.error(err);
  process.exit(1);
});
