#!/usr/bin/env node
const { Command } = require('commander');
const path = require('path');
const fs = require('fs-extra');
const { createArchive } = require('./archive');
const { restoreArchive, fixPaths } = require('./restore');

const program = new Command();

// Helper to get password
function getPassword(options) {
    if (options.password) return options.password;
    if (process.env.MIGRATOR_PASSWORD) return process.env.MIGRATOR_PASSWORD;
    console.error("❌ Error: Password required. Use --password or set MIGRATOR_PASSWORD env var.");
    process.exit(1);
}

program
  .name('migrator')
  .description('Securely migrate OpenClaw agents.')
  .version('0.1.0');

program.command('export')
  .description('Export agent state to an encrypted archive')
  .option('-o, --output <path>', 'Output archive path', 'agent-backup.oca')
  .option('-p, --password <string>', 'Encryption password')
  .option('--source <paths...>', 'Source directories', [
      path.join(process.env.HOME, '.openclaw'),
      path.join(process.env.HOME, 'clawd')
  ])
  .action(async (options) => {
      try {
          const password = getPassword(options);
          const outputPath = path.resolve(options.output);
          
          console.log(`📦 Archiving sources: ${options.source.join(', ')}`);
          
          await createArchive(options.source, outputPath, password);
          console.log(`✅ Export successful: ${outputPath}`);
      } catch (e) {
          console.error(`❌ Export failed: ${e.message}`);
          process.exit(1);
      }
  });

program.command('import')
  .description('Restore agent state from an archive')
  .requiredOption('-i, --input <path>', 'Input archive path')
  .option('-p, --password <string>', 'Decryption password')
  .option('-d, --dest <path>', 'Destination directory (defaults to HOME)', process.env.HOME)
  .action(async (options) => {
      try {
          const password = getPassword(options);
          const inputPath = path.resolve(options.input);
          const destDir = path.resolve(options.dest);
          
          if (!fs.existsSync(inputPath)) {
              throw new Error(`Archive not found: ${inputPath}`);
          }
          
          await fs.ensureDir(destDir);
          console.log(`🔓 Restoring to: ${destDir}`);
          
          await restoreArchive(inputPath, destDir, password);
          await fixPaths(destDir);
          
          console.log(`✅ Import successful.`);
      } catch (e) {
          console.error(`❌ Import failed: ${e.message}`);
          process.exit(1);
      }
  });

program.parse();
