---
name: microsoft-to-do
description: |
  Microsoft To Do API integration with managed OAuth. Manage task lists, tasks, checklist items, and linked resources.
  Use this skill when users want to create, read, update, or delete tasks and task lists in Microsoft To Do.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    requires:
      env:
        - MATON_API_KEY
---

# Microsoft To Do

Access the Microsoft To Do API with managed OAuth authentication. Manage task lists, tasks, checklist items, and linked resources with full CRUD operations.

## Quick Start

```bash
# List all task lists
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/microsoft-to-do/v1.0/me/todo/lists')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/microsoft-to-do/{native-api-path}
```

Replace `{native-api-path}` with the actual Microsoft Graph API endpoint path. The gateway proxies requests to `graph.microsoft.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Microsoft To Do OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=microsoft-to-do&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'microsoft-to-do'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "21fd90f9-5935-43cd-b6c8-bde9d915ca80",
    "status": "ACTIVE",
    "creation_time": "2025-12-08T07:20:53.488460Z",
    "last_updated_time": "2026-01-31T20:03:32.593153Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "microsoft-to-do",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Microsoft To Do connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/microsoft-to-do/v1.0/me/todo/lists')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '21fd90f9-5935-43cd-b6c8-bde9d915ca80')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Task List Operations

#### List Task Lists

```bash
GET /microsoft-to-do/v1.0/me/todo/lists
```

**Response:**
```json
{
  "value": [
    {
      "id": "AAMkADIyAAAhrbPWAAA=",
      "displayName": "Tasks",
      "isOwner": true,
      "isShared": false,
      "wellknownListName": "defaultList"
    }
  ]
}
```

#### Get Task List

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}
```

#### Create Task List

```bash
POST /microsoft-to-do/v1.0/me/todo/lists
Content-Type: application/json

{
  "displayName": "Travel items"
}
```

**Response (201 Created):**
```json
{
  "id": "AAMkADIyAAAhrbPWAAA=",
  "displayName": "Travel items",
  "isOwner": true,
  "isShared": false,
  "wellknownListName": "none"
}
```

#### Update Task List

```bash
PATCH /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}
Content-Type: application/json

{
  "displayName": "Vacation Plan"
}
```

#### Delete Task List

```bash
DELETE /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}
```

Returns `204 No Content` on success.

### Task Operations

#### List Tasks

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks
```

**Response:**
```json
{
  "value": [
    {
      "id": "AlMKXwbQAAAJws6wcAAAA=",
      "title": "Buy groceries",
      "status": "notStarted",
      "importance": "normal",
      "isReminderOn": false,
      "createdDateTime": "2024-01-15T10:00:00Z",
      "lastModifiedDateTime": "2024-01-15T10:00:00Z",
      "body": {
        "content": "",
        "contentType": "text"
      },
      "categories": []
    }
  ]
}
```

#### Get Task

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}
```

#### Create Task

```bash
POST /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks
Content-Type: application/json

{
  "title": "A new task",
  "importance": "high",
  "status": "notStarted",
  "categories": ["Important"],
  "dueDateTime": {
    "dateTime": "2024-12-31T17:00:00",
    "timeZone": "Eastern Standard Time"
  },
  "startDateTime": {
    "dateTime": "2024-12-01T08:00:00",
    "timeZone": "Eastern Standard Time"
  },
  "isReminderOn": true,
  "reminderDateTime": {
    "dateTime": "2024-12-01T09:00:00",
    "timeZone": "Eastern Standard Time"
  },
  "body": {
    "content": "Task details here",
    "contentType": "text"
  }
}
```

**Task Fields:**

| Field | Type | Description |
|-------|------|-------------|
| `title` | String | Brief description of the task |
| `body` | itemBody | Task body with content and contentType (text/html) |
| `importance` | String | `low`, `normal`, or `high` |
| `status` | String | `notStarted`, `inProgress`, `completed`, `waitingOnOthers`, `deferred` |
| `categories` | String[] | Associated category names |
| `dueDateTime` | dateTimeTimeZone | Due date and time |
| `startDateTime` | dateTimeTimeZone | Start date and time |
| `completedDateTime` | dateTimeTimeZone | Completion date and time |
| `reminderDateTime` | dateTimeTimeZone | Reminder date and time |
| `isReminderOn` | Boolean | Whether reminder is enabled |
| `recurrence` | patternedRecurrence | Recurrence pattern |

#### Update Task

```bash
PATCH /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}
Content-Type: application/json

{
  "status": "completed",
  "completedDateTime": {
    "dateTime": "2024-01-20T15:00:00",
    "timeZone": "UTC"
  }
}
```

#### Delete Task

```bash
DELETE /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}
```

Returns `204 No Content` on success.

### Checklist Item Operations

Checklist items are subtasks within a task.

#### List Checklist Items

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/checklistItems
```

**Response:**
```json
{
  "value": [
    {
      "id": "51d8a471-2e9d-4f53-9937-c33a8742d28f",
      "displayName": "Create draft",
      "createdDateTime": "2024-01-17T05:22:14Z",
      "isChecked": false
    }
  ]
}
```

#### Create Checklist Item

```bash
POST /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/checklistItems
Content-Type: application/json

{
  "displayName": "Final sign-off from the team"
}
```

#### Update Checklist Item

```bash
PATCH /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/checklistItems/{checklistItemId}
Content-Type: application/json

{
  "isChecked": true
}
```

#### Delete Checklist Item

```bash
DELETE /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/checklistItems/{checklistItemId}
```

Returns `204 No Content` on success.

### Linked Resource Operations

Linked resources connect tasks to external items (e.g., emails, files).

#### List Linked Resources

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/linkedResources
```

**Response:**
```json
{
  "value": [
    {
      "id": "f9cddce2-dce2-f9cd-e2dc-cdf9e2dccdf9",
      "webUrl": "https://example.com/item",
      "applicationName": "MyApp",
      "displayName": "Related Document",
      "externalId": "external-123"
    }
  ]
}
```

#### Create Linked Resource

```bash
POST /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/linkedResources
Content-Type: application/json

{
  "webUrl": "https://example.com/item",
  "applicationName": "MyApp",
  "displayName": "Related Document",
  "externalId": "external-123"
}
```

#### Delete Linked Resource

```bash
DELETE /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks/{taskId}/linkedResources/{linkedResourceId}
```

Returns `204 No Content` on success.

## Pagination

Microsoft Graph uses OData pagination. Use `$top` to limit results and `$skip` for offset:

```bash
GET /microsoft-to-do/v1.0/me/todo/lists/{todoTaskListId}/tasks?$top=10&$skip=0
```

Response includes `@odata.nextLink` when more results exist:

```json
{
  "value": [...],
  "@odata.nextLink": "https://graph.microsoft.com/v1.0/me/todo/lists/{id}/tasks?$skip=10"
}
```

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/microsoft-to-do/v1.0/me/todo/lists',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const data = await response.json();
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/microsoft-to-do/v1.0/me/todo/lists',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
data = response.json()
```

## Notes

- Task list IDs and task IDs are opaque strings (e.g., `AAMkADIyAAAhrbPWAAA=`)
- Timestamps use ISO 8601 format in UTC by default
- The `dateTimeTimeZone` type requires both `dateTime` and `timeZone` fields
- `wellknownListName` can be `defaultList`, `flaggedEmails`, or `none`
- Task `status` values: `notStarted`, `inProgress`, `completed`, `waitingOnOthers`, `deferred`
- Task `importance` values: `low`, `normal`, `high`
- Supports OData query parameters: `$select`, `$filter`, `$orderby`, `$top`, `$skip`
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Microsoft To Do connection or invalid request |
| 401 | Invalid or missing Maton API key |
| 404 | Resource not found |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Microsoft Graph API |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `microsoft-to-do`. For example:

- Correct: `https://gateway.maton.ai/microsoft-to-do/v1.0/me/todo/lists`
- Incorrect: `https://gateway.maton.ai/v1.0/me/todo/lists`

## Resources

- [Microsoft To Do API Overview](https://learn.microsoft.com/en-us/graph/api/resources/todo-overview)
- [todoTaskList Resource](https://learn.microsoft.com/en-us/graph/api/resources/todotasklist)
- [todoTask Resource](https://learn.microsoft.com/en-us/graph/api/resources/todotask)
- [checklistItem Resource](https://learn.microsoft.com/en-us/graph/api/resources/checklistitem)
- [linkedResource Resource](https://learn.microsoft.com/en-us/graph/api/resources/linkedresource)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
