---
name: metals-desk-os
description: Institutional Desk-Level Fully Automated Trading OS for XAU/USD and XAG/USD. Event-driven, risk-first, multi-engine architecture that runs as a continuous analysis and execution pipeline inside OpenClaw's trader agent.
---

# Metals Desk OS

An institutional-grade metals trading operating system for gold (XAU/USD) and silver (XAG/USD). This skill transforms the OpenClaw trader agent into a full prop-desk assistant with real-time analysis, automated execution, risk management, and performance tracking.

## When to Use This Skill

Use this skill when:
- Analyzing gold or silver market structure (ICT/SMC methodology)
- Generating trade signals with full entry/SL/TP plans
- Running automated or semi-automated trade execution via MT5
- Monitoring real-time risk, drawdown, and position management
- Tracking trading performance (win rate, expectancy, Sharpe, drawdown)
- Getting session-aware, macro-aware trading intelligence

## Architecture Overview

The system runs as a continuous event-driven pipeline:

```
PRICE FEED → SESSION ENGINE → STRUCTURE ENGINE → LIQUIDITY ENGINE →
MACRO ENGINE → BIAS ENGINE → VOLATILITY ENGINE → RISK ENGINE →
EXECUTION ENGINE → BROKER → PERFORMANCE ENGINE → DASHBOARD → ALERTS
```

All engines communicate through a central **Event Bus** (EventEmitter pattern). Every event is logged and can be monitored in real-time via WebSocket.

## Core Engines

| Engine | Purpose | Key Outputs |
|--------|---------|-------------|
| **Structure Engine** | Detects HH/HL/LH/LL, BOS, CHoCH, FVG, Order Blocks | Market structure, trend direction, structural shifts |
| **Liquidity Engine** | Finds equal highs/lows, stop clusters, session levels, sweeps | Liquidity pools, sweep confirmations |
| **Bias Engine** | Multi-timeframe directional bias with conviction scoring | HTF bias, intraday bias, conviction 0-100, alignment score |
| **Macro Engine** | Tracks DXY, US10Y yields, risk sentiment, news proximity | macro_bias (bullish/bearish/neutral gold), macro_risk level |
| **Volatility Engine** | ATR calculation, spike detection, regime classification | ATR, volatility regime (low/normal/high/extreme) |
| **Risk Engine** | Position sizing, drawdown monitoring, halt triggers | Lot size, risk %, pass/fail gate for every trade |
| **Execution Engine** | Combines all engines to generate trade signals | Full trade plan: entry, SL, TP1/TP2/TP3, position size |
| **Performance Engine** | Win rate, expectancy, Sharpe, monthly breakdown | Comprehensive performance metrics and trade log |

## System Modes

| Mode | Name | Behavior |
|------|------|----------|
| 1 | **Advisory** | Full analysis pipeline runs, signals logged, no execution |
| 2 | **Semi-Automated** | Signals generated + automatic TP/SL/trailing management |
| 3 | **Fully-Automated** | Autonomous execution: signal → order → manage → close |
| 4 | **Risk-Off** | Monitoring only, no signals generated |

Set the mode in `data/state.json` or via WebSocket command. Always start in Mode 1 for validation.

## Risk Rules (Non-Negotiable)

These hard rules are enforced by the Risk Engine and cannot be overridden:

- **Max 2% risk per trade** — Position sized via `(Balance × Risk%) / StopDistance`
- **Max 5% daily exposure** — Cumulative open risk capped
- **3 consecutive losses → halt trading** for the session
- **Volatility spike → reduce size 50%** (extreme = 75% reduction)
- **Spread widening → block entry** (symbol-specific thresholds)
- **High-impact news < 20 minutes → block all entries**
- **Equity drawdown > 8% → halt trading** until manual resume
- **API error on order → cancel, do NOT retry**
- **Broker disconnect → close all positions immediately**

## Trade Management Protocol

1. **TP1 at 1.5R** → Partial close 40% → Move SL to break-even
2. **TP2 at 2.5R** → Partial close 30% → Trail SL under structure
3. **TP3 at 4.0R** → Close remaining 30% → Log trade to performance

## Entry Conditions (All Must Pass)

For the Execution Engine to generate a signal, every condition must be true:
1. Bias conviction ≥ 65/100
2. Alignment score ≥ 0.6
3. Liquidity sweep confirmed (aligned with direction)
4. BOS or CHoCH confirmed in trade direction
5. Session = London or New York (kill zones preferred)
6. Macro risk = low or medium
7. Spread within threshold
8. No news block active
9. Daily loss below 80% of limit
10. Consecutive losses < 3

## File Structure

```
metals-desk-os/
├── SKILL.md                    # This file
├── index.js                    # Main orchestrator & entry point
├── package.json                # npm dependencies
├── skill.json                  # OpenClaw registration manifest
├── manifest.json               # Runtime configuration
├── core/                       # 8 trading engines
│   ├── structure-engine.js     # HH/HL/LH/LL, BOS, CHoCH, FVG, OB
│   ├── liquidity-engine.js     # Pools, sweeps, equal levels
│   ├── bias-engine.js          # MTF bias, conviction scoring
│   ├── macro-engine.js         # DXY, yields, sentiment, news
│   ├── volatility-engine.js    # ATR, regime, spikes
│   ├── risk-engine.js          # Position sizing, halts
│   ├── execution-engine.js     # Signal generation
│   └── performance-engine.js   # Metrics tracking
├── automation/                 # System automation layer
│   ├── event-bus.js            # Central event system
│   ├── price-feed.js           # MT5/MetaAPI price data
│   ├── session-engine.js       # London/NY/Asian sessions
│   ├── scheduler.js            # Cron tasks
│   └── news-monitor.js         # Economic calendar
├── broker/                     # MT5 broker integration
│   ├── mt5-connector.js        # Order execution via MetaAPI
│   ├── risk-guard.js           # Position monitoring & trailing
│   └── order-manager.js        # Order lifecycle management
├── dashboard/                  # Real-time monitoring
│   ├── websocket-feed.js       # WebSocket broadcaster (port 3078)
│   ├── desk-dashboard.json     # Widget layout config
│   └── metrics.json            # Metrics template
├── alerts/                     # Notification channels
│   ├── whatsapp-alert.js       # WhatsApp Business API
│   ├── telegram-alert.js       # Telegram Bot
│   └── risk-alert.js           # Centralized dispatcher
├── data/                       # Persistent state files
│   ├── state.json              # System mode & connections
│   ├── trade-log.json          # Trade history
│   ├── performance.json        # Performance metrics
│   └── bias-memory.json        # Bias state persistence
└── prompts/                    # AI agent prompts
    ├── system.txt              # Main system prompt
    ├── intraday.txt            # Intraday trading protocol
    ├── swing.txt               # Swing trading protocol
    └── execution.txt           # Execution protocol
```

## Installation

### Step 1: Deploy to OpenClaw

```bash
cp -r metals-desk-os/ ~/.openclaw/agents/trader/agent/metals-desk-os/
cd ~/.openclaw/agents/trader/agent/metals-desk-os/
```

### Step 2: Install Dependencies

```bash
npm install
```

### Step 3: Configure Environment

Create a `.env` file in the skill root with the following variables:

```
# MetaAPI / MT5 Connection
METAAPI_TOKEN=your_metaapi_token_here
MT5_ACCOUNT_ID=your_mt5_account_id_here

# WhatsApp Business API (optional)
WHATSAPP_API_URL=https://graph.facebook.com/v18.0/YOUR_PHONE_NUMBER_ID/messages
WHATSAPP_TOKEN=your_whatsapp_token_here
WHATSAPP_PHONE=your_phone_number_with_country_code

# Telegram Bot (optional)
TELEGRAM_BOT_TOKEN=your_telegram_bot_token_here
TELEGRAM_CHAT_ID=your_telegram_chat_id_here

# News API (optional)
NEWS_API_KEY=

# AI Keys (optional, for enhanced analysis)
OPENROUTER_API_KEY=
ANTHROPIC_API_KEY=
```

**MetaAPI**: Sign up at https://metaapi.cloud and connect your Fusion Markets MT5 account.
**Telegram**: Create a bot via @BotFather, get chat ID from @userinfobot.
**WhatsApp**: Configure via Meta Business API dashboard.

### Step 4: Set Initial Mode

Edit `data/state.json` and set `"mode": 1` (Advisory) to start safely.

### Step 5: Start

```bash
# Direct
node index.js

# Or with PM2 for production
pm2 start index.js --name metals-desk-os
pm2 save
pm2 startup
```

## Dashboard

Connect any WebSocket client to `ws://localhost:3078` to receive real-time JSON payloads containing:
- Live prices and spread statistics
- HTF and intraday bias with conviction scores
- Active positions with live P&L and R:R
- Liquidity map (equal highs/lows, sweep markers)
- Macro panel (DXY, yields, news countdown)
- Performance metrics (win rate, expectancy, drawdown, Sharpe)
- Risk status (halt state, daily P&L, exposure)
- Event log (all system events)

## Alert Formats

**Trade opened:**
```
TRADE OPENED
Pair: XAUUSD
Direction: Long
Entry: 5024.50
SL: 5010.00
TP1: 5046.25
Risk: 1.5%
Session: London
Conviction: 82/100
```

**Risk halt:**
```
RISK HALT ACTIVATED
Reason: 3 consecutive losses
Trading paused for session
```

## Key Events

The Event Bus broadcasts these events that external systems can subscribe to:

- `price.update` — New price tick with candle data
- `structure.shift` — BOS or CHoCH detected
- `liquidity.sweep` — Liquidity pool swept
- `bias.update` — Bias recalculated
- `bias.flip` — HTF bias direction changed
- `execution.signal` — Valid trade signal generated
- `risk.halt` / `risk.resume` — Trading halted or resumed
- `order.filled` / `order.closed` — Order lifecycle events
- `performance.update` — New trade recorded to performance
- `macro.news.block` — News event blocking execution

## Validation Checklist

Before moving beyond Mode 1:
1. Price feed produces ticks (live or simulated)
2. Structure engine detects swing points and BOS/CHoCH
3. Liquidity engine identifies pools and sweeps
4. Bias engine produces conviction scores > 0
5. Session engine correctly identifies current session
6. Risk engine validates and rejects test scenarios
7. Execution engine generates signals with full entry plans
8. Alerts arrive on configured channels
9. WebSocket feed broadcasts state updates

Once all pass, move to Mode 2 (Semi-Auto) to test position management, then Mode 3 (Full-Auto) for live execution.
