"""
Video Ad Analyzer - Extract content from video ads using Gemini Vision AI.

Supports:
- Frame extraction with scene change detection
- OCR text detection using EasyOCR
- Audio transcription using Google Cloud Speech
- AI-powered scene analysis using Gemini Vision
- Native video analysis for direct understanding
"""
import cv2
import numpy as np
import os
import logging
import subprocess
import io
import base64
import re
import time
import tempfile
from typing import List, Dict, Optional
from PIL import Image

logger = logging.getLogger(__name__)

# Optional imports with graceful degradation
try:
    import ffmpeg
    FFMPEG_AVAILABLE = True
except ImportError:
    FFMPEG_AVAILABLE = False
    logger.warning("ffmpeg-python not installed. Audio features disabled.")

try:
    from google.cloud import speech
    SPEECH_AVAILABLE = True
except ImportError:
    SPEECH_AVAILABLE = False
    logger.warning("google-cloud-speech not installed. Transcription disabled.")

try:
    import easyocr
    EASYOCR_AVAILABLE = True
except ImportError:
    EASYOCR_AVAILABLE = False
    logger.warning("easyocr not installed. OCR disabled.")

try:
    from vertexai.generative_models import Part, GenerationConfig
    VERTEXAI_AVAILABLE = True
except ImportError:
    VERTEXAI_AVAILABLE = False
    logger.warning("vertexai not installed. AI features disabled.")

from .models import ExtractedVideoContent
from .prompt_manager import PromptManager


class VideoExtractor:
    """
    Extracts content from videos and images for marketing analysis.

    Main entry point: extract_content(file_path)

    Features:
    - Frame extraction with scene change detection
    - OCR text detection (EasyOCR)
    - Audio transcription (Google Cloud Speech)
    - AI scene analysis (Gemini Vision)
    - Native video understanding
    """

    def __init__(self, gemini_model=None, prompts_dir=None):
        """
        Initialize VideoExtractor.

        Args:
            gemini_model: Initialized Gemini GenerativeModel instance
            prompts_dir: Optional path to prompts directory
        """
        self.gemini_model = gemini_model
        self.gemini_available = gemini_model is not None

        # Initialize PromptManager
        self.prompt_manager = PromptManager(prompts_dir)

        # Initialize EasyOCR reader
        self.reader = None
        if EASYOCR_AVAILABLE:
            try:
                self.reader = easyocr.Reader(['en'], gpu=False)
                logger.info("EasyOCR reader initialized")
            except Exception as e:
                logger.warning(f"Failed to initialize EasyOCR: {e}")

    def extract_content(self, file_path: str) -> ExtractedVideoContent:
        """
        Main extraction method - handles both videos and images.

        Args:
            file_path: Path to video or image file

        Returns:
            ExtractedVideoContent with all extracted data
        """
        logger.info(f"Starting content extraction for: {file_path}")

        if self._is_image_file(file_path):
            return self._extract_image_content(file_path)
        else:
            return self._extract_video_content(file_path)

    def _is_image_file(self, file_path: str) -> bool:
        """Check if file is an image."""
        image_extensions = ('.jpg', '.jpeg', '.png', '.webp', '.bmp', '.tiff')
        return file_path.lower().endswith(image_extensions)

    def _get_video_duration(self, video_path: str) -> float:
        """Get video duration using ffprobe."""
        try:
            result = subprocess.run(
                ["ffprobe", "-v", "error", "-show_entries", "format=duration",
                 "-of", "default=noprint_wrappers=1:nokey=1", video_path],
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                text=True
            )
            return float(result.stdout)
        except Exception as e:
            logger.error(f"Error getting video duration: {e}")
            return 0.0

    def extract_smart_frames(self, video_path: str, scene_interval: float = 2,
                             text_interval: float = 0.5):
        """
        Extract frames with scene change detection and OCR.

        Args:
            video_path: Path to video file
            scene_interval: Seconds between scene checks
            text_interval: Seconds between text/OCR checks

        Returns:
            Tuple of (frames, timestamps, text_timeline, scene_timeline, thumbnail_url)
        """
        if not os.path.exists(video_path):
            raise FileNotFoundError(f"Video file not found: {video_path}")

        video = cv2.VideoCapture(video_path)
        if not video.isOpened():
            raise ValueError("Failed to open video file")

        fps = video.get(cv2.CAP_PROP_FPS)
        total_frames = int(video.get(cv2.CAP_PROP_FRAME_COUNT))
        duration = total_frames / fps

        vision_frames = []
        vision_timestamps = []
        all_detected_text = []

        prev_frame = None
        first_frame = None
        last_scene_time = -scene_interval
        frame_count = 0

        logger.info(f"Processing video: {duration:.1f}s duration, {fps:.1f} fps")

        current_time = 0
        while current_time < duration:
            frame_idx = int(current_time * fps)
            video.set(cv2.CAP_PROP_POS_FRAMES, frame_idx)
            ret, frame = video.read()

            if not ret:
                break

            # Save first frame for thumbnail
            if first_frame is None:
                first_frame = frame.copy()

            # Check for scene changes
            if current_time - last_scene_time >= scene_interval:
                if self.detect_scene_change(prev_frame, frame, threshold=65):
                    frame_rgb = cv2.cvtColor(frame, cv2.COLOR_BGR2RGB)
                    pil_image = Image.fromarray(frame_rgb)

                    img_byte_arr = io.BytesIO()
                    pil_image.save(img_byte_arr, format='JPEG')
                    base64_frame = base64.b64encode(img_byte_arr.getvalue()).decode('utf-8')

                    vision_frames.append(base64_frame)
                    vision_timestamps.append(current_time)
                    frame_count += 1

                    # OCR text detection
                    detected_text = self.detect_text_in_frame(frame, current_time)
                    if detected_text:
                        all_detected_text.append((detected_text, current_time))

                    prev_frame = frame

                last_scene_time = current_time

            current_time += text_interval

        video.release()
        logger.info(f"Extracted {frame_count} frames with {len(all_detected_text)} OCR detections")

        # Save thumbnail
        thumbnail_url = self._save_thumbnail(first_frame, video_path)

        # Process OCR text
        ocr_timeline = []
        if all_detected_text:
            for text, time_val in all_detected_text:
                ocr_timeline = self.process_new_text(text, time_val, ocr_timeline)

        # Proofread OCR text
        ocr_timeline = self.proofread_detected_text(ocr_timeline, vision_timestamps)

        # Analyze frames with Gemini Vision
        scene_timeline = self.analyze_frames_batch(vision_frames, vision_timestamps, ocr_timeline)

        # Enrich with native video analysis
        native_scenes = self.analyze_video_natively(video_path)
        if native_scenes:
            scene_timeline = self.reconcile_scenes_with_ai(scene_timeline, native_scenes)

        # Extract and reconcile text
        gemini_text_timeline = self.extract_text_from_scenes(scene_timeline)
        final_text_timeline = self.reconcile_text_sources(ocr_timeline, gemini_text_timeline)

        return vision_frames, vision_timestamps, final_text_timeline, scene_timeline, thumbnail_url

    def detect_scene_change(self, prev_frame, current_frame, threshold: int = 65) -> bool:
        """Detect if scene has changed significantly using histogram difference."""
        if prev_frame is None:
            return True

        prev_gray = cv2.cvtColor(prev_frame, cv2.COLOR_BGR2GRAY)
        curr_gray = cv2.cvtColor(current_frame, cv2.COLOR_BGR2GRAY)

        diff = cv2.absdiff(prev_gray, curr_gray)
        diff_score = np.mean(diff)

        return diff_score > threshold

    def detect_text_in_frame(self, frame, timestamp: float) -> List[str]:
        """Detect text in frame using EasyOCR with confidence thresholds."""
        if not self.reader:
            return []

        try:
            results = self.reader.readtext(frame)
            high_confidence = []
            low_confidence = []

            for (bbox, text, prob) in results:
                clean_text = text.strip()
                if not clean_text:
                    continue

                if prob > 0.5:
                    high_confidence.append(clean_text)
                elif prob > 0.3:
                    low_confidence.append(clean_text)

            # Use low confidence only when high confidence is sparse
            if len(high_confidence) < 3:
                return high_confidence + low_confidence
            return high_confidence

        except Exception as e:
            logger.error(f"Error in OCR: {e}")
            return []

    def process_new_text(self, detected_text, current_time: float, text_timeline: List) -> List:
        """Add detected text to timeline."""
        if not detected_text:
            return text_timeline

        text_timeline.append({
            'at': current_time,
            'text': detected_text if isinstance(detected_text, list) else [detected_text]
        })

        return text_timeline

    def proofread_detected_text(self, text_timeline: List, timestamps: List) -> List:
        """Proofread OCR text using AI while preserving frame associations."""
        if not text_timeline or not self.gemini_model:
            return text_timeline

        try:
            # Flatten with timestamp tracking
            all_texts = []
            timestamp_map = []

            for entry in text_timeline:
                frame_time = entry['at']
                texts = entry.get('text', [])
                if not isinstance(texts, list):
                    texts = [texts]

                for text in texts:
                    if text:
                        all_texts.append(str(text))
                        timestamp_map.append(frame_time)

            if not all_texts:
                return text_timeline

            # Batch proofread
            batch_text = "\n".join([f"{i+1}. {text}" for i, text in enumerate(all_texts)])

            prompt = self.prompt_manager.get_prompt('proofread_ocr_text', {
                'texts': batch_text,
                'count': str(len(all_texts))
            })

            response = self.gemini_model.generate_content(prompt)
            time.sleep(1)  # Rate limit

            # Parse response
            cleaned_texts = []
            for line in response.text.strip().split('\n'):
                line = line.strip()
                if not line:
                    continue
                if line[0].isdigit() and '. ' in line:
                    text = line.split('. ', 1)[1]
                else:
                    text = line
                if text:
                    cleaned_texts.append(text)

            # Restore timeline
            cleaned_timeline = []
            if len(cleaned_texts) == len(timestamp_map):
                for i, text in enumerate(cleaned_texts):
                    cleaned_timeline.append({'at': timestamp_map[i], 'text': [text]})
            else:
                for i, text in enumerate(cleaned_texts):
                    timestamp = timestamp_map[i % len(timestamp_map)]
                    cleaned_timeline.append({'at': timestamp, 'text': [text]})

            return cleaned_timeline

        except Exception as e:
            logger.error(f"Proofreading failed: {e}")
            return text_timeline

    def analyze_frames_batch(self, frames: List[str], timestamps: List[float],
                             text_timeline: List) -> List[Dict]:
        """Analyze frames using Gemini Vision to create scene descriptions."""
        if not frames or not self.gemini_model or not VERTEXAI_AVAILABLE:
            return []

        try:
            prompt_text = self.prompt_manager.get_prompt('frame_analysis')

            content_parts = [Part.from_text(prompt_text)]

            for frame_b64, timestamp in zip(frames, timestamps):
                image_bytes = base64.b64decode(frame_b64)
                content_parts.append(Part.from_text(f"Frame at {timestamp:.2f}s:"))
                content_parts.append(Part.from_data(image_bytes, mime_type="image/jpeg"))

            response = self.gemini_model.generate_content(
                contents=content_parts,
                generation_config=GenerationConfig(
                    temperature=0.2,
                    max_output_tokens=2000,
                    top_p=0.95
                )
            )

            # Parse response
            frame_descriptions = []
            frame_pattern = re.compile(r'Frame\s+(\d+\.\d+)(?:s)?:\s*(.*?)(?=Frame\s+\d+\.\d+|$)', re.DOTALL)
            matches = frame_pattern.findall(response.text)

            for timestamp_str, description in matches:
                try:
                    frame_descriptions.append({
                        "timestamp": float(timestamp_str),
                        "description": description.strip()
                    })
                except Exception:
                    continue

            logger.info(f"Generated {len(frame_descriptions)} scene descriptions")
            return frame_descriptions

        except Exception as e:
            logger.error(f"Error in analyze_frames_batch: {e}")
            return []

    def analyze_video_natively(self, video_path: str) -> List[str]:
        """Analyze video directly using Gemini for motion/flow understanding."""
        if not self.gemini_model or not VERTEXAI_AVAILABLE:
            return []

        try:
            file_size_mb = os.path.getsize(video_path) / (1024 * 1024)
            duration = self._get_video_duration(video_path)

            if duration > 180:
                logger.warning(f"Video too long ({duration:.1f}s), skipping native analysis")
                return []

            # Small files: use Part.from_data()
            if file_size_mb < 20:
                with open(video_path, 'rb') as f:
                    video_bytes = f.read()
                video_part = Part.from_data(data=video_bytes, mime_type="video/mp4")
            else:
                # Large files: try file:// URI
                try:
                    abs_path = os.path.abspath(video_path)
                    video_part = Part.from_uri(uri=f"file://{abs_path}", mime_type="video/mp4")
                except Exception:
                    logger.warning("file:// URI not supported, skipping native analysis")
                    return []

            prompt = self.prompt_manager.get_prompt('native_video_timeline')

            response = self.gemini_model.generate_content(
                [video_part, prompt],
                generation_config={'temperature': 0.2}
            )

            scenes = []
            for line in response.text.strip().split('\n'):
                line = line.strip()
                if line and ':' in line and not line.startswith(('#', 'Example', '-', '**')):
                    scenes.append(line)

            logger.info(f"Native analysis: {len(scenes)} scenes")
            return scenes

        except Exception as e:
            logger.warning(f"Native video analysis failed: {e}")
            return []

    def reconcile_scenes_with_ai(self, frame_scenes: List, native_scenes: List) -> List:
        """Combine frame-based and native scene analysis."""
        if not frame_scenes:
            return []
        if not native_scenes or not self.gemini_model:
            return frame_scenes

        try:
            frame_scenes_text = "\n".join([
                f"{scene['timestamp']:.2f}s: {scene['description']}"
                for scene in frame_scenes
            ])
            native_scenes_text = "\n".join(native_scenes)

            prompt = self.prompt_manager.get_prompt('batch_combine_scene_descriptions', {
                'frame_scenes': frame_scenes_text,
                'native_scenes': native_scenes_text,
                'frame_count': str(len(frame_scenes))
            })

            response = self.gemini_model.generate_content(
                prompt,
                generation_config={'temperature': 0.1, 'max_output_tokens': 2000}
            )
            time.sleep(1)

            # Parse response
            enriched_scenes = []
            for line in response.text.strip().split('\n'):
                line = line.strip()
                if not line or not line[0].isdigit() or ':' not in line:
                    continue

                try:
                    parts = line.split(':', 1)
                    timestamp = float(parts[0].replace('s', '').strip())
                    description = parts[1].strip()
                    if description:
                        enriched_scenes.append({
                            'timestamp': timestamp,
                            'description': description
                        })
                except (ValueError, IndexError):
                    continue

            # Validate
            if len(enriched_scenes) != len(frame_scenes):
                return frame_scenes

            return enriched_scenes

        except Exception as e:
            logger.error(f"Batch enrichment failed: {e}")
            return frame_scenes

    def extract_text_from_scenes(self, scene_timeline: List) -> List:
        """Extract text overlays mentioned in scene descriptions using AI."""
        if not scene_timeline or not self.gemini_model:
            return []

        try:
            scenes_text = []
            for scene in scene_timeline:
                scenes_text.append(f"At {scene.get('timestamp', 0):.2f}s: {scene.get('description', '')}")

            prompt = self.prompt_manager.get_prompt('extract_text_from_scenes', {
                'scenes': "\n".join(scenes_text)
            })

            response = self.gemini_model.generate_content(prompt)

            if "NO TEXT FOUND" in response.text.upper():
                return []

            extracted = []
            for line in response.text.split('\n'):
                if ':' in line and line.strip():
                    try:
                        parts = line.split(':', 1)
                        timestamp = float(parts[0].strip())
                        text = parts[1].strip()
                        if text:
                            extracted.append({'at': timestamp, 'text': [text], 'source': 'gemini_vision'})
                    except (ValueError, IndexError):
                        continue

            return extracted

        except Exception as e:
            logger.error(f"Error extracting text from scenes: {e}")
            return []

    def reconcile_text_sources(self, ocr_timeline: List, gemini_timeline: List) -> List:
        """Merge OCR and Gemini Vision text intelligently using AI."""
        if not self.gemini_model:
            return ocr_timeline + gemini_timeline

        if not gemini_timeline:
            return ocr_timeline
        if not ocr_timeline:
            return gemini_timeline

        try:
            ocr_texts = [f"{e['at']:.2f}s: {' '.join(e.get('text', []))}" for e in ocr_timeline]
            gemini_texts = [f"{e['at']:.2f}s: {' '.join(e.get('text', []))}" for e in gemini_timeline]

            prompt = self.prompt_manager.get_prompt('reconcile_text_sources', {
                'ocr_texts': "\n".join(ocr_texts) if ocr_texts else 'None',
                'gemini_texts': "\n".join(gemini_texts) if gemini_texts else 'None'
            })

            response = self.gemini_model.generate_content(prompt)

            reconciled = []
            for line in response.text.strip().split('\n'):
                if ':' in line and line.strip():
                    try:
                        parts = line.split(':', 1)
                        timestamp = float(parts[0].strip().replace('s', ''))
                        text = parts[1].strip()
                        if text:
                            reconciled.append({'at': timestamp, 'text': [text]})
                    except (ValueError, IndexError):
                        continue

            return reconciled

        except Exception as e:
            logger.error(f"AI reconciliation failed: {e}")
            return ocr_timeline + gemini_timeline

    def _save_thumbnail(self, frame, video_path: str) -> Optional[str]:
        """Save first frame as thumbnail."""
        if frame is None:
            return None

        try:
            thumbnail_dir = os.path.join(os.getcwd(), 'static', 'thumbnails')
            os.makedirs(thumbnail_dir, exist_ok=True)

            video_basename = os.path.splitext(os.path.basename(video_path))[0]
            thumbnail_filename = f"{video_basename}_thumb.jpg"
            thumbnail_path = os.path.join(thumbnail_dir, thumbnail_filename)

            cv2.imwrite(thumbnail_path, frame)
            return f'/static/thumbnails/{thumbnail_filename}'

        except Exception as e:
            logger.error(f"Error saving thumbnail: {e}")
            return None

    def _extract_image_content(self, image_path: str) -> ExtractedVideoContent:
        """Extract content from a static image."""
        logger.info(f"Processing image: {image_path}")

        try:
            with open(image_path, 'rb') as f:
                image_bytes = f.read()
            image_b64 = base64.b64encode(image_bytes).decode('utf-8')

            # OCR text extraction
            text_timeline = []
            if self.reader:
                try:
                    image = Image.open(image_path)
                    results = self.reader.readtext(np.array(image))
                    for (bbox, text, confidence) in results:
                        if confidence > 0.5 and text.strip():
                            text_timeline.append({
                                'at': 0.0,
                                'text': [text.strip()],
                                'source': 'easyocr'
                            })
                except Exception as e:
                    logger.warning(f"OCR extraction failed: {e}")

            # Analyze with Gemini
            frames = [image_b64]
            timestamps = [0.0]
            scene_timeline = self.analyze_frames_batch(frames, timestamps, text_timeline)

            # Create thumbnail
            thumbnail_url = self._create_image_thumbnail(image_path)

            return ExtractedVideoContent(
                video_path=image_path,
                duration=0.0,
                text_timeline=text_timeline,
                scene_timeline=scene_timeline,
                transcript=None,
                thumbnail_url=thumbnail_url,
                extraction_complete=True
            )

        except Exception as e:
            logger.error(f"Error extracting image content: {e}")
            return ExtractedVideoContent(
                video_path=image_path,
                duration=0.0,
                extraction_complete=False
            )

    def _create_image_thumbnail(self, image_path: str) -> Optional[str]:
        """Create thumbnail for image."""
        try:
            thumbnail_dir = os.path.join('static', 'thumbnails')
            os.makedirs(thumbnail_dir, exist_ok=True)

            base_name = os.path.splitext(os.path.basename(image_path))[0]
            thumbnail_filename = f"{base_name}_thumb.jpg"
            thumbnail_path = os.path.join(thumbnail_dir, thumbnail_filename)

            image = Image.open(image_path)
            image.thumbnail((300, 300), Image.Resampling.LANCZOS)
            if image.mode != 'RGB':
                image = image.convert('RGB')
            image.save(thumbnail_path, 'JPEG', quality=85)

            return f'/static/thumbnails/{thumbnail_filename}'

        except Exception as e:
            logger.warning(f"Thumbnail creation failed: {e}")
            return None

    def _extract_video_content(self, video_path: str) -> ExtractedVideoContent:
        """Extract content from video file."""
        logger.info(f"Processing video: {video_path}")

        try:
            frames, timestamps, text_timeline, scene_timeline, thumbnail_url = \
                self.extract_smart_frames(video_path)

            # Transcribe audio
            transcript = None
            if FFMPEG_AVAILABLE and SPEECH_AVAILABLE:
                transcript = self._transcribe_video_audio(video_path)

            return ExtractedVideoContent(
                video_path=video_path,
                duration=self._get_video_duration(video_path),
                text_timeline=text_timeline,
                scene_timeline=scene_timeline,
                transcript=transcript,
                thumbnail_url=thumbnail_url,
                extraction_complete=True
            )

        except Exception as e:
            logger.error(f"Error in extract_content: {e}")
            return ExtractedVideoContent(
                video_path=video_path,
                duration=0.0,
                extraction_complete=False
            )

    def _transcribe_video_audio(self, video_path: str) -> Optional[str]:
        """Extract and transcribe audio from video."""
        if not FFMPEG_AVAILABLE or not SPEECH_AVAILABLE:
            return None

        try:
            key_path = os.environ.get("GOOGLE_APPLICATION_CREDENTIALS")
            if not key_path or not os.path.exists(key_path):
                logger.warning("GOOGLE_APPLICATION_CREDENTIALS not set")
                return None

            client = speech.SpeechClient.from_service_account_file(key_path)

            # Extract audio
            with tempfile.NamedTemporaryFile(suffix=".wav", delete=False) as temp_f:
                audio_path = temp_f.name

            try:
                subprocess.run([
                    'ffmpeg', '-i', video_path, '-vn', '-f', 'wav',
                    '-acodec', 'pcm_s16le', '-ar', '16000', '-ac', '1',
                    '-y', audio_path
                ], check=True, capture_output=True)

                # Transcribe
                with open(audio_path, "rb") as audio_file:
                    content = audio_file.read()

                audio = speech.RecognitionAudio(content=content)
                config = speech.RecognitionConfig(
                    encoding=speech.RecognitionConfig.AudioEncoding.LINEAR16,
                    language_code="en-US",
                    sample_rate_hertz=16000,
                    model="telephony",
                    enable_automatic_punctuation=True,
                )

                response = client.recognize(config=config, audio=audio)
                if response.results:
                    return response.results[0].alternatives[0].transcript
                return None

            finally:
                if os.path.exists(audio_path):
                    os.remove(audio_path)

        except Exception as e:
            logger.error(f"Transcription failed: {e}")
            return None


# CLI interface
if __name__ == '__main__':
    import sys

    if len(sys.argv) < 2:
        print("Usage: python video_extractor.py <video_path>")
        print("\nExample:")
        print("  python video_extractor.py /path/to/video.mp4")
        sys.exit(1)

    video_path = sys.argv[1]

    # Initialize (requires Vertex AI setup)
    try:
        import vertexai
        from vertexai.generative_models import GenerativeModel

        vertexai.init()
        model = GenerativeModel("gemini-1.5-flash")
        extractor = VideoExtractor(gemini_model=model)

        result = extractor.extract_content(video_path)

        print(f"\nDuration: {result.duration:.1f}s")
        print(f"Scenes: {len(result.scene_timeline)}")
        print(f"Text overlays: {len(result.text_timeline)}")
        print(f"Transcript: {result.transcript[:200] if result.transcript else 'None'}...")
        print(f"Thumbnail: {result.thumbnail_url}")

    except Exception as e:
        print(f"Error: {e}")
        print("\nMake sure to set up Vertex AI:")
        print("  export GOOGLE_APPLICATION_CREDENTIALS=/path/to/service-account.json")
        sys.exit(1)
