"""
Meta Ad Creatives - Creative Performance & Hit Rate Tracking

Track creative performance and hit rates across multiple Meta Ads accounts.
Designed for performance marketing agencies managing multiple clients.

Usage:
    from scripts.meta_ad_creatives import (
        get_all_hit_rates,
        get_individual_ads,
        get_monthly_comparison,
        get_available_months,
        get_account_names
    )
"""
import json
import logging
import os
import sqlite3
from datetime import datetime, timedelta
from pathlib import Path
from typing import Dict, List, Any, Optional
import requests
from dotenv import load_dotenv

load_dotenv()

logger = logging.getLogger(__name__)

# Meta API configuration
BASE_URL = 'https://graph.facebook.com'
API_VERSION = 'v24.0'

# Database path
DB_PATH = Path(__file__).parent.parent / "data" / "hit_rates.db"

# Config path
CONFIG_PATH = Path(__file__).parent.parent / "accounts_config.json"


def get_accounts_config() -> Dict:
    """Load accounts configuration from config file."""
    if CONFIG_PATH.exists():
        with open(CONFIG_PATH) as f:
            config = json.load(f)
            return {
                name: acc for name, acc in config.get('accounts', {}).items()
                if acc.get('active', True)
            }
    logger.warning("No accounts config found")
    return {}


def get_facebook_credentials() -> tuple:
    """Get Facebook API credentials from environment."""
    access_token = os.getenv("FACEBOOK_ACCESS_TOKEN")
    app_id = os.getenv("FACEBOOK_APP_ID")
    app_secret = os.getenv("FACEBOOK_APP_SECRET")
    return access_token, app_id, app_secret


class MetaAdsClient:
    """Meta Ads API client for querying ad metrics."""

    def __init__(self, access_token: str, ad_account_id: str):
        self.access_token = access_token
        self.ad_account_id = ad_account_id if ad_account_id.startswith('act_') else f'act_{ad_account_id}'

    def get_ads_created_in_period(
        self,
        since: str,
        until: str,
        campaign_filter: Optional[str] = None,
        geo_filter: Optional[str] = None,
        ad_name_filter: Optional[str] = None
    ) -> List[Dict]:
        """Get all ads created in a specific time period."""
        params = {
            'access_token': self.access_token,
            'fields': 'id,name,created_time,status,effective_status,campaign{name}',
            'limit': 500
        }

        if campaign_filter:
            filtering = [{
                'field': 'campaign.name',
                'operator': 'CONTAIN',
                'value': campaign_filter
            }]
            params['filtering'] = json.dumps(filtering)

        url = f'{BASE_URL}/{API_VERSION}/{self.ad_account_id}/ads'

        all_ads = []
        try:
            while url:
                response = requests.get(url, params=params, timeout=60)
                response.raise_for_status()
                data = response.json()
                all_ads.extend(data.get('data', []))

                paging = data.get('paging', {})
                url = paging.get('next')
                params = {}

            # Filter by created_time
            since_dt = datetime.strptime(since, '%Y-%m-%d')
            until_dt = datetime.strptime(until, '%Y-%m-%d').replace(hour=23, minute=59, second=59)

            ads_in_period = []
            for ad in all_ads:
                created_str = ad.get('created_time', '')
                if created_str:
                    created = datetime.fromisoformat(created_str.replace('Z', '+00:00'))
                    created_naive = created.replace(tzinfo=None)
                    if since_dt <= created_naive <= until_dt:
                        # Geo filter
                        if geo_filter:
                            campaign_name = ad.get('campaign', {}).get('name', '')
                            if f'_{geo_filter}_' not in campaign_name and not campaign_name.endswith(f'_{geo_filter}'):
                                continue
                        # Ad name filter
                        if ad_name_filter:
                            ad_name = ad.get('name', '')
                            if ad_name_filter.lower() not in ad_name.lower():
                                continue
                        ads_in_period.append(ad)

            return ads_in_period

        except requests.exceptions.RequestException as e:
            logger.error(f"Error fetching ads: {e}")
            return []

    def get_ad_insights(self, ad_ids: List[str], since: str, until: str) -> Dict[str, Dict]:
        """Get insights (metrics) for a list of ads."""
        if not ad_ids:
            return {}

        fields = ['ad_id', 'spend', 'impressions', 'clicks', 'actions', 'conversions']
        insights_map = {}
        batch_size = 50

        for i in range(0, len(ad_ids), batch_size):
            batch = ad_ids[i:i + batch_size]

            params = {
                'access_token': self.access_token,
                'fields': ','.join(fields),
                'level': 'ad',
                'time_range': json.dumps({'since': since, 'until': until}),
                'filtering': json.dumps([{
                    'field': 'ad.id',
                    'operator': 'IN',
                    'value': batch
                }]),
                'limit': 500
            }

            url = f'{BASE_URL}/{API_VERSION}/{self.ad_account_id}/insights'

            try:
                response = requests.get(url, params=params, timeout=60)
                response.raise_for_status()
                data = response.json()

                for insight in data.get('data', []):
                    ad_id = insight.get('ad_id')
                    if ad_id:
                        insights_map[ad_id] = insight

            except requests.exceptions.RequestException as e:
                logger.warning(f"Error fetching insights batch: {e}")

        return insights_map


def is_us_campaign(campaign_name: str) -> bool:
    """Check if a campaign targets US based on campaign name."""
    if not campaign_name:
        return False
    return '_US_' in campaign_name or campaign_name.endswith('_US')


def extract_metric(insight: Dict, metric: str, currency_conversion: float = 1.0) -> float:
    """
    Extract a metric value from an insight record.

    Args:
        insight: Insight dict from API
        metric: Metric name ('cpt', 'cpi', 'ipm', 'roas')
        currency_conversion: Multiplier for currency conversion (e.g., 0.012 for INR to USD)

    Returns:
        Metric value as float
    """
    spend = float(insight.get('spend', 0) or 0) * currency_conversion
    impressions = int(insight.get('impressions', 0) or 0)

    if metric == 'cpt':
        # Cost per trial
        trials = 0
        for conv in insight.get('conversions', []):
            action_type = conv.get('action_type', '')
            if action_type in ['start_trial_total', 'start_trial_mobile_app']:
                trials = max(trials, int(float(conv.get('value', 0))))
        if trials > 0:
            return round(spend / trials, 2)
        return 0

    elif metric == 'cpi':
        # Cost per install
        installs = 0
        for action in insight.get('actions', []):
            action_type = action.get('action_type', '')
            if action_type in ['mobile_app_install', 'app_install', 'omni_app_install']:
                installs = max(installs, float(action.get('value', 0)))
        if installs > 0:
            return round(spend / installs, 2)
        return 0

    elif metric == 'ipm':
        # Installs per mille (per 1000 impressions)
        installs = 0
        for action in insight.get('actions', []):
            action_type = action.get('action_type', '')
            if action_type in ['mobile_app_install', 'app_install', 'omni_app_install']:
                installs = max(installs, float(action.get('value', 0)))
        if impressions > 0:
            return round((installs / impressions) * 1000, 2)
        return 0

    elif metric == 'roas':
        # Return on ad spend
        revenue = 0
        for action_value in insight.get('action_values', []):
            action_type = action_value.get('action_type', '')
            if action_type in ['purchase', 'omni_purchase']:
                revenue = max(revenue, float(action_value.get('value', 0)))
        if spend > 0:
            return round(revenue / spend, 2)
        return 0

    return 0


def extract_trials(insight: Dict) -> int:
    """Extract trial count from insight record."""
    trials = 0
    for conv in insight.get('conversions', []):
        action_type = conv.get('action_type', '')
        if action_type in ['start_trial_total', 'start_trial_mobile_app']:
            trials = max(trials, int(float(conv.get('value', 0))))
    return trials


def extract_cpt(insight: Dict, currency_conversion: float = 1.0) -> float:
    """Extract Cost Per Trial from insight record."""
    spend = float(insight.get('spend', 0) or 0) * currency_conversion
    trials = extract_trials(insight)
    if trials > 0:
        return round(spend / trials, 2)
    return 0


def get_month_date_range(month_offset: int = 0, all_time: bool = False) -> tuple:
    """Get date range for a month or all time."""
    today = datetime.now()

    if all_time:
        first_day = datetime(2025, 8, 1)
        since = first_day.strftime('%Y-%m-%d')
        until = today.strftime('%Y-%m-%d')
        month_key = 'all-time'
        month_label = 'All Time'
        return since, until, month_key, month_label

    target_month = today.month + month_offset
    target_year = today.year

    while target_month <= 0:
        target_month += 12
        target_year -= 1
    while target_month > 12:
        target_month -= 12
        target_year += 1

    first_day = datetime(target_year, target_month, 1)

    if target_month == 12:
        next_month = datetime(target_year + 1, 1, 1)
    else:
        next_month = datetime(target_year, target_month + 1, 1)
    last_day = next_month - timedelta(days=1)

    if month_offset == 0:
        last_day = min(last_day, today)

    since = first_day.strftime('%Y-%m-%d')
    until = last_day.strftime('%Y-%m-%d')
    month_key = first_day.strftime('%Y-%m')
    month_label = first_day.strftime('%B %Y')

    return since, until, month_key, month_label


# =============================================================================
# DATABASE OPERATIONS (SQLite)
# =============================================================================

def init_db():
    """Initialize SQLite database."""
    DB_PATH.parent.mkdir(parents=True, exist_ok=True)
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute('''
        CREATE TABLE IF NOT EXISTS hit_rate_snapshots (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            account_name TEXT NOT NULL,
            month_key TEXT NOT NULL,
            month_label TEXT,
            date_range_start TEXT,
            date_range_end TEXT,
            total_ads INTEGER,
            ads_with_spend INTEGER,
            ads_hitting_benchmark INTEGER,
            hit_rate REAL,
            benchmark_display TEXT,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(account_name, month_key)
        )
    ''')

    cursor.execute('''
        CREATE TABLE IF NOT EXISTS ad_performance (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            ad_id TEXT NOT NULL,
            ad_name TEXT,
            account_name TEXT NOT NULL,
            month_key TEXT NOT NULL,
            campaign_name TEXT,
            created_time TEXT,
            spend REAL,
            impressions INTEGER,
            clicks INTEGER,
            trials INTEGER,
            cpt REAL,
            benchmark_value REAL,
            hit_benchmark BOOLEAN,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            UNIQUE(ad_id, month_key)
        )
    ''')

    conn.commit()
    conn.close()


def save_hit_rate_snapshot(account_name: str, month_key: str, month_label: str,
                           since: str, until: str, data: Dict):
    """Save or update hit rate snapshot."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute('''
        INSERT OR REPLACE INTO hit_rate_snapshots
        (account_name, month_key, month_label, date_range_start, date_range_end,
         total_ads, ads_with_spend, ads_hitting_benchmark, hit_rate, benchmark_display, updated_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP)
    ''', (
        account_name, month_key, month_label, since, until,
        data['total_ads'], data['ads_with_spend'], data['ads_hitting_benchmark'],
        data['hit_rate'], data['benchmark_display']
    ))

    conn.commit()
    conn.close()


def save_ad_performance(account_name: str, month_key: str, ads: List[Dict],
                        insights: Dict, benchmark_value: float,
                        benchmark_metric: str = 'cpt', currency_conversion: float = 1.0):
    """Save individual ad performance."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    # Delete existing for refresh
    cursor.execute('DELETE FROM ad_performance WHERE account_name = ? AND month_key = ?',
                   (account_name, month_key))

    for ad in ads:
        ad_id = ad['id']
        insight = insights.get(ad_id, {})
        campaign_name = ad.get('campaign', {}).get('name', '')

        spend = float(insight.get('spend', 0) or 0)
        impressions = int(insight.get('impressions', 0) or 0)
        clicks = int(insight.get('clicks', 0) or 0)

        # For CPT, only count US campaigns; for other metrics, count all
        if benchmark_metric == 'cpt' and not is_us_campaign(campaign_name):
            trials = 0
            cpt = 0
            hit_benchmark = False
        else:
            trials = extract_trials(insight)
            cpt = extract_metric(insight, benchmark_metric, currency_conversion) if spend > 0 else 0
            hit_benchmark = 0 < cpt < benchmark_value if spend > 0 else False

        cursor.execute('''
            INSERT OR REPLACE INTO ad_performance
            (ad_id, ad_name, account_name, month_key, campaign_name, created_time,
             spend, impressions, clicks, trials, cpt, benchmark_value, hit_benchmark)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            ad_id, ad.get('name', ''), account_name, month_key, campaign_name,
            ad.get('created_time', ''), spend, impressions, clicks, trials, cpt,
            benchmark_value, hit_benchmark
        ))

    conn.commit()
    conn.close()


def get_hit_rates_from_db(month_key: str) -> Optional[Dict]:
    """Get hit rates from database."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute('''
        SELECT account_name, total_ads, ads_with_spend, ads_hitting_benchmark,
               hit_rate, benchmark_display, month_label, date_range_start, date_range_end
        FROM hit_rate_snapshots WHERE month_key = ?
    ''', (month_key,))

    rows = cursor.fetchall()
    conn.close()

    if not rows:
        return None

    accounts = []
    for row in rows:
        accounts.append({
            'account_name': row[0],
            'total_ads': row[1],
            'ads_with_spend': row[2],
            'ads_hitting_benchmark': row[3],
            'hit_rate': row[4],
            'benchmark_display': row[5],
        })

    total_ads = sum(a['total_ads'] for a in accounts)
    total_with_spend = sum(a['ads_with_spend'] for a in accounts)
    total_hitting = sum(a['ads_hitting_benchmark'] for a in accounts)
    overall_hit_rate = round((total_hitting / total_with_spend * 100), 1) if total_with_spend > 0 else 0

    return {
        'accounts': accounts,
        'month_label': rows[0][6],
        'date_range': f"{rows[0][7]} to {rows[0][8]}",
        'totals': {
            'total_ads': total_ads,
            'ads_with_spend': total_with_spend,
            'ads_hitting_benchmark': total_hitting,
            'hit_rate': overall_hit_rate,
        },
        'from_db': True,
    }


# =============================================================================
# MAIN API FUNCTIONS
# =============================================================================

def fetch_and_save_hit_rates(month_offset: int = 0, all_time: bool = False) -> Dict:
    """Fetch hit rates from Meta API and save to database."""
    since, until, month_key, month_label = get_month_date_range(month_offset, all_time=all_time)

    access_token, _, _ = get_facebook_credentials()
    if not access_token:
        return {'error': 'Missing Facebook credentials', 'accounts': [], 'month_label': month_label}

    accounts_config = get_accounts_config()
    if not accounts_config:
        return {'error': 'No accounts configured', 'accounts': [], 'month_label': month_label}

    results = []

    for account_name, config in accounts_config.items():
        logger.info(f"Fetching hit rates for {account_name}")

        client = MetaAdsClient(
            access_token=access_token,
            ad_account_id=config['account_id']
        )

        campaign_filter = config.get('filter') or config.get('campaign_filter')
        geo_filter = config.get('geo_filter')
        ad_name_filter = config.get('ad_name_filter')
        benchmark_metric = config.get('benchmark_metric', 'cpt')
        benchmark_value = config.get('benchmark_value', 100)
        benchmark_display = config.get('benchmark_display', f'{benchmark_metric.upper()} < ${benchmark_value}')
        currency_conversion = config.get('currency_conversion', 1.0)

        ads = client.get_ads_created_in_period(since, until, campaign_filter, geo_filter, ad_name_filter)
        total_ads = len(ads)

        if total_ads == 0:
            account_data = {
                'account_name': account_name,
                'total_ads': 0,
                'ads_with_spend': 0,
                'ads_hitting_benchmark': 0,
                'hit_rate': 0,
                'benchmark_display': benchmark_display,
            }
            results.append(account_data)
            save_hit_rate_snapshot(account_name, month_key, month_label, since, until, account_data)
            continue

        ad_ids = [ad['id'] for ad in ads]
        insights = client.get_ad_insights(ad_ids, since, until)
        ad_campaigns = {ad['id']: ad.get('campaign', {}).get('name', '') for ad in ads}

        ads_with_spend = 0
        ads_hitting_benchmark = 0

        for ad_id, insight in insights.items():
            spend = float(insight.get('spend', 0) or 0)
            if spend > 0:
                ads_with_spend += 1
                campaign_name = ad_campaigns.get(ad_id, '')
                # For CPT, only count US campaigns; for other metrics, count all
                if benchmark_metric == 'cpt' and not is_us_campaign(campaign_name):
                    continue
                metric_value = extract_metric(insight, benchmark_metric, currency_conversion)
                if 0 < metric_value < benchmark_value:
                    ads_hitting_benchmark += 1

        hit_rate = round((ads_hitting_benchmark / ads_with_spend * 100), 1) if ads_with_spend > 0 else 0

        account_data = {
            'account_name': account_name,
            'total_ads': total_ads,
            'ads_with_spend': ads_with_spend,
            'ads_hitting_benchmark': ads_hitting_benchmark,
            'hit_rate': hit_rate,
            'benchmark_display': benchmark_display,
        }
        results.append(account_data)

        save_hit_rate_snapshot(account_name, month_key, month_label, since, until, account_data)
        save_ad_performance(account_name, month_key, ads, insights, benchmark_value, benchmark_metric, currency_conversion)

    total_ads = sum(r['total_ads'] for r in results)
    total_with_spend = sum(r['ads_with_spend'] for r in results)
    total_hitting = sum(r['ads_hitting_benchmark'] for r in results)
    overall_hit_rate = round((total_hitting / total_with_spend * 100), 1) if total_with_spend > 0 else 0

    return {
        'accounts': results,
        'month_label': month_label,
        'date_range': f"{since} to {until}",
        'totals': {
            'total_ads': total_ads,
            'ads_with_spend': total_with_spend,
            'ads_hitting_benchmark': total_hitting,
            'hit_rate': overall_hit_rate,
        },
    }


def get_all_hit_rates(month_offset: int = 0, force_refresh: bool = False, all_time: bool = False) -> Dict:
    """
    Get hit rates for all active accounts.
    Checks database first, fetches from API if not found or force_refresh.
    """
    since, until, month_key, month_label = get_month_date_range(month_offset, all_time=all_time)

    if not force_refresh:
        db_data = get_hit_rates_from_db(month_key)
        if db_data:
            return db_data

    return fetch_and_save_hit_rates(month_offset, all_time=all_time)


def get_monthly_comparison(num_months: int = 3) -> List[Dict]:
    """Get hit rates for multiple months for comparison."""
    months = []
    for i in range(num_months):
        data = get_all_hit_rates(month_offset=-i)
        months.append(data)
    return months


def get_individual_ads(account_name: Optional[str] = None, month_key: Optional[str] = None,
                       hit_only: bool = False, sort_by: str = 'spend') -> List[Dict]:
    """Get individual ad performance data from database."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    query = 'SELECT * FROM ad_performance WHERE 1=1'
    params = []

    if account_name:
        query += ' AND account_name = ?'
        params.append(account_name)
    if month_key and month_key != 'all-time':
        query += ' AND month_key = ?'
        params.append(month_key)
    if hit_only:
        query += ' AND hit_benchmark = 1'

    sort_map = {'spend': 'spend DESC', 'cpt': 'cpt ASC', 'trials': 'trials DESC', 'created_time': 'created_time DESC'}
    query += f' ORDER BY {sort_map.get(sort_by, "spend DESC")}'

    cursor.execute(query, params)
    rows = cursor.fetchall()
    columns = [desc[0] for desc in cursor.description]
    conn.close()

    return [dict(zip(columns, row)) for row in rows]


def get_available_months() -> List[Dict]:
    """Get list of months that have data."""
    init_db()
    conn = sqlite3.connect(DB_PATH)
    cursor = conn.cursor()

    cursor.execute('''
        SELECT DISTINCT month_key, month_label FROM hit_rate_snapshots
        WHERE month_key != 'all-time' ORDER BY month_key DESC
    ''')

    rows = cursor.fetchall()
    conn.close()

    months = [{'month_key': row[0], 'month_label': row[1]} for row in rows]
    if months:
        months.insert(0, {'month_key': 'all-time', 'month_label': 'All Time'})

    return months


def get_account_names() -> List[str]:
    """Get list of account names from config."""
    return list(get_accounts_config().keys())


# =============================================================================
# CLI
# =============================================================================

if __name__ == '__main__':
    import sys

    if len(sys.argv) < 2:
        print("Meta Ad Creatives - Creative Performance Tracking")
        print()
        print("Usage:")
        print("  python meta_ad_creatives.py hit-rates           Get current month hit rates")
        print("  python meta_ad_creatives.py hit-rates -1        Get last month hit rates")
        print("  python meta_ad_creatives.py compare             Compare last 3 months")
        print("  python meta_ad_creatives.py ads [account]       List individual ads")
        print("  python meta_ad_creatives.py winners [account]   List winning ads only")
        sys.exit(0)

    cmd = sys.argv[1]

    if cmd == 'hit-rates':
        offset = int(sys.argv[2]) if len(sys.argv) > 2 else 0
        data = get_all_hit_rates(month_offset=offset)
        print(f"\n=== {data['month_label']} ===")
        print(f"Date range: {data.get('date_range', 'N/A')}\n")
        for acc in data['accounts']:
            print(f"  {acc['account_name']}: {acc['hit_rate']}% ({acc['ads_hitting_benchmark']}/{acc['ads_with_spend']})")
        print(f"\n  TOTAL: {data['totals']['hit_rate']}%")

    elif cmd == 'compare':
        months = get_monthly_comparison()
        print("\n=== Monthly Comparison ===\n")
        for m in months:
            print(f"  {m['month_label']}: {m['totals']['hit_rate']}%")

    elif cmd == 'ads':
        account = sys.argv[2] if len(sys.argv) > 2 else None
        ads = get_individual_ads(account_name=account)
        print(f"\n=== {len(ads)} Ads ===\n")
        for ad in ads[:20]:
            print(f"  {ad['ad_name'][:50]} | ${ad['spend']:.2f} | CPT: ${ad['cpt']:.2f}")

    elif cmd == 'winners':
        account = sys.argv[2] if len(sys.argv) > 2 else None
        ads = get_individual_ads(account_name=account, hit_only=True)
        print(f"\n=== {len(ads)} Winning Ads ===\n")
        for ad in ads[:20]:
            print(f"  {ad['ad_name'][:50]} | ${ad['spend']:.2f} | CPT: ${ad['cpt']:.2f}")

    else:
        print(f"Unknown command: {cmd}")
