---
name: meta-ad-creatives
version: 1.0.0
description: Track Meta (Facebook/Instagram) ad creative performance and hit rates across multiple accounts. Use when asked about creative win rates, which ads are hitting benchmarks, CPT/CPI/ROAS analysis, or comparing creative performance across accounts and time periods. Supports multiple benchmark metrics (CPT, CPI, IPM, ROAS) and currency conversion.
---

# Meta Ad Creatives

Track creative performance and hit rates across multiple Meta Ads accounts.

## What This Skill Does

- Calculate **hit rates** (% of creatives meeting performance benchmarks)
- Track multiple metrics: **CPT** (cost per trial), **CPI** (cost per install), **IPM** (installs per mille), **ROAS**
- Compare performance across **multiple accounts**
- Store **historical data** for trend analysis
- Identify **winning creatives** vs underperformers
- Support **currency conversion** for international accounts

## Setup

### 1. Environment Variables

```bash
FACEBOOK_ACCESS_TOKEN=your_token_here
FACEBOOK_APP_ID=your_app_id
FACEBOOK_APP_SECRET=your_app_secret
```

### 2. Accounts Configuration

Create `accounts_config.json`:

```json
{
  "accounts": {
    "ClientName": {
      "account_id": "123456789",
      "filter": "CampaignNameFilter",
      "geo_filter": "US",
      "benchmark_value": 100,
      "benchmark_display": "CPT < $100",
      "active": true
    }
  }
}
```

Configuration fields:
- `account_id`: Meta Ad Account ID (without `act_` prefix)
- `filter`: Campaign name filter (optional)
- `geo_filter`: Geographic filter like "US" (optional)
- `benchmark_value`: CPT threshold for "winning" creatives
- `benchmark_display`: Human-readable benchmark description

## Usage

### Get Hit Rates for Current Month

```python
from scripts.meta_ad_creatives import get_all_hit_rates

data = get_all_hit_rates(month_offset=0)
print(f"Overall hit rate: {data['totals']['hit_rate']}%")
for account in data['accounts']:
    print(f"  {account['account_name']}: {account['hit_rate']}%")
```

### Get Hit Rates for Previous Months

```python
# Last month
data = get_all_hit_rates(month_offset=-1)

# Two months ago
data = get_all_hit_rates(month_offset=-2)
```

### Get All-Time Aggregated Data

```python
data = get_all_hit_rates(all_time=True)
```

### Get Individual Ad Performance

```python
from scripts.meta_ad_creatives import get_individual_ads

# All ads for an account
ads = get_individual_ads(account_name="ClientName", month_key="2026-01")

# Only winning ads
winners = get_individual_ads(account_name="ClientName", hit_only=True)

# Sort by CPT
ads = get_individual_ads(sort_by="cpt")
```

### Monthly Comparison

```python
from scripts.meta_ad_creatives import get_monthly_comparison

# Compare last 3 months
months = get_monthly_comparison(num_months=3)
for month in months:
    print(f"{month['month_label']}: {month['totals']['hit_rate']}%")
```

## Key Metrics

| Metric | Description |
|--------|-------------|
| Total Ads | Ads created in the period |
| Ads with Spend | Ads that received budget |
| Ads Hitting Benchmark | Ads meeting CPT threshold |
| Hit Rate | % of ads meeting benchmark |
| CPT | Cost Per Trial (spend / trials) |

## Hit Rate Calculation

A creative "hits" the benchmark when:
1. It has spend > $0
2. It has trials > 0
3. CPT < benchmark_value (e.g., $100)

Hit Rate = (Ads Hitting Benchmark / Ads with Spend) × 100

## Data Storage

The skill stores historical data for trend analysis:
- **Firestore** (default for cloud deployments)
- **SQLite** (local fallback)

Set `USE_FIRESTORE=false` to use SQLite locally.

## Common Questions This Answers

- "What's our creative hit rate this month?"
- "Which creatives are winning for [Client]?"
- "How does this month compare to last month?"
- "Show me all ads that hit the benchmark"
- "What's our all-time hit rate?"
