import json
import os
import sys
from datetime import datetime

MEMORY_FILE = "/home/sampple/clawd/memory/mersoom_memory/knowledge.json"

def load_memory():
    if not os.path.exists(MEMORY_FILE):
        return {"entities": {}, "events": [], "meta": {"last_update": None}}
    with open(MEMORY_FILE, "r", encoding="utf-8") as f:
        return json.load(f)

def save_memory(memory):
    memory["meta"]["last_update"] = datetime.now().isoformat()
    with open(MEMORY_FILE, "w", encoding="utf-8") as f:
        json.dump(memory, f, ensure_ascii=False, indent=2)

def update_entity(nickname, notes=None, entity_type=None, trust=None):
    mem = load_memory()
    if nickname not in mem["entities"]:
        mem["entities"][nickname] = {"notes": "", "type": "Unknown", "trust": 0, "first_seen": datetime.now().strftime("%Y-%m-%d")}
    
    if notes:
        mem["entities"][nickname]["notes"] = notes
    if entity_type:
        mem["entities"][nickname]["type"] = entity_type
    if trust is not None:
        mem["entities"][nickname]["trust"] = int(trust)
    
    save_memory(mem)
    print(f"Updated entity: {nickname}")

def add_event(title, summary):
    mem = load_memory()
    mem["events"].append({
        "date": datetime.now().strftime("%Y-%m-%d"),
        "title": title,
        "summary": summary
    })
    # Keep last 20 events
    mem["events"] = mem["events"][-20:]
    save_memory(mem)
    print(f"Added event: {title}")

def get_context():
    mem = load_memory()
    summary = "=== MERSOOM KNOWLEDGE BASE ===\n"
    summary += "--- Major Events ---\n"
    for e in mem["events"][-5:]:
        summary += f"- [{e['date']}] {e['title']}: {e['summary']}\n"
    
    summary += "\n--- Key Entities ---\n"
    for nick, data in mem["entities"].items():
        if abs(data.get("trust", 0)) > 5 or data.get("type") != "Unknown":
            summary += f"- {nick} ({data['type']}): {data['notes']} (Trust: {data['trust']})\n"
    
    return summary

if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("Usage: python3 mersoom_memory.py [update-entity|add-event|get-context] ...")
        sys.exit(1)
    
    cmd = sys.argv[1]
    if cmd == "update-entity":
        # python3 scripts/mersoom_memory.py update-entity "닉네임" "노트" "타입" "신뢰도"
        nickname = sys.argv[2]
        notes = sys.argv[3] if len(sys.argv) > 3 else None
        etype = sys.argv[4] if len(sys.argv) > 4 else None
        trust = sys.argv[5] if len(sys.argv) > 5 else None
        update_entity(nickname, notes, etype, trust)
    elif cmd == "add-event":
        add_event(sys.argv[2], sys.argv[3])
    elif cmd == "get-context":
        print(get_context())
