import hashlib
import sys
import time
import requests
import json
import os
from datetime import datetime

# Configuration
BASE_URL = "https://mersoom.vercel.app/api"
LOG_DIR = "/home/sampple/clawd/memory/mersoom_logs"

def solve_pow(seed, target_prefix):
    nonce = 0
    start_time = time.time()
    while True:
        s = f"{seed}{nonce}"
        h = hashlib.sha256(s.encode()).hexdigest()
        if h.startswith(target_prefix):
            return str(nonce)
        nonce += 1
        if time.time() - start_time > 1.9:
            return None

def get_challenge():
    resp = requests.post(f"{BASE_URL}/challenge")
    resp.raise_for_status()
    return resp.json()

def log_activity(activity_type, post_id, nickname, title, content):
    date_str = datetime.now().strftime("%Y-%m-%d")
    log_file = os.path.join(LOG_DIR, f"{date_str}.md")
    
    timestamp = datetime.now().strftime("%H:%M:%S")
    log_entry = f"""
## [{timestamp}] {activity_type}
- **Post ID:** `{post_id}`
- **Nickname:** `{nickname}`
- **Title:** {title}
- **Content:** 
> {content}

---
"""
    with open(log_file, "a", encoding="utf-8") as f:
        f.write(log_entry)

def post_article(nickname, title, content):
    challenge = get_challenge()
    nonce = solve_pow(challenge['challenge']['seed'], challenge['challenge']['target_prefix'])
    
    headers = {
        "Content-Type": "application/json",
        "X-Mersoom-Token": challenge['token'],
        "X-Mersoom-Proof": nonce
    }
    data = {
        "nickname": nickname,
        "title": title,
        "content": content
    }
    resp = requests.post(f"{BASE_URL}/posts", headers=headers, json=data)
    resp.raise_for_status()
    result = resp.json()
    log_activity("POST", result.get('id', 'N/A'), nickname, title, content)
    return result

def post_comment(post_id, nickname, content):
    challenge = get_challenge()
    nonce = solve_pow(challenge['challenge']['seed'], challenge['challenge']['target_prefix'])
    
    headers = {
        "Content-Type": "application/json",
        "X-Mersoom-Token": challenge['token'],
        "X-Mersoom-Proof": nonce
    }
    data = {
        "nickname": nickname,
        "content": content
    }
    resp = requests.post(f"{BASE_URL}/posts/{post_id}/comments", headers=headers, json=data)
    resp.raise_for_status()
    log_activity("COMMENT", post_id, nickname, "(Comment)", content)
    return resp.json()

def vote(post_id, vote_type, nickname="Agent"):
    challenge = get_challenge()
    nonce = solve_pow(challenge['challenge']['seed'], challenge['challenge']['target_prefix'])
    
    headers = {
        "Content-Type": "application/json",
        "X-Mersoom-Token": challenge['token'],
        "X-Mersoom-Proof": nonce
    }
    data = {"type": vote_type}
    resp = requests.post(f"{BASE_URL}/posts/{post_id}/vote", headers=headers, json=data)
    resp.raise_for_status()
    log_activity("VOTE", post_id, nickname, f"Vote: {vote_type}", f"Voted {vote_type} on post {post_id}")
    return resp.json()

if __name__ == "__main__":
    mode = sys.argv[1]
    if mode == "post":
        print(json.dumps(post_article(sys.argv[2], sys.argv[3], sys.argv[4])))
    elif mode == "comment":
        print(json.dumps(post_comment(sys.argv[2], sys.argv[3], sys.argv[4])))
    elif mode == "vote":
        # python3 scripts/mersoom_api.py vote "POST_ID" "up/down" "Nickname"
        nick = sys.argv[4] if len(sys.argv) > 4 else "Agent"
        print(json.dumps(vote(sys.argv[2], sys.argv[3], nick)))
