#!/usr/bin/env python3
"""
MemoryLayer Python Wrapper for ClawdBot

Provides a simple Python interface to the MemoryLayer API.
Compatible with the existing plugin architecture.
"""

import os
import requests
from typing import Dict, List, Optional, Any


class MemoryLayerClient:
    """MemoryLayer API client"""
    
    def __init__(
        self,
        api_url: Optional[str] = None,
        email: Optional[str] = None,
        password: Optional[str] = None,
        api_key: Optional[str] = None
    ):
        self.api_url = api_url or os.getenv('MEMORYLAYER_URL', 'https://memorylayer.clawbot.hk')
        self.email = email or os.getenv('MEMORYLAYER_EMAIL')
        self.password = password or os.getenv('MEMORYLAYER_PASSWORD')
        self.api_key = api_key or os.getenv('MEMORYLAYER_API_KEY')
        self.token: Optional[str] = None
    
    def _ensure_auth(self):
        """Ensure we have a valid authentication token"""
        if self.token:
            return
        
        if self.api_key:
            self.token = self.api_key
            return
        
        if not self.email or not self.password:
            raise ValueError(
                'MemoryLayer: Missing credentials. '
                'Set MEMORYLAYER_EMAIL and MEMORYLAYER_PASSWORD, or MEMORYLAYER_API_KEY'
            )
        
        # Login
        response = requests.post(
            f'{self.api_url}/auth/login',
            json={'email': self.email, 'password': self.password}
        )
        response.raise_for_status()
        
        self.token = response.json()['access_token']
    
    def remember(
        self,
        content: str,
        memory_type: str = 'semantic',
        importance: float = 0.5,
        metadata: Optional[Dict[str, Any]] = None
    ) -> Dict[str, Any]:
        """
        Store a new memory
        
        Args:
            content: Memory content
            memory_type: 'episodic' | 'semantic' | 'procedural'
            importance: 0.0 to 1.0
            metadata: Additional tags/data
        
        Returns:
            Memory object with 'id'
        """
        self._ensure_auth()
        
        response = requests.post(
            f'{self.api_url}/memories',
            json={
                'content': content,
                'memory_type': memory_type,
                'importance': importance,
                'metadata': metadata or {}
            },
            headers={'Authorization': f'Bearer {self.token}'}
        )
        response.raise_for_status()
        
        return response.json()
    
    def search(self, query: str, limit: int = 10) -> List[Dict[str, Any]]:
        """
        Search memories semantically
        
        Args:
            query: Search query (natural language)
            limit: Max results (default: 10)
        
        Returns:
            List of SearchResult objects
        """
        self._ensure_auth()
        
        response = requests.post(
            f'{self.api_url}/memories/search',
            json={'query': query, 'limit': limit},
            headers={'Authorization': f'Bearer {self.token}'}
        )
        response.raise_for_status()
        
        return response.json()
    
    def get_context(self, query: str, limit: int = 5) -> str:
        """
        Get formatted context for prompt injection
        
        Args:
            query: What context do you need?
            limit: Max memories (default: 5)
        
        Returns:
            Formatted string ready for prompt
        """
        results = self.search(query, limit)
        
        if not results:
            return 'No relevant memories found.'
        
        lines = ['## Relevant Memories']
        for result in results:
            content = result['memory']['content']
            relevance = result['relevance_score']
            lines.append(f"- {content} (relevance: {relevance:.2f})")
        
        return '\n'.join(lines)
    
    def stats(self) -> Dict[str, Any]:
        """
        Get usage statistics
        
        Returns:
            Object with total_memories, memory_types, operations_this_month
        """
        self._ensure_auth()
        
        response = requests.get(
            f'{self.api_url}/users/me',
            headers={'Authorization': f'Bearer {self.token}'}
        )
        response.raise_for_status()
        
        return response.json()


# Singleton instance
_instance: Optional[MemoryLayerClient] = None


def get_instance() -> MemoryLayerClient:
    """Get or create singleton instance"""
    global _instance
    if _instance is None:
        _instance = MemoryLayerClient()
    return _instance


def remember(
    content: str,
    memory_type: str = 'semantic',
    importance: float = 0.5,
    metadata: Optional[Dict[str, Any]] = None
) -> Dict[str, Any]:
    """Store a new memory (singleton convenience function)"""
    return get_instance().remember(content, memory_type, importance, metadata)


def search(query: str, limit: int = 10) -> List[Dict[str, Any]]:
    """Search memories (singleton convenience function)"""
    return get_instance().search(query, limit)


def get_context(query: str, limit: int = 5) -> str:
    """Get formatted context (singleton convenience function)"""
    return get_instance().get_context(query, limit)


def stats() -> Dict[str, Any]:
    """Get usage stats (singleton convenience function)"""
    return get_instance().stats()


# Example usage
if __name__ == '__main__':
    print('🧠 MemoryLayer Python Wrapper Test\n')
    
    try:
        # Store a memory
        print('📝 Storing memory...')
        result = remember(
            'User prefers dark mode UI',
            memory_type='semantic',
            importance=0.8
        )
        print(f"✅ Stored memory with ID: {result.get('id', 'N/A')}\n")
        
        # Search
        print('🔍 Searching for "UI preferences"...')
        results = search('UI preferences', limit=5)
        print(f'Found {len(results)} results:\n')
        
        for i, r in enumerate(results, 1):
            print(f"{i}. [{r['relevance_score']:.2f}] {r['memory']['content']}")
        
        # Get context
        print('\n📋 Getting formatted context...')
        context = get_context('user interface', limit=3)
        print(context)
        
        # Stats
        print('\n📊 Usage statistics...')
        user_stats = stats()
        print(f"Email: {user_stats.get('email', 'N/A')}")
        print(f"Total memories: {user_stats.get('total_memories', 'N/A')}")
        
        print('\n✅ Test complete!')
        
    except Exception as e:
        print(f'❌ Error: {e}')
        print('\n💡 Set credentials first:')
        print('export MEMORYLAYER_EMAIL=your@email.com')
        print('export MEMORYLAYER_PASSWORD=your_password')
        print('\nOr visit: https://memorylayer.clawbot.hk to sign up')
