const axios = require('axios');

class MemoryLayer {
  constructor(options = {}) {
    this.apiUrl = options.apiUrl || process.env.MEMORYLAYER_URL || 'https://memorylayer.clawbot.hk';
    this.email = options.email || process.env.MEMORYLAYER_EMAIL;
    this.password = options.password || process.env.MEMORYLAYER_PASSWORD;
    this.apiKey = options.apiKey || process.env.MEMORYLAYER_API_KEY;
    this.token = null;
  }

  async ensureAuth() {
    if (this.token) return;
    
    if (this.apiKey) {
      this.token = this.apiKey;
      return;
    }
    
    if (!this.email || !this.password) {
      throw new Error('MemoryLayer: Missing credentials. Set MEMORYLAYER_EMAIL and MEMORYLAYER_PASSWORD, or MEMORYLAYER_API_KEY');
    }
    
    // Login
    const res = await axios.post(`${this.apiUrl}/auth/login`, {
      email: this.email,
      password: this.password
    });
    
    this.token = res.data.access_token;
  }

  async remember(content, options = {}) {
    await this.ensureAuth();
    
    const res = await axios.post(`${this.apiUrl}/memories`, {
      content,
      memory_type: options.type || 'semantic',
      importance: options.importance || 0.5,
      metadata: options.metadata || {}
    }, {
      headers: { Authorization: `Bearer ${this.token}` }
    });
    
    return res.data;
  }

  async search(query, limit = 10) {
    await this.ensureAuth();
    
    const res = await axios.post(`${this.apiUrl}/memories/search`, {
      query,
      limit
    }, {
      headers: { Authorization: `Bearer ${this.token}` }
    });
    
    return res.data;
  }

  async get_context(query, limit = 5) {
    const results = await this.search(query, limit);
    
    if (!results || results.length === 0) {
      return 'No relevant memories found.';
    }
    
    const lines = ['## Relevant Memories'];
    for (const result of results) {
      lines.push(`- ${result.memory.content} (relevance: ${result.relevance_score.toFixed(2)})`);
    }
    
    return lines.join('\n');
  }

  async stats() {
    await this.ensureAuth();
    
    const res = await axios.get(`${this.apiUrl}/users/me`, {
      headers: { Authorization: `Bearer ${this.token}` }
    });
    
    return res.data;
  }
}

// Export singleton
let instance = null;

function getInstance() {
  if (!instance) {
    instance = new MemoryLayer();
  }
  return instance;
}

module.exports = {
  async remember(...args) {
    return getInstance().remember(...args);
  },
  
  async search(...args) {
    return getInstance().search(...args);
  },
  
  async get_context(...args) {
    return getInstance().get_context(...args);
  },
  
  async stats(...args) {
    return getInstance().stats(...args);
  },
  
  MemoryLayer
};
