#!/usr/bin/env node

/**
 * MemoryLayer Basic Usage Example
 * 
 * This example shows:
 * 1. Storing memories
 * 2. Searching for relevant memories
 * 3. Getting formatted context for prompts
 * 4. Checking usage stats
 */

const memory = require('../index.js');

async function main() {
  console.log('🧠 MemoryLayer Basic Usage Example\n');
  
  try {
    // 1. Store some memories
    console.log('📝 Storing memories...');
    
    await memory.remember(
      'User prefers dark mode UI with blue accent colors',
      { type: 'semantic', importance: 0.8 }
    );
    
    await memory.remember(
      'User completed onboarding on 2026-02-03',
      { type: 'episodic', importance: 0.6 }
    );
    
    await memory.remember(
      'To export data: Settings > Account > Export Data button',
      { type: 'procedural', importance: 0.7 }
    );
    
    console.log('✅ Stored 3 memories\n');
    
    // 2. Search for relevant memories
    console.log('🔍 Searching for "UI preferences"...');
    const results = await memory.search('UI preferences', 5);
    
    console.log(`Found ${results.length} results:\n`);
    results.forEach((result, i) => {
      console.log(`${i + 1}. [${result.relevance_score.toFixed(2)}] ${result.memory.content}`);
      console.log(`   Type: ${result.memory.memory_type}, Importance: ${result.memory.importance}`);
    });
    console.log();
    
    // 3. Get formatted context for prompt injection
    console.log('📋 Getting formatted context...');
    const context = await memory.get_context('user interface settings', 3);
    console.log(context);
    console.log();
    
    // 4. Check usage stats
    console.log('📊 Usage statistics...');
    const stats = await memory.stats();
    console.log(`Email: ${stats.email}`);
    console.log(`Total memories: ${stats.total_memories || 'N/A'}`);
    console.log(`Operations this month: ${stats.operations_this_month || 'N/A'}`);
    console.log();
    
    // Token savings calculation
    console.log('💰 Token Savings:');
    console.log('Before MemoryLayer: 10,500 tokens (entire MEMORY.md)');
    console.log('After MemoryLayer: ~500 tokens (5 relevant memories)');
    console.log('Savings: 95% = ~$900/month at scale');
    console.log();
    
    console.log('✅ Example complete!');
    
  } catch (error) {
    console.error('❌ Error:', error.message);
    
    if (error.message.includes('Missing credentials')) {
      console.log('\n💡 Set credentials first:');
      console.log('export MEMORYLAYER_EMAIL=your@email.com');
      console.log('export MEMORYLAYER_PASSWORD=your_password');
      console.log('\nOr visit: https://memorylayer.clawbot.hk to sign up');
    }
    
    process.exit(1);
  }
}

main();
