#!/usr/bin/env node

/**
 * MemoryLayer Agent Integration Example
 * 
 * This example shows how to integrate MemoryLayer into an AI agent workflow:
 * 1. Inject relevant context into prompts
 * 2. Store important learnings from conversations
 * 3. Build up agent knowledge over time
 */

const memory = require('../index.js');

// Simulated AI agent
class AIAgent {
  constructor() {
    this.name = 'ClawdBot';
  }
  
  async processUserMessage(message) {
    console.log(`\n👤 User: ${message}`);
    
    // 1. Get relevant context from MemoryLayer
    const context = await memory.get_context(message, 5);
    
    // 2. Build prompt with context
    const prompt = `${context}

User message: ${message}

Based on the relevant memories above, provide a helpful response.`;
    
    console.log('\n🧠 Agent thinking...');
    console.log('Context retrieved:');
    console.log(context);
    
    // 3. Simulate AI response (in real use, this would call your LLM)
    let response;
    if (message.includes('preferences')) {
      response = "Based on your preferences, I see you prefer dark mode UI with blue accents. I'll keep that in mind!";
      
      // 4. Store this interaction as a memory
      await memory.remember(
        `User asked about their preferences on ${new Date().toISOString().split('T')[0]}`,
        { type: 'episodic', importance: 0.5 }
      );
      
    } else if (message.includes('export')) {
      response = "To export your data, go to Settings > Account > Export Data button. Let me know if you need help!";
      
    } else {
      response = "I'm here to help! What would you like to know?";
    }
    
    console.log(`\n🤖 ${this.name}: ${response}`);
    
    return response;
  }
  
  async learnFromFeedback(feedback, importance = 0.7) {
    // Store important learnings
    await memory.remember(feedback, {
      type: 'semantic',
      importance,
      metadata: {
        source: 'user_feedback',
        timestamp: new Date().toISOString()
      }
    });
    
    console.log(`\n📚 Learned: ${feedback}`);
  }
}

async function main() {
  console.log('🤖 MemoryLayer Agent Integration Example\n');
  console.log('=' .repeat(50));
  
  try {
    const agent = new AIAgent();
    
    // Store some initial preferences
    console.log('\n📝 Setting up agent knowledge...');
    await memory.remember(
      'User prefers dark mode UI with blue accent colors',
      { type: 'semantic', importance: 0.8 }
    );
    
    await memory.remember(
      'User is a developer who likes concise, technical explanations',
      { type: 'semantic', importance: 0.9 }
    );
    
    await memory.remember(
      'To export data: Settings > Account > Export Data button',
      { type: 'procedural', importance: 0.7 }
    );
    
    console.log('✅ Initial knowledge stored');
    
    // Simulate conversation
    console.log('\n' + '='.repeat(50));
    console.log('💬 Starting conversation...');
    console.log('='.repeat(50));
    
    await agent.processUserMessage("What are my UI preferences?");
    
    await agent.processUserMessage("How do I export my data?");
    
    // Learn from user feedback
    console.log('\n' + '='.repeat(50));
    await agent.learnFromFeedback(
      'User found the export feature confusing - add a tutorial',
      0.8
    );
    
    // Show how memory grows
    console.log('\n' + '='.repeat(50));
    console.log('\n📊 Memory Status:');
    const stats = await memory.stats();
    console.log(`Total memories stored: ${stats.total_memories || 'N/A'}`);
    console.log(`Operations used: ${stats.operations_this_month || 'N/A'}`);
    
    console.log('\n💡 Key Benefits:');
    console.log('✅ Only relevant memories loaded (not entire history)');
    console.log('✅ Memory grows over time without token bloat');
    console.log('✅ 95% token savings vs. loading full context');
    console.log('✅ Sub-200ms retrieval speed');
    
    console.log('\n✅ Example complete!');
    
  } catch (error) {
    console.error('❌ Error:', error.message);
    
    if (error.message.includes('Missing credentials')) {
      console.log('\n💡 Set credentials first:');
      console.log('export MEMORYLAYER_EMAIL=your@email.com');
      console.log('export MEMORYLAYER_PASSWORD=your_password');
    }
    
    process.exit(1);
  }
}

main();
