---
slug: memorylayer
name: MemoryLayer
description: Semantic memory for AI agents. 95% token savings with vector search.
homepage: https://memorylayer.clawbot.hk
metadata:
  clawdbot:
    emoji: "🧠"
---

# MemoryLayer

Semantic memory infrastructure for AI agents that actually scales.

## Features

- **95% Token Savings** - Retrieve only relevant memories
- **Semantic Search** - Find memories by meaning, not keywords
- **Sub-200ms** - Lightning-fast memory retrieval
- **Multi-tenant** - Isolated memory per agent instance

## Setup

### 1. Sign up for FREE account

Visit https://memorylayer.clawbot.hk and sign up with Google. You'll get:
- 10,000 operations/month
- 1GB storage
- Community support

### 2. Configure credentials

```bash
# Option 1: Email/Password
export MEMORYLAYER_EMAIL=your@email.com
export MEMORYLAYER_PASSWORD=your_password

# Option 2: API Key (recommended for production)
export MEMORYLAYER_API_KEY=ml_your_api_key_here
```

### 3. Install Python SDK (if not using skill wrapper)

```bash
pip install memorylayer
```

## Usage

### Basic Example

```javascript
// In your Clawdbot agent
const memory = require('memorylayer');

// Store a memory
await memory.remember(
  'User prefers dark mode UI',
  { type: 'semantic', importance: 0.8 }
);

// Search memories
const results = await memory.search('UI preferences');
console.log(results[0].content); // "User prefers dark mode UI"
```

### Python Example

```python
from plugins.memorylayer import memory

# Store
memory.remember(
    "Boss prefers direct reporting with zero bullshit",
    memory_type="semantic",
    importance=0.9
)

# Search
results = memory.recall("What are Boss's preferences?")
for r in results:
    print(f"{r.relevance_score:.2f}: {r.memory.content}")
```

### Token Savings

**Before MemoryLayer:**
```python
# Inject entire memory files
context = open('MEMORY.md').read()  # 10,500 tokens
prompt = f"{context}\n\nUser: What are my preferences?"
```

**After MemoryLayer:**
```python
# Inject only relevant memories
context = memory.get_context("user preferences", limit=5)  # ~500 tokens
prompt = f"{context}\n\nUser: What are my preferences?"
```

**Result:** 95% token reduction, $900/month savings at scale

## API Reference

### `memory.remember(content, options)`

Store a new memory.

**Parameters:**
- `content` (string): Memory content
- `options.type` (string): 'episodic' | 'semantic' | 'procedural'
- `options.importance` (number): 0.0 to 1.0
- `options.metadata` (object): Additional tags/data

**Returns:** Memory object with `id`

### `memory.search(query, limit)`

Search memories semantically.

**Parameters:**
- `query` (string): Search query (natural language)
- `limit` (number): Max results (default: 10)

**Returns:** Array of SearchResult objects

### `memory.get_context(query, limit)`

Get formatted context for prompt injection.

**Parameters:**
- `query` (string): What context do you need?
- `limit` (number): Max memories (default: 5)

**Returns:** Formatted string ready for prompt

### `memory.stats()`

Get usage statistics.

**Returns:** Object with `total_memories`, `memory_types`, `operations_this_month`

## Advanced

### Memory Types

**Episodic** - Events and experiences
```javascript
memory.remember('Deployed MemoryLayer on 2026-02-03', { type: 'episodic' });
```

**Semantic** - Facts and knowledge
```javascript
memory.remember('Boss prefers concise reports', { type: 'semantic' });
```

**Procedural** - How-to and processes
```javascript
memory.remember('To restart server: ssh root@... && systemctl restart...', { type: 'procedural' });
```

### Metadata Tagging

```javascript
memory.remember('User likes blue', {
  type: 'semantic',
  metadata: {
    category: 'preferences',
    subcategory: 'colors',
    source: 'user_profile'
  }
});
```

### Usage Tracking

```javascript
const stats = await memory.stats();
console.log(`Total memories: ${stats.total_memories}`);
console.log(`Operations this month: ${stats.operations_this_month}`);
console.log(`Plan: ${stats.plan} (${stats.operations_limit}/month)`);
```

## Pricing

**FREE Plan** (Current)
- 10,000 operations/month
- 1GB storage
- Community support

**Pro Plan** ($99/mo)
- 1M operations/month
- 10GB storage
- Email support
- 99.9% SLA

**Enterprise** (Custom)
- Unlimited operations
- Unlimited storage
- Dedicated support
- Self-hosted option
- Custom SLA

## Support

- **Documentation:** https://memorylayer.clawbot.hk/docs
- **API Reference:** https://memorylayer.clawbot.hk/api
- **Community:** Discord (link in docs)
- **Issues:** GitHub (link in docs)

## Links

- **Homepage:** https://memorylayer.clawbot.hk
- **Dashboard:** https://dashboard.memorylayer.clawbot.hk
- **API Docs:** https://memorylayer.clawbot.hk/docs
- **Python SDK:** https://pypi.org/project/memorylayer (when published)
