# MemoryLayer ClawdBot Skill

Semantic memory for AI agents with **95% token savings**.

[![Install with ClawdBot](https://img.shields.io/badge/ClawdBot-Install-blue)](https://clawhub.ai/skills/memorylayer)
[![Homepage](https://img.shields.io/badge/Homepage-memorylayer.clawbot.hk-green)](https://memorylayer.clawbot.hk)

## 🎯 What is MemoryLayer?

MemoryLayer provides semantic long-term memory for AI agents, replacing bloated file-based memory systems with efficient vector search.

**The Problem:**
- Dumping entire chat history = 10,500+ tokens per request
- Keyword search misses semantic matches
- File-based memory doesn't scale
- **Cost:** $945/month at 30K requests

**The Solution:**
- Semantic search via embeddings
- 95% token reduction (10.5K → 500 tokens)
- <200ms retrieval
- **Cost:** $45/month at 30K requests

**Savings: $900/month** 💰

## 🚀 Quick Start

### Install

```bash
clawdbot skill install memorylayer
```

> **Note for developers:** If cloning from GitHub, run `npm install` first to install dependencies.

### Setup

```bash
# Sign up for FREE account at https://memorylayer.clawbot.hk
# Then configure credentials:

export MEMORYLAYER_EMAIL=your@email.com
export MEMORYLAYER_PASSWORD=your_password
```

### Usage

**JavaScript:**
```javascript
const memory = require('memorylayer');

// Store a memory
await memory.remember(
  'User prefers dark mode UI',
  { type: 'semantic', importance: 0.8 }
);

// Search memories
const results = await memory.search('UI preferences');
console.log(results[0].content); // "User prefers dark mode UI"

// Get formatted context for prompt injection
const context = await memory.get_context('user preferences', 5);
// Returns: "## Relevant Memories\n- User prefers dark mode..."
```

**Python:**
```python
from memorylayer import memory

# Store
memory.remember(
    "User prefers dark mode UI",
    memory_type="semantic",
    importance=0.8
)

# Search
results = memory.recall("UI preferences")
for r in results:
    print(f"{r.relevance_score:.2f}: {r.memory.content}")
```

## 📊 Token Savings Example

**Before MemoryLayer:**
```python
# Inject entire memory files
context = open('MEMORY.md').read()  # 10,500 tokens
prompt = f"{context}\n\nUser: What are my preferences?"
```

**After MemoryLayer:**
```python
# Inject only relevant memories
context = memory.get_context("user preferences", limit=5)  # ~500 tokens
prompt = f"{context}\n\nUser: What are my preferences?"
```

**Result:** 95% token reduction, $900/month savings at scale

## 🌟 Features

- **Semantic Search** - Find by meaning, not keywords
- **Multi-tenant** - Isolated memory per agent
- **Fast** - <200ms average search time
- **Memory Types** - Episodic, semantic, procedural
- **FREE Plan** - 10,000 operations/month
- **Dual Language** - JavaScript + Python support

## 📖 API Reference

### `memory.remember(content, options)`

Store a new memory.

**Parameters:**
- `content` (string): Memory content
- `options.type` (string): 'episodic' | 'semantic' | 'procedural'
- `options.importance` (number): 0.0 to 1.0
- `options.metadata` (object): Additional tags/data

**Returns:** Memory object with `id`

### `memory.search(query, limit)`

Search memories semantically.

**Parameters:**
- `query` (string): Search query (natural language)
- `limit` (number): Max results (default: 10)

**Returns:** Array of SearchResult objects

### `memory.get_context(query, limit)`

Get formatted context for prompt injection.

**Parameters:**
- `query` (string): What context do you need?
- `limit` (number): Max memories (default: 5)

**Returns:** Formatted string ready for prompt

### `memory.stats()`

Get usage statistics.

**Returns:** Object with `total_memories`, `memory_types`, `operations_this_month`

## 💰 Pricing

### FREE Plan
- 10,000 operations/month
- 1GB storage
- Community support
- **Perfect for side projects**

### Pro Plan ($99/mo)
- 1M operations/month
- 10GB storage
- Email support
- 99.9% SLA

### Enterprise (Custom)
- Unlimited operations
- Unlimited storage
- Dedicated support
- Self-hosted option

## 🔗 Links

- **Homepage:** https://memorylayer.clawbot.hk
- **Dashboard:** https://dashboard.memorylayer.clawbot.hk
- **API Docs:** https://memorylayer.clawbot.hk/docs
- **ClawdHub:** https://clawhub.ai/skills/memorylayer

## 📝 Examples

See the `examples/` directory for:
- `basic-usage.js` - Simple remember + search demo
- `agent-integration.js` - Agent workflow integration
- `token-savings-demo.js` - Before/after ROI comparison

## 🤝 Support

- **Documentation:** https://memorylayer.clawbot.hk/docs
- **Issues:** GitHub Issues
- **Community:** Discord (link in docs)

## 📄 License

MIT

---

**Built by [QuantechCo](https://github.com/davidhx1000-cloud)** | Powered by [MemoryLayer](https://memorylayer.clawbot.hk)
