#!/usr/bin/env python3
"""
Quarantine detected threats in memory files.
Creates backup and redacts the malicious content.
"""

import os
import sys
import shutil
from datetime import datetime
from pathlib import Path

WORKSPACE = os.path.expanduser("~/.openclaw/workspace")
QUARANTINE_DIR = os.path.join(WORKSPACE, ".memory-scan", "quarantine")

def quarantine_line(file_path, line_number):
    """Quarantine a specific line in a file"""
    
    # Ensure quarantine directory exists
    os.makedirs(QUARANTINE_DIR, exist_ok=True)
    
    # Read file
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            lines = f.readlines()
    except Exception as e:
        print(f"Error reading file: {e}", file=sys.stderr)
        return False
    
    # Validate line number
    if line_number < 1 or line_number > len(lines):
        print(f"Invalid line number: {line_number} (file has {len(lines)} lines)", file=sys.stderr)
        return False
    
    # Create backup filename
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    basename = os.path.basename(file_path).replace('.md', '').replace('/', '_')
    backup_name = f"{basename}_line{line_number}_{timestamp}.backup"
    backup_path = os.path.join(QUARANTINE_DIR, backup_name)
    
    # Save backup of original line
    original_line = lines[line_number - 1]
    with open(backup_path, 'w', encoding='utf-8') as f:
        f.write(f"File: {file_path}\n")
        f.write(f"Line: {line_number}\n")
        f.write(f"Timestamp: {timestamp}\n")
        f.write(f"Original content:\n")
        f.write(original_line)
    
    print(f"✓ Backup created: {backup_path}")
    
    # Redact line
    redaction = f"[QUARANTINED BY MEMORY-SCAN: {timestamp}]\n"
    lines[line_number - 1] = redaction
    
    # Write modified file
    try:
        with open(file_path, 'w', encoding='utf-8') as f:
            f.writelines(lines)
        print(f"✓ Line {line_number} redacted in {file_path}")
        return True
    except Exception as e:
        print(f"Error writing file: {e}", file=sys.stderr)
        # Restore from backup
        shutil.copy(backup_path, file_path)
        return False

def quarantine_file(file_path):
    """Quarantine entire file"""
    
    # Ensure quarantine directory exists
    os.makedirs(QUARANTINE_DIR, exist_ok=True)
    
    # Create backup filename
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    basename = os.path.basename(file_path).replace('/', '_')
    backup_name = f"{basename}_{timestamp}.backup"
    backup_path = os.path.join(QUARANTINE_DIR, backup_name)
    
    # Copy entire file to quarantine
    try:
        shutil.copy(file_path, backup_path)
        print(f"✓ File backed up: {backup_path}")
    except Exception as e:
        print(f"Error creating backup: {e}", file=sys.stderr)
        return False
    
    # Replace file with quarantine notice
    quarantine_notice = f"""# QUARANTINED BY MEMORY-SCAN

**Timestamp:** {timestamp}
**Reason:** Malicious content detected
**Backup:** {backup_path}

This file has been quarantined due to security threats.
The original content is preserved in the backup location above.

To restore, review the backup and manually copy safe content back.
"""
    
    try:
        with open(file_path, 'w', encoding='utf-8') as f:
            f.write(quarantine_notice)
        print(f"✓ File quarantined: {file_path}")
        return True
    except Exception as e:
        print(f"Error writing quarantine notice: {e}", file=sys.stderr)
        return False

def main():
    import argparse
    
    parser = argparse.ArgumentParser(description="Quarantine memory threats")
    parser.add_argument("file", help="File path")
    parser.add_argument("line", type=int, nargs='?', help="Line number (optional, omit to quarantine entire file)")
    
    args = parser.parse_args()
    
    file_path = args.file
    if not os.path.isabs(file_path):
        file_path = os.path.join(WORKSPACE, file_path)
    
    if not os.path.exists(file_path):
        print(f"File not found: {file_path}", file=sys.stderr)
        sys.exit(1)
    
    print(f"\n🛡️ Memory Quarantine")
    print("━" * 60)
    
    if args.line:
        success = quarantine_line(file_path, args.line)
    else:
        success = quarantine_file(file_path)
    
    print("━" * 60)
    
    if success:
        print("✓ Quarantine complete")
        sys.exit(0)
    else:
        print("✗ Quarantine failed")
        sys.exit(1)

if __name__ == "__main__":
    main()
