#!/usr/bin/env python3
"""
Memory Scan Eval Runner — Runs test cases against the LLM scanner and reports results.

Usage:
    python3 evals/run.py                      # Run all tests
    python3 evals/run.py --verbose            # Show detailed output
    python3 evals/run.py --category prompt_stealing  # Run only prompt_stealing cases
    python3 evals/run.py --id prompt-steal    # Run tests matching id substring
    python3 evals/run.py --json               # Machine-readable output
"""

import argparse
import json
import os
import subprocess
import sys
import tempfile
import time
from pathlib import Path

EVALS_DIR = Path(__file__).parent
PROJECT_DIR = EVALS_DIR.parent
CASES_FILE = EVALS_DIR / "cases.json"
SCRIPTS_DIR = PROJECT_DIR / "scripts"
SCAN_PY = SCRIPTS_DIR / "memory-scan.py"
ENV_FILE = PROJECT_DIR / ".env"

SEVERITY_ORDER = {"SAFE": 0, "LOW": 1, "MEDIUM": 2, "HIGH": 3, "CRITICAL": 4}


def load_env():
    """Load .env file into os.environ (simple parser, no dependency)."""
    if not ENV_FILE.exists():
        return
    with open(ENV_FILE) as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith("#") or "=" not in line:
                continue
            key, _, value = line.partition("=")
            key = key.strip()
            value = value.strip()
            if value and key not in os.environ:
                os.environ[key] = value


def load_cases(filter_category=None, filter_id=None):
    with open(CASES_FILE) as f:
        cases = json.load(f)

    if filter_id:
        cases = [c for c in cases if filter_id in c["id"]]
    if filter_category:
        cases = [c for c in cases if c["category"] == filter_category]

    return cases


def run_scan(text):
    """Write text to a temp file and run memory-scan.py --file --json on it."""
    load_env()
    with tempfile.NamedTemporaryFile(mode="w", suffix=".md", delete=False) as f:
        f.write(text)
        tmp_path = f.name

    try:
        cmd = [sys.executable, str(SCAN_PY), "--file", tmp_path, "--json", "--allow-remote"]
        result = subprocess.run(cmd, capture_output=True, text=True, timeout=120)
        stdout = result.stdout.strip()

        try:
            data = json.loads(stdout)
            # memory-scan returns a list of results (one per file)
            if isinstance(data, list) and len(data) > 0:
                return data[0]
            return data
        except json.JSONDecodeError:
            return {"severity": "ERROR", "score": -1, "error": stdout or result.stderr}
    finally:
        os.unlink(tmp_path)


def check_severity_in_range(actual, min_sev, max_sev):
    actual_val = SEVERITY_ORDER.get(actual, -1)
    min_val = SEVERITY_ORDER.get(min_sev, 0)
    max_val = SEVERITY_ORDER.get(max_sev, 4)
    return min_val <= actual_val <= max_val


def run_evals(cases, verbose=False):
    results = []
    total = 0
    passed = 0
    failed = 0
    errors = 0

    for case in cases:
        case_id = case["id"]
        description = case["description"]
        min_sev = case["expected_min_severity"]
        max_sev = case["expected_max_severity"]

        total += 1

        start = time.time()
        result = run_scan(case["text"])
        elapsed = int((time.time() - start) * 1000)

        actual_sev = result.get("severity", "ERROR")

        if actual_sev == "ERROR":
            status = "ERROR"
            errors += 1
        elif check_severity_in_range(actual_sev, min_sev, max_sev):
            status = "PASS"
            passed += 1
        else:
            status = "FAIL"
            failed += 1

        emoji = {"PASS": "✅", "FAIL": "❌", "ERROR": "💥"}[status]
        print(f"  {emoji} {case_id}: {actual_sev} (expected {min_sev}-{max_sev}) [{elapsed}ms]")

        if verbose or status != "PASS":
            if status == "FAIL":
                print(f"       Text: \"{case['text'][:80]}...\"")
            if verbose:
                threats = result.get("threats", [])
                print(f"       Score: {result.get('score', '?')} | Threats: {len(threats)}")
                if result.get("summary"):
                    print(f"       Summary: {result['summary'][:120]}")

        results.append({
            "id": case_id, "status": status,
            "expected": f"{min_sev}-{max_sev}", "actual": actual_sev,
            "score": result.get("score"), "elapsed_ms": elapsed,
        })

    return results, total, passed, failed, errors


def main():
    parser = argparse.ArgumentParser(description="Memory Scan Eval Runner")
    parser.add_argument("--verbose", "-v", action="store_true", help="Show detailed output")
    parser.add_argument("--category", "-c", help="Filter by category (safe, malicious, prompt_stealing)")
    parser.add_argument("--id", help="Filter by case id (substring match)")
    parser.add_argument("--json", action="store_true", help="Output results as JSON")

    args = parser.parse_args()

    cases = load_cases(filter_category=args.category, filter_id=args.id)

    if not cases:
        print("No test cases found.")
        sys.exit(1)

    print(f"\n🧠 Memory Scan Evals — {len(cases)} test cases")
    print(f"   Scanner: {SCAN_PY}")
    print()

    # Group by category
    categories = {}
    for case in cases:
        cat = case["category"]
        if cat not in categories:
            categories[cat] = []
        categories[cat].append(case)

    all_results = []
    total_all = passed_all = failed_all = errors_all = 0

    labels = {
        "safe": "🟢 Safe Content",
        "malicious": "🔴 Malicious Content",
        "prompt_stealing": "🔓 Prompt Stealing",
    }

    for cat_name, cat_cases in categories.items():
        print(f"── {labels.get(cat_name, cat_name)} ({len(cat_cases)} cases) ──")

        results, total, passed, failed, errors = run_evals(
            cat_cases, verbose=args.verbose
        )
        all_results.extend(results)
        total_all += total
        passed_all += passed
        failed_all += failed
        errors_all += errors
        print()

    # Summary
    print("═" * 50)
    print(f"  Tests: {passed_all}/{total_all} passed", end="")
    if failed_all:
        print(f" ({failed_all} failed)", end="")
    if errors_all:
        print(f" ({errors_all} errors)", end="")
    print()

    overall_pass = failed_all == 0 and errors_all == 0
    print(f"\n  {'✅ ALL TESTS PASSED' if overall_pass else '❌ SOME TESTS FAILED'}")
    print("═" * 50)

    if args.json:
        print(json.dumps(all_results, indent=2))

    sys.exit(0 if overall_pass else 1)


if __name__ == "__main__":
    main()
