# Memory-Scan - OpenClaw Memory Security Scanner

Security scanner for OpenClaw agent memory files. Detects malicious instructions, prompt injection, credential leakage, and other threats embedded in MEMORY.md, daily logs, and workspace configuration files.

## Prerequisites

- **Python 3** — check with `python3 --version`
- **API key** (for `--allow-remote` mode) — requires `OPENAI_API_KEY` or `ANTHROPIC_API_KEY`

No pip install is needed — memory-scan uses only the Python standard library (`urllib`).

### Environment Variables

Create a `.env` file in the repository root with any needed keys:

| Variable | Required For | Description |
|----------|-------------|-------------|
| `OPENAI_API_KEY` | `--allow-remote` | OpenAI API key (uses gpt-4o-mini) |
| `ANTHROPIC_API_KEY` | `--allow-remote` | Anthropic API key (alternative to OpenAI) |
| `PROMPTINTEL_API_KEY` | Taxonomy refresh, reporting | MoltThreats / PromptIntel API key |

Pattern-based scanning requires **no keys** — it works out of the box with Python 3.

## Quick Start

### On-Demand Scan

Scan all memory files (local pattern matching only):
```bash
python3 skills/memory-scan/scripts/memory-scan.py
```

Scan with LLM analysis for deeper detection (redacted content sent to LLM):
```bash
python3 skills/memory-scan/scripts/memory-scan.py --allow-remote
```

> **Note:** Without `--allow-remote`, only local pattern matching runs (fast, no API calls). With `--allow-remote`, content is redacted and sent to an LLM for deeper analysis of prompt injection, prompt stealing, and other subtle threats.

### Scheduled Monitoring

Set up daily cron job (3pm PT):
```bash
bash skills/memory-scan/scripts/schedule-scan.sh
```

## What It Does

- **Scans** MEMORY.md, daily logs (last 30 days), and workspace config files
- **Detects** threats using local pattern matching (add `--allow-remote` for deeper LLM analysis on redacted content)
- **Alerts** via configured OpenClaw channel on MEDIUM/HIGH/CRITICAL findings
- **Quarantines** threats with backup + redaction (opt-in)

## Threat Categories

1. **Malicious Instructions** - Commands to harm user/data
2. **Prompt Injection** - Embedded manipulation patterns
3. **Credential Leakage** - Exposed API keys, passwords, tokens
4. **Data Exfiltration** - Instructions to leak data
5. **Guardrail Bypass** - Security policy override attempts
6. **Behavioral Manipulation** - Unauthorized personality changes
7. **Privilege Escalation** - Unauthorized access attempts
8. **Prompt Stealing** - System prompt extraction attempts

## Security Levels

- **SAFE (90-100)** - No threats
- **LOW (70-89)** - Minor concerns
- **MEDIUM (50-69)** - Review recommended
- **HIGH (20-49)** - Immediate attention
- **CRITICAL (0-19)** - Quarantine recommended

## Usage Examples

### Scan Specific File

```bash
python3 skills/memory-scan/scripts/memory-scan.py --file memory/2026-02-01.md
```

### Quiet Mode (Automation)

```bash
python3 skills/memory-scan/scripts/memory-scan.py --quiet
# Output: SEVERITY SCORE (e.g., "MEDIUM 65")
```

### JSON Output

```bash
python3 skills/memory-scan/scripts/memory-scan.py --json
```

### Quarantine Threat

Quarantine specific line:
```bash
python3 skills/memory-scan/scripts/quarantine.py memory/2026-02-01.md 42
```

Quarantine entire file:
```bash
python3 skills/memory-scan/scripts/quarantine.py memory/2026-02-01.md
```

## Agent Workflow

When running memory scan via agent:

1. **Invoke scan:**
   ```bash
   python3 skills/memory-scan/scripts/memory-scan.py
   ```

2. **If MEDIUM/HIGH/CRITICAL detected:**
   - Immediately send alert via configured channel with:
     - Severity level
     - File and line location
     - Threat description
   - Ask user if they want to quarantine

3. **Do NOT auto-quarantine** - always ask first

4. **Example alert:**
   ```
   🛡️ Memory Scan Alert: HIGH
   
   File: memory/2026-01-30.md:42
   Category: Credential Leakage
   Finding: Exposed OpenAI API key
   
   Quarantine this threat? Reply "yes" to redact line 42.
   ```

## Cron Job Operation

The scheduled scan follows **silent operation rules** (AGENTS.md):
- Only sends alerts if threats detected (MEDIUM+)
- No progress updates or status messages
- Replies with NO_REPLY if SAFE/LOW

## Integration

### With safe-install Daily Audit

Memory-scan is automatically included in the daily security audit:
```bash
bash skills/safe-install/scripts/daily-audit.sh
```

### With MoltThreats

Detected threats can be reported to community feed:
```bash
python3 skills/molthreats/scripts/molthreats.py report \
  "Memory injection in daily log" \
  prompt \
  high \
  confirmed
```

### With Input-Guard

Complementary tools:
- **input-guard** - Scans EXTERNAL inputs (web, tweets, search)
- **memory-scan** - Scans INTERNAL memory (agent's stored knowledge)

## Files

```
skills/memory-scan/
├── SKILL.md                      # Skill documentation
├── TESTING.md                    # Eval approach and results
├── README.md                     # This file
├── docs/
│   └── detection-prompt.md       # LLM detection prompt template
├── evals/
│   ├── cases.json                # Test cases (safe, malicious, prompt_stealing)
│   └── run.py                    # Eval runner
└── scripts/
    ├── memory-scan.py           # Main scanner (local patterns + optional LLM)
    ├── schedule-scan.sh         # Create cron job
    └── quarantine.py            # Quarantine detected threats
```

## LLM Provider

Auto-detects from OpenClaw gateway config:
- Prefers OpenAI (gpt-4o-mini) if OPENAI_API_KEY available
- Falls back to Anthropic (claude-sonnet-4-5)
- Uses same approach as input-guard for consistency

## Exit Codes

- `0` - SAFE
- `1` - LOW
- `2` - MEDIUM
- `3` - HIGH
- `4` - CRITICAL

Use in automation:
```bash
if ! python3 skills/memory-scan/scripts/memory-scan.py --quiet; then
  echo "Threats detected!"
fi
```

## Security Notes

- **Does NOT auto-modify** memory files (quarantine is opt-in)
- **Creates backups** before any modifications
- **Preserves evidence** in .memory-scan/quarantine/
- **Safe to run frequently** (minimal API cost with efficient chunking)
- **Context-aware** - understands legitimate agent operations vs threats

## Example Output

```bash
$ python3 skills/memory-scan/scripts/memory-scan.py

🧠 Memory Security Scan
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

Scanning memory files...

✓ MEMORY.md - SAFE
✓ memory/2026-02-01.md - SAFE
⚠ memory/2026-01-30.md - HIGH (line 42)
  → Credential Leakage: Exposed OpenAI API key
✓ AGENTS.md - SAFE
✓ SOUL.md - SAFE
✓ USER.md - SAFE

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
Overall: HIGH
Action: Review memory/2026-01-30.md:42 and consider quarantine
```

## Contributing

To improve detection:
1. Edit `docs/detection-prompt.md` with new patterns
2. Test against known malicious samples
3. Update threat categories as new attack vectors emerge

## Uninstalling

### 1. Remove the scheduled cron job (if configured)

If you set up daily scanning with `schedule-scan.sh`:

```bash
# List cron jobs and find memory-scan-daily
openclaw cron list

# Disable it (replace <ID> with the job ID)
openclaw cron update --jobId <ID> --patch '{"enabled": false}'
```

### 2. Remove quarantine data

If you quarantined any threats, backups are stored in the workspace:

```bash
rm -rf .memory-scan/
```

### 3. Remove the skill directory

```bash
rm -rf skills/memory-scan
```

### 4. Clean up environment variables

Remove from your `.env` (if no other skill uses them):
- `OPENAI_API_KEY`
- `ANTHROPIC_API_KEY`
- `PROMPTINTEL_API_KEY`
- `OPENCLAW_ALERT_CHANNEL`
- `OPENCLAW_ALERT_TO`

memory-scan does not add a section to `AGENTS.md`, so no changes are needed there.

## Related Skills

- **input-guard** - External input scanning
- **skill-guard** - Skill package security
- **molthreats** - Threat intelligence feed
- **safe-install** - Secure skill installation
- **guardrails** - Security policy configuration
