/**
 * OpenClaw Memory (LanceDB) Plugin — Hybrid Search override (BM25 + vector)
 *
 * Long-term memory with vector search for AI conversations.
 * Uses LanceDB for storage and OpenAI for embeddings.
 * Provides seamless auto-recall and auto-capture via lifecycle hooks.
 */

import type * as LanceDB from "@lancedb/lancedb";
import type { OpenClawPluginApi } from "openclaw/plugin-sdk";
import { Type } from "@sinclair/typebox";
import { randomUUID } from "node:crypto";
import OpenAI, {
  APIConnectionError,
  APIConnectionTimeoutError,
  RateLimitError,
  InternalServerError,
  APIError,
} from "openai";
import { stringEnum } from "openclaw/plugin-sdk";
import {
  MEMORY_CATEGORIES,
  type HybridConfig,
  type MemoryCategory,
  memoryConfigSchema,
  vectorDimsForModel,
} from "./config.js";

// ============================================================================
// Retry Infrastructure
// ============================================================================

/** Returns true for transient errors that are worth retrying. */
function isTransientError(err: unknown): boolean {
  if (err instanceof APIConnectionError) return true;
  if (err instanceof APIConnectionTimeoutError) return true;
  if (err instanceof RateLimitError) return true;
  if (err instanceof InternalServerError) return true;
  // OpenAI APIError with 5xx status
  if (err instanceof APIError && typeof err.status === "number" && err.status >= 500) return true;

  // Generic Node.js / LanceDB connection errors
  if (err instanceof Error) {
    const code = (err as NodeJS.ErrnoException).code;
    if (code === "ECONNRESET" || code === "ETIMEDOUT" || code === "ECONNREFUSED") return true;
    const msg = err.message.toLowerCase();
    if (msg.includes("connection error") || msg.includes("connection reset") || msg.includes("timeout")) return true;
  }
  return false;
}

/** Format an error for logging: "ErrorName: message (first stack line)" */
function formatError(err: unknown): string {
  if (!(err instanceof Error)) return String(err);
  const name = err.name || err.constructor?.name || "Error";
  let s = `${name}: ${err.message}`;
  if (err.stack) {
    const lines = err.stack.split("\n");
    const frame = lines.find((l) => l.trimStart().startsWith("at "));
    if (frame) s += ` [${frame.trim()}]`;
  }
  return s;
}

const MAX_RETRY_ATTEMPTS = 4;
const MAX_RETRY_DELAY_MS = 4000;
const BASE_DELAY_MS = 250;

/**
 * Retry an async operation with exponential backoff + jitter.
 * Only retries on transient errors. Non-transient errors are thrown immediately.
 */
async function retryWithBackoff<T>(
  fn: () => Promise<T>,
  opts?: { label?: string; logger?: Logger; onTransientFail?: () => void },
): Promise<T> {
  let lastErr: unknown;
  for (let attempt = 1; attempt <= MAX_RETRY_ATTEMPTS; attempt++) {
    try {
      return await fn();
    } catch (err) {
      lastErr = err;
      if (!isTransientError(err) || attempt === MAX_RETRY_ATTEMPTS) {
        throw err;
      }
      opts?.logger?.debug?.(
        `memory-lancedb-hybrid: ${opts.label ?? "op"} transient error (attempt ${attempt}/${MAX_RETRY_ATTEMPTS}): ${err instanceof Error ? err.message : String(err)}`,
      );
      opts?.onTransientFail?.();
      const delay = Math.min(BASE_DELAY_MS * 2 ** (attempt - 1), MAX_RETRY_DELAY_MS);
      const jitter = delay * 0.5 * Math.random();
      await new Promise((r) => setTimeout(r, delay + jitter));
    }
  }
  throw lastErr; // unreachable but satisfies TS
}

// ============================================================================
// Types
// ============================================================================

let lancedbImportPromise: Promise<typeof import("@lancedb/lancedb")> | null = null;
const loadLanceDB = async (): Promise<typeof import("@lancedb/lancedb")> => {
  if (!lancedbImportPromise) {
    lancedbImportPromise = import("@lancedb/lancedb");
  }
  try {
    return await lancedbImportPromise;
  } catch (err) {
    // Common on macOS today: upstream package may not ship darwin native bindings.
    throw new Error(`memory-lancedb-hybrid: failed to load LanceDB. ${String(err)}`);
  }
};

type MemoryEntry = {
  id: string;
  text: string;
  vector: number[];
  importance: number;
  category: MemoryCategory;
  createdAt: number;
};

type MemorySearchResult = {
  entry: MemoryEntry;
  score: number;
};

// ============================================================================
// LanceDB Provider
// ============================================================================

const TABLE_NAME = "memories";

type Logger = {
  info: (msg: string) => void;
  warn: (msg: string) => void;
  debug?: (msg: string) => void;
};

class MemoryDB {
  private db: LanceDB.Connection | null = null;
  private table: LanceDB.Table | null = null;
  private initPromise: Promise<void> | null = null;
  private reranker: LanceDB.rerankers.RRFReranker | null = null;
  private ftsReady = false;
  /** Lightweight mutex: chains LanceDB ops so they don't overlap. */
  private opLock: Promise<unknown> = Promise.resolve();

  constructor(
    private readonly dbPath: string,
    private readonly vectorDim: number,
    private readonly hybridConfig: HybridConfig = {
      enabled: true,
      reranker: "rrf",
      vectorWeight: 0.7,
      textWeight: 0.3,
    },
    private readonly logger?: Logger,
  ) {}

  /** Reset all LanceDB state so next operation re-initializes from scratch. */
  private reset(): void {
    this.db = null;
    this.table = null;
    this.initPromise = null;
    this.reranker = null;
    this.ftsReady = false;
  }

  /**
   * Serialize a LanceDB operation through the mutex and wrap with retry.
   * On transient errors, resets state and retries (which re-initializes the connection).
   */
  private withMutex<T>(label: string, fn: () => Promise<T>): Promise<T> {
    const next = this.opLock.then(
      () =>
        retryWithBackoff(fn, {
          label,
          logger: this.logger,
          onTransientFail: () => this.reset(),
        }),
      () =>
        retryWithBackoff(fn, {
          label,
          logger: this.logger,
          onTransientFail: () => this.reset(),
        }),
    );
    // Update lock: settle regardless of success/failure so the queue doesn't stall
    this.opLock = next.then(() => {}, () => {});
    return next;
  }

  private async ensureInitialized(): Promise<void> {
    if (this.table) {
      return;
    }
    if (this.initPromise) {
      return this.initPromise;
    }

    this.initPromise = this.doInitialize();
    return this.initPromise;
  }

  private async doInitialize(): Promise<void> {
    const lancedb = await loadLanceDB();
    this.db = await lancedb.connect(this.dbPath);
    const tables = await this.db.tableNames();

    if (tables.includes(TABLE_NAME)) {
      this.table = await this.db.openTable(TABLE_NAME);
    } else {
      this.table = await this.db.createTable(TABLE_NAME, [
        {
          id: "__schema__",
          text: "",
          vector: Array.from({ length: this.vectorDim }).fill(0),
          importance: 0,
          category: "other",
          createdAt: 0,
        },
      ]);
      await this.table.delete('id = "__schema__"');
    }

    // Create FTS index for hybrid search if enabled
    if (this.hybridConfig.enabled) {
      try {
        const indices = await this.table.listIndices();
        // LanceDB typically names indices as "{column}_idx".
        const hasFtsIndex = indices.some(
          (idx) =>
            idx.name === "text_idx" ||
            (idx.columns && idx.columns.includes("text") && idx.indexType === "FTS"),
        );
        if (!hasFtsIndex) {
          await this.table.createIndex("text", { config: lancedb.Index.fts() });
        }
        this.ftsReady = true;

        // Initialize RRF reranker for hybrid search (only if using RRF mode)
        if (this.hybridConfig.reranker !== "linear") {
          this.reranker = await lancedb.rerankers.RRFReranker.create();
        }
      } catch (err) {
        // FTS index creation may fail on empty tables, older bindings, etc.
        // This is not critical - search will fall back to vector-only.
        this.logger?.debug?.(`memory-lancedb-hybrid: FTS index setup failed: ${String(err)}`);
      }
    }
  }

  async store(entry: Omit<MemoryEntry, "id" | "createdAt">): Promise<MemoryEntry> {
    return this.withMutex("store", async () => {
      await this.ensureInitialized();

      const fullEntry: MemoryEntry = {
        ...entry,
        id: randomUUID(),
        createdAt: Date.now(),
      };

      await this.table!.add([fullEntry]);
      return fullEntry;
    });
  }

  async search(
    vector: number[],
    limit = 5,
    minScore = 0.5,
    queryText?: string,
  ): Promise<MemorySearchResult[]> {
    return this.withMutex("search", async () => {
    await this.ensureInitialized();

    let results: Record<string, unknown>[];

    // Use hybrid search if enabled and we have query text
    if (this.hybridConfig.enabled && queryText && this.ftsReady) {
      try {
        if (this.hybridConfig.reranker === "linear") {
          // Linear combination: run vector + FTS separately and combine with weights
          results = await this.linearCombinationSearch(vector, queryText, limit);
        } else if (this.reranker) {
          // RRF reranking via LanceDB built-in
          results = await this.table!.query()
            .nearestTo(vector)
            .fullTextSearch(queryText, { columns: "text" })
            .rerank(this.reranker)
            .limit(limit)
            .toArray();
        } else {
          // Fallback to vector-only
          this.logger?.debug?.(
            "memory-lancedb-hybrid: reranker not ready, falling back to vector search",
          );
          results = await this.table!.vectorSearch(vector).limit(limit).toArray();
        }
      } catch (err) {
        // Fallback to vector-only search if hybrid fails
        this.logger?.debug?.(
          `memory-lancedb-hybrid: hybrid search failed, falling back to vector-only: ${String(err)}`,
        );
        results = await this.table!.vectorSearch(vector).limit(limit).toArray();
      }
    } else {
      // Pure vector search
      results = await this.table!.vectorSearch(vector).limit(limit).toArray();
    }

    // LanceDB uses L2 distance by default; convert to similarity score
    const mapped = results.map((row) => {
      const r = row as Record<string, unknown>;

      // For linear combination results, _combinedScore is already set
      if (r._combinedScore !== undefined) {
        return {
          entry: {
            id: r.id as string,
            text: r.text as string,
            vector: r.vector as number[],
            importance: r.importance as number,
            category: r.category as MemoryEntry["category"],
            createdAt: r.createdAt as number,
          },
          score: r._combinedScore as number,
        };
      }

      const distance = (r._distance as number) ?? 0;
      // Use inverse for a 0-1 range: sim = 1 / (1 + d)
      const score = 1 / (1 + distance);
      return {
        entry: {
          id: r.id as string,
          text: r.text as string,
          vector: r.vector as number[],
          importance: r.importance as number,
          category: r.category as MemoryEntry["category"],
          createdAt: r.createdAt as number,
        },
        score,
      };
    });

    return mapped.filter((res) => res.score >= minScore);
    }); // end withMutex("search")
  }

  /**
   * FTS-only search (no embeddings required).
   * Used as fallback when embedding generation fails.
   */
  async ftsSearch(queryText: string, limit = 5, minScore = 0.3): Promise<MemorySearchResult[]> {
    return this.withMutex("ftsSearch", async () => {
      await this.ensureInitialized();
      if (!this.ftsReady || !this.hybridConfig.enabled) return [];

      const ftsResults = await this.table!.search(queryText, "fts", "text")
        .limit(limit)
        .toArray();

      return ftsResults
        .map((row) => {
          const r = row as Record<string, unknown>;
          const rawScore = (r._score as number) ?? 0;
          const score = rawScore / (1 + rawScore);
          return {
            entry: {
              id: r.id as string,
              text: r.text as string,
              vector: r.vector as number[],
              importance: r.importance as number,
              category: r.category as MemoryEntry["category"],
              createdAt: r.createdAt as number,
            },
            score,
          };
        })
        .filter((res) => res.score >= minScore);
    });
  }

  /**
   * Linear combination search: runs vector and FTS searches separately,
   * then combines results using configured weights.
   */
  private async linearCombinationSearch(
    vector: number[],
    queryText: string,
    limit: number,
  ): Promise<Record<string, unknown>[]> {
    const vectorWeight = this.hybridConfig.vectorWeight ?? 0.7;
    const textWeight = this.hybridConfig.textWeight ?? 0.3;

    // Run both searches in parallel
    const [vectorResults, ftsResults] = await Promise.all([
      this.table!.vectorSearch(vector)
        .limit(limit * 2)
        .toArray(),
      this.table!.search(queryText, "fts", "text")
        .limit(limit * 2)
        .toArray(),
    ]);

    // Build score maps
    const scoreMap = new Map<
      string,
      { row: Record<string, unknown>; vectorScore: number; ftsScore: number }
    >();

    // Process vector results
    for (const row of vectorResults) {
      const id = row.id as string;
      const distance = (row._distance as number) ?? 0;
      const vectorScore = 1 / (1 + distance); // Convert L2 distance to similarity
      scoreMap.set(id, { row, vectorScore, ftsScore: 0 });
    }

    // Process FTS results and merge
    for (const row of ftsResults) {
      const id = row.id as string;
      // FTS returns _score (BM25), normalize to 0-1 range
      const rawScore = (row._score as number) ?? 0;
      // BM25 scores typically range 0-20+, use sigmoid-like normalization
      const ftsScore = rawScore / (1 + rawScore);

      const existing = scoreMap.get(id);
      if (existing) {
        existing.ftsScore = ftsScore;
      } else {
        scoreMap.set(id, { row, vectorScore: 0, ftsScore });
      }
    }

    // Compute combined scores and sort
    const combined = [...scoreMap.entries()]
      .map(([_id, { row, vectorScore, ftsScore }]) => ({
        ...row,
        _combinedScore: vectorWeight * vectorScore + textWeight * ftsScore,
      }))
      .toSorted((a, b) => (b._combinedScore as number) - (a._combinedScore as number))
      .slice(0, limit);

    return combined;
  }

  async delete(id: string): Promise<boolean> {
    return this.withMutex("delete", async () => {
      await this.ensureInitialized();
      // Validate UUID format to prevent injection
      const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
      if (!uuidRegex.test(id)) {
        throw new Error(`Invalid memory ID format: ${id}`);
      }
      await this.table!.delete(`id = '${id}'`);
      return true;
    });
  }

  async count(): Promise<number> {
    return this.withMutex("count", async () => {
      await this.ensureInitialized();
      return this.table!.countRows();
    });
  }
}

// ============================================================================
// OpenAI Embeddings
// ============================================================================

class Embeddings {
  private client: OpenAI;

  constructor(
    apiKey: string,
    private model: string,
    private logger?: Logger,
  ) {
    this.client = new OpenAI({ apiKey, maxRetries: 5, timeout: 30_000 });
  }

  async embed(text: string): Promise<number[]> {
    return retryWithBackoff(
      async () => {
        const response = await this.client.embeddings.create({
          model: this.model,
          input: text,
        });
        return response.data[0].embedding;
      },
      { label: "embed", logger: this.logger },
    );
  }
}

// ============================================================================
// Rule-based capture filter
// ============================================================================

const MEMORY_TRIGGERS = [
  /zapamatuj si|pamatuj|remember/i,
  /preferuji|radši|nechci|prefer/i,
  /rozhodli jsme|budeme používat/i,
  /\+\d{10,}/,
  /[\w.-]+@[\w.-]+\.\w+/,
  /můj\s+\w+\s+je|je\s+můj/i,
  /my\s+\w+\s+is|is\s+my/i,
  /i (like|prefer|hate|love|want|need)/i,
  /always|never|important/i,
];

function shouldCapture(text: string): boolean {
  if (text.length < 10 || text.length > 500) {
    return false;
  }
  // Skip injected context from memory recall
  if (text.includes("<relevant-memories>")) {
    return false;
  }
  // Skip system-generated content
  if (text.startsWith("<") && text.includes("</")) {
    return false;
  }
  // Skip agent summary responses (contain markdown formatting)
  if (text.includes("**") && text.includes("\n-")) {
    return false;
  }
  // Skip emoji-heavy responses (likely agent output)
  const emojiCount = (text.match(/[\u{1F300}-\u{1F9FF}]/gu) || []).length;
  if (emojiCount > 3) {
    return false;
  }
  return MEMORY_TRIGGERS.some((r) => r.test(text));
}

function detectCategory(text: string): MemoryCategory {
  const lower = text.toLowerCase();
  if (/prefer|radši|like|love|hate|want/i.test(lower)) {
    return "preference";
  }
  if (/rozhodli|decided|will use|budeme/i.test(lower)) {
    return "decision";
  }
  if (/\+\d{10,}|@[\w.-]+\.\w+|is called|jmenuje se/i.test(lower)) {
    return "entity";
  }
  if (/is|are|has|have|je|má|jsou/i.test(lower)) {
    return "fact";
  }
  return "other";
}

// ============================================================================
// Plugin Definition
// ============================================================================

const memoryPlugin = {
  // Keep the id as "memory-lancedb" so this plugin can override the bundled one
  // when loaded via plugins.load.paths (higher precedence than bundled extensions).
  id: "memory-lancedb",
  name: "Memory (LanceDB Hybrid)",
  description: "LanceDB-backed long-term memory with BM25+vector hybrid search + auto-recall/capture",
  kind: "memory" as const,
  configSchema: memoryConfigSchema,

  register(api: OpenClawPluginApi) {
    const cfg = memoryConfigSchema.parse(api.pluginConfig);
    const resolvedDbPath = api.resolvePath(cfg.dbPath!);
    const vectorDim = vectorDimsForModel(cfg.embedding.model ?? "text-embedding-3-small");
    const hybridConfig = cfg.hybrid ?? {
      enabled: true,
      reranker: "rrf",
      vectorWeight: 0.7,
      textWeight: 0.3,
    };
    const db = new MemoryDB(resolvedDbPath, vectorDim, hybridConfig, api.logger);
    const embeddings = new Embeddings(cfg.embedding.apiKey, cfg.embedding.model!, api.logger);

    const rerankerDesc =
      hybridConfig.reranker === "linear"
        ? `linear (vec=${hybridConfig.vectorWeight}, text=${hybridConfig.textWeight})`
        : "rrf";

    api.logger.info(
      `memory-lancedb-hybrid: plugin registered (db: ${resolvedDbPath}, hybrid: ${hybridConfig.enabled}, reranker: ${rerankerDesc}, lazy init)`,
    );

    // ========================================================================
    // Tools
    // ========================================================================

    api.registerTool(
      {
        name: "memory_recall",
        label: "Memory Recall",
        description:
          "Search through long-term memories. Use when you need context about user preferences, past decisions, or previously discussed topics.",
        parameters: Type.Object({
          query: Type.String({ description: "Search query" }),
          limit: Type.Optional(Type.Number({ description: "Max results (default: 5)" })),
        }),
        async execute(_toolCallId, params) {
          const { query, limit = 5 } = params as { query: string; limit?: number };

          const vector = await embeddings.embed(query);
          const results = await db.search(vector, limit, 0.1, query);

          if (results.length === 0) {
            return {
              content: [{ type: "text", text: "No relevant memories found." }],
              details: { count: 0 },
            };
          }

          const text = results
            .map(
              (r, i) =>
                `${i + 1}. [${r.entry.category}] ${r.entry.text} (${(r.score * 100).toFixed(0)}%)`,
            )
            .join("\n");

          // Strip vector data for serialization (typed arrays can't be cloned)
          const sanitizedResults = results.map((r) => ({
            id: r.entry.id,
            text: r.entry.text,
            category: r.entry.category,
            importance: r.entry.importance,
            score: r.score,
          }));

          return {
            content: [{ type: "text", text: `Found ${results.length} memories:\n\n${text}` }],
            details: { count: results.length, memories: sanitizedResults },
          };
        },
      },
      { name: "memory_recall" },
    );

    api.registerTool(
      {
        name: "memory_store",
        label: "Memory Store",
        description:
          "Save important information in long-term memory. Use for preferences, facts, decisions.",
        parameters: Type.Object({
          text: Type.String({ description: "Information to remember" }),
          importance: Type.Optional(Type.Number({ description: "Importance 0-1 (default: 0.7)" })),
          category: Type.Optional(stringEnum(MEMORY_CATEGORIES)),
        }),
        async execute(_toolCallId, params) {
          const {
            text,
            importance = 0.7,
            category = "other",
          } = params as {
            text: string;
            importance?: number;
            category?: MemoryEntry["category"];
          };

          const vector = await embeddings.embed(text);

          // Check for duplicates
          const existing = await db.search(vector, 1, 0.95);
          if (existing.length > 0) {
            return {
              content: [
                {
                  type: "text",
                  text: `Similar memory already exists: "${existing[0].entry.text}"`,
                },
              ],
              details: {
                action: "duplicate",
                existingId: existing[0].entry.id,
                existingText: existing[0].entry.text,
              },
            };
          }

          const entry = await db.store({
            text,
            vector,
            importance,
            category,
          });

          return {
            content: [{ type: "text", text: `Stored: "${text.slice(0, 100)}..."` }],
            details: { action: "created", id: entry.id },
          };
        },
      },
      { name: "memory_store" },
    );

    api.registerTool(
      {
        name: "memory_forget",
        label: "Memory Forget",
        description: "Delete specific memories. GDPR-compliant.",
        parameters: Type.Object({
          query: Type.Optional(Type.String({ description: "Search to find memory" })),
          memoryId: Type.Optional(Type.String({ description: "Specific memory ID" })),
        }),
        async execute(_toolCallId, params) {
          const { query, memoryId } = params as { query?: string; memoryId?: string };

          if (memoryId) {
            await db.delete(memoryId);
            return {
              content: [{ type: "text", text: `Memory ${memoryId} forgotten.` }],
              details: { action: "deleted", id: memoryId },
            };
          }

          if (query) {
            const vector = await embeddings.embed(query);
            const results = await db.search(vector, 5, 0.7, query);

            if (results.length === 0) {
              return {
                content: [{ type: "text", text: "No matching memories found." }],
                details: { found: 0 },
              };
            }

            if (results.length === 1 && results[0].score > 0.9) {
              await db.delete(results[0].entry.id);
              return {
                content: [{ type: "text", text: `Forgotten: "${results[0].entry.text}"` }],
                details: { action: "deleted", id: results[0].entry.id },
              };
            }

            const list = results
              .map((r) => `- [${r.entry.id.slice(0, 8)}] ${r.entry.text.slice(0, 60)}...`)
              .join("\n");

            // Strip vector data for serialization
            const sanitizedCandidates = results.map((r) => ({
              id: r.entry.id,
              text: r.entry.text,
              category: r.entry.category,
              score: r.score,
            }));

            return {
              content: [
                {
                  type: "text",
                  text: `Found ${results.length} candidates. Specify memoryId:\n${list}`,
                },
              ],
              details: { action: "candidates", candidates: sanitizedCandidates },
            };
          }

          return {
            content: [{ type: "text", text: "Provide query or memoryId." }],
            details: { error: "missing_param" },
          };
        },
      },
      { name: "memory_forget" },
    );

    // ========================================================================
    // CLI Commands
    // ========================================================================

    api.registerCli(
      ({ program }) => {
        const memory = program.command("ltm").description("LanceDB memory plugin commands");

        memory
          .command("list")
          .description("List memories")
          .action(async () => {
            const count = await db.count();
            console.log(`Total memories: ${count}`);
          });

        memory
          .command("search")
          .description("Search memories")
          .argument("<query>", "Search query")
          .option("--limit <n>", "Max results", "5")
          .action(async (query, opts) => {
            const vector = await embeddings.embed(query);
            const results = await db.search(vector, parseInt(opts.limit), 0.3, query);
            // Strip vectors for output
            const output = results.map((r) => ({
              id: r.entry.id,
              text: r.entry.text,
              category: r.entry.category,
              importance: r.entry.importance,
              score: r.score,
            }));
            console.log(JSON.stringify(output, null, 2));
          });

        memory
          .command("stats")
          .description("Show memory statistics")
          .action(async () => {
            const count = await db.count();
            console.log(`Total memories: ${count}`);
          });
      },
      { commands: ["ltm"] },
    );

    // ========================================================================
    // Lifecycle Hooks
    // ========================================================================

    // Auto-recall: inject relevant memories before agent starts
    if (cfg.autoRecall) {
      api.on("before_agent_start", async (event) => {
        if (!event.prompt || event.prompt.length < 5) {
          return;
        }

        let results: MemorySearchResult[] = [];

        try {
          const vector = await embeddings.embed(event.prompt);
          results = await db.search(vector, 3, 0.3, event.prompt);
        } catch (embedErr) {
          // Embeddings failed even after retries — fall back to FTS-only if available
          api.logger.warn(
            `memory-lancedb-hybrid: recall embedding failed: ${formatError(embedErr)}`,
          );
          try {
            results = await db.ftsSearch(event.prompt, 3, 0.3);
            if (results.length > 0) {
              api.logger.info(
                `memory-lancedb-hybrid: fell back to FTS-only recall (${results.length} results)`,
              );
            }
          } catch (ftsErr) {
            api.logger.warn(
              `memory-lancedb-hybrid: FTS fallback also failed: ${formatError(ftsErr)}`,
            );
            return;
          }
        }

        try {
          if (results.length === 0) {
            return;
          }

          const memoryContext = results
            .map((r) => `- [${r.entry.category}] ${r.entry.text}`)
            .join("\n");

          api.logger.info?.(
            `memory-lancedb-hybrid: injecting ${results.length} memories into context`,
          );

          return {
            prependContext: `<relevant-memories>\nThe following memories may be relevant to this conversation:\n${memoryContext}\n</relevant-memories>`,
          };
        } catch (err) {
          api.logger.warn(`memory-lancedb-hybrid: recall failed: ${formatError(err)}`);
        }
      });
    }

    // Auto-capture: analyze and store important information after agent ends
    if (cfg.autoCapture) {
      api.on("agent_end", async (event) => {
        if (!event.success || !event.messages || event.messages.length === 0) {
          return;
        }

        try {
          // Extract text content from messages (handling unknown[] type)
          const texts: string[] = [];
          for (const msg of event.messages) {
            // Type guard for message object
            if (!msg || typeof msg !== "object") {
              continue;
            }
            const msgObj = msg as Record<string, unknown>;

            // Only process user and assistant messages
            const role = msgObj.role;
            if (role !== "user" && role !== "assistant") {
              continue;
            }

            const content = msgObj.content;

            // Handle string content directly
            if (typeof content === "string") {
              texts.push(content);
              continue;
            }

            // Handle array content (content blocks)
            if (Array.isArray(content)) {
              for (const block of content) {
                if (
                  block &&
                  typeof block === "object" &&
                  "type" in block &&
                  (block as Record<string, unknown>).type === "text" &&
                  "text" in block &&
                  typeof (block as Record<string, unknown>).text === "string"
                ) {
                  texts.push((block as Record<string, unknown>).text as string);
                }
              }
            }
          }

          // Filter for capturable content
          const toCapture = texts.filter((text) => text && shouldCapture(text));
          if (toCapture.length === 0) {
            return;
          }

          // Store each capturable piece (limit to 3 per conversation)
          let stored = 0;
          for (const text of toCapture.slice(0, 3)) {
            const category = detectCategory(text);
            const vector = await embeddings.embed(text);

            // Check for duplicates (high similarity threshold)
            const existing = await db.search(vector, 1, 0.95);
            if (existing.length > 0) {
              continue;
            }

            await db.store({
              text,
              vector,
              importance: 0.7,
              category,
            });
            stored++;
          }

          if (stored > 0) {
            api.logger.info(`memory-lancedb-hybrid: auto-captured ${stored} memories`);
          }
        } catch (err) {
          api.logger.warn(`memory-lancedb-hybrid: capture failed: ${formatError(err)}`);
        }
      });
    }

    // ========================================================================
    // Service
    // ========================================================================

    api.registerService({
      id: "memory-lancedb",
      start: () => {
        api.logger.info(
          `memory-lancedb-hybrid: initialized (db: ${resolvedDbPath}, model: ${cfg.embedding.model})`,
        );
      },
      stop: () => {
        api.logger.info("memory-lancedb-hybrid: stopped");
      },
    });
  },
};

export default memoryPlugin;
