#!/bin/bash
# 记忆系统技能完整验证脚本
# 在启动记忆相关功能前运行此脚本

echo "🧠 记忆系统技能完整验证"
echo "========================"

# 1. 检查环境变量
echo "🔍 1. 检查环境变量..."
if [ ! -z "$BAIDU_API_STRING" ] && [ ! -z "$BAIDU_SECRET_KEY" ]; then
    echo "   ✅ 百度API环境变量已设置"
else
    echo "   ❌ 缺少百度API环境变量"
    echo "   请设置:"
    echo "   export BAIDU_API_STRING='your_bce_v3_api_string'"
    echo "   export BAIDU_SECRET_KEY='${BAIDU_SECRET_KEY}'"
    exit 1
fi

# 2. 测试百度API连接
echo "🌐 2. 测试百度API连接..."
python3 -c "
import sys
import os
sys.path.append('/root/clawd/skills/baidu-vector-db/')
from baidu_embedding_bce_v3 import BaiduEmbeddingBCEV3

api_string = os.getenv('BAIDU_API_STRING')
secret_key = os.getenv('BAIDU_SECRET_KEY')

try:
    client = BaiduEmbeddingBCEV3(api_string, secret_key)
    test_text = '记忆系统技能验证'
    embedding = client.get_embedding_vector(test_text, model='embedding-v1')
    
    if embedding and len(embedding) == 384:  # 标准维度
        print('   ✅ 百度API连接正常')
        print('   ✅ 向量维度正确 (384)')
        sys.exit(0)
    else:
        print('   ❌ API返回异常')
        sys.exit(1)
except Exception as e:
    print(f'   ❌ API连接失败: {str(e)}')
    sys.exit(1)
"

if [ $? -ne 0 ]; then
    echo "   ❌ 百度API连接测试失败"
    exit 1
fi

# 3. 检查百度Embedding数据库
echo "💾 3. 检查百度Embedding数据库..."
python3 -c "
import os
import sys
sys.path.append('/root/clawd/skills/memory-baidu-embedding-db')
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

try:
    mem_db = MemoryBaiduEmbeddingDB()
    stats = mem_db.get_statistics()
    print(f'   ✅ 数据库连接正常')
    print(f'   📊 当前记忆总数: {stats[\"total_memories\"]}')
    sys.exit(0)
except Exception as e:
    print(f'   ❌ 数据库连接失败: {str(e)}')
    sys.exit(1)
"

if [ $? -ne 0 ]; then
    echo "   ❌ 百度Embedding数据库检查失败"
    exit 1
fi

# 4. 检查Git Notes系统
echo "📋 4. 检查Git Notes系统..."
if python3 /root/clawd/skills/git-notes-memory/memory.py -p /root/clawd branches >/dev/null 2>&1; then
    echo "   ✅ Git Notes系统正常"
else
    echo "   ⚠️ Git Notes系统可能存在问题"
fi

# 5. 检查文件系统访问
echo "📁 5. 检查文件系统访问..."
if [ -d "/root/clawd/memory/" ] && [ -w "/root/clawd/memory/" ]; then
    echo "   ✅ 记忆文件目录正常"
else
    echo "   ⚠️ 记忆文件目录可能存在问题"
fi

# 6. 测试完整记忆流程
echo "🔄 6. 测试完整记忆流程..."
python3 -c "
import sys
sys.path.append('/root/clawd/skills/memory-baidu-embedding-db')
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

try:
    mem_db = MemoryBaiduEmbeddingDB()
    
    # 测试添加记忆
    success_add = mem_db.add_memory(
        '记忆系统技能验证测试',
        tags=['verification', 'skill-test'],
        metadata={'timestamp': 'verification', 'type': 'health-check'}
    )
    
    if not success_add:
        print('   ❌ 记忆添加失败')
        sys.exit(1)
    
    # 测试搜索记忆
    results = mem_db.search_memories('记忆系统验证', limit=1)
    if len(results) > 0:
        print('   ✅ 记忆添加和搜索功能正常')
        print(f'   🎯 找到匹配结果: {results[0][\"content\"][:30]}...')
    else:
        print('   ⚠️ 记忆添加成功但搜索未找到结果')
    
    sys.exit(0)
except Exception as e:
    print(f'   ❌ 完整流程测试失败: {str(e)}')
    sys.exit(1)
"

if [ $? -ne 0 ]; then
    echo "   ❌ 完整记忆流程测试失败"
    exit 1
fi

echo ""
echo "🎉 记忆系统技能验证完成!"
echo "=========================="
echo "✅ 所有检查项目均已通过"
echo "✅ 三重记忆系统完全就绪"
echo "✅ 可以安全使用记忆系统技能"
echo ""
echo "📋 系统状态概览:"
echo "   • 语义搜索 (百度Embedding): 正常"
echo "   • 结构化记忆 (Git Notes): 正常" 
echo "   • 文件系统搜索: 正常"
echo "   • 环境变量配置: 完整"
echo "   • API连接状态: 活跃"
echo ""
echo "🚀 记忆系统已为技能调用做好准备!"