"""
百度Embedding内存数据库 - 增强版
用于替代memory-lancedb的向量内存系统
包含增强的错误处理、缓存机制和降级功能
"""

import json
import os
import sqlite3
import hashlib
import traceback
from datetime import datetime
from typing import List, Dict, Optional, Tuple
from pathlib import Path
import threading
from functools import wraps

# 导入百度Embedding客户端
import sys
sys.path.append('/root/clawd/skills/baidu-vector-db/')
from baidu_embedding_bce_v3 import BaiduEmbeddingBCEV3


def with_fallback(fallback_func=None):
    """装饰器：当主要功能失败时调用降级函数"""
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except Exception as e:
                if fallback_func:
                    print(f"⚠️  主功能失败，使用降级功能: {str(e)}")
                    return fallback_func(*args, **kwargs)
                else:
                    raise
        return wrapper
    return decorator


class EnhancedMemoryBaiduEmbeddingDB:
    """
    增强版基于百度Embedding的内存数据库
    包含错误处理、缓存机制和降级功能
    """
    
    def __init__(self, db_path: str = None, cache_size: int = 1000):
        """
        初始化内存数据库
        
        Args:
            db_path: SQLite数据库路径
            cache_size: 缓存大小
        """
        # 从环境变量或配置文件加载API凭据
        api_string = os.getenv("BAIDU_API_STRING")
        secret_key = os.getenv("BAIDU_SECRET_KEY")
        
        # 检查API凭据是否存在
        if not api_string or not secret_key:
            print("⚠️  警告: 缺少百度API凭据，将使用降级模式!")
            print("   请设置以下环境变量以启用完整功能:")
            print("   export BAIDU_API_STRING='your_bce_v3_api_string'")
            print("   export BAIDU_SECRET_KEY='${BAIDU_SECRET_KEY}'")
            self.client = None
        else:
            try:
                self.client = BaiduEmbeddingBCEV3(api_string, secret_key)
            except Exception as e:
                print(f"⚠️  百度API客户端初始化失败: {str(e)}，使用降级模式")
                self.client = None
        
        # 设置数据库路径
        self.db_path = db_path or os.path.join(os.path.expanduser("~"), ".clawd", "enhanced_memory_baidu.db")
        Path(self.db_path).parent.mkdir(parents=True, exist_ok=True)
        
        # 初始化数据库
        self._init_db()
        
        # 初始化缓存
        self.cache_size = cache_size
        self.embedding_cache = {}
        self.cache_lock = threading.Lock()
        
        # 降级模式标志
        self.fallback_mode = self.client is None
    
    def _init_db(self):
        """
        初始化SQLite数据库
        """
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            # 创建记忆表
            cursor.execute('''
                CREATE TABLE IF NOT EXISTS memories (
                    id INTEGER PRIMARY KEY AUTOINCREMENT,
                    content TEXT NOT NULL,
                    embedding_json TEXT,
                    timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
                    tags TEXT,
                    metadata_json TEXT,
                    content_hash TEXT UNIQUE
                )
            ''')
            
            # 创建索引
            cursor.execute('CREATE INDEX IF NOT EXISTS idx_timestamp ON memories(timestamp)')
            cursor.execute('CREATE INDEX IF NOT EXISTS idx_tags ON memories(tags)')
            cursor.execute('CREATE INDEX IF NOT EXISTS idx_content_hash ON memories(content_hash)')
            
            conn.commit()
            conn.close()
        except sqlite3.Error as e:
            print(f"❌ 数据库初始化错误: {str(e)}")
            print(f"   请检查数据库路径是否有效: {self.db_path}")
            print("   可能的原因: 权限不足、磁盘空间不足或路径不存在")
            raise
        except Exception as e:
            print(f"❌ 初始化数据库时发生未知错误: {str(e)}")
            print("   详细错误信息:")
            traceback.print_exc()
            raise
    
    def _get_content_hash(self, content: str) -> str:
        """生成内容哈希用于去重"""
        return hashlib.md5(content.encode()).hexdigest()
    
    def _add_to_cache(self, content: str, embedding: List[float]):
        """添加到嵌入缓存"""
        with self.cache_lock:
            if len(self.embedding_cache) >= self.cache_size:
                # 移除最老的条目
                oldest_key = next(iter(self.embedding_cache))
                del self.embedding_cache[oldest_key]
            self.embedding_cache[content] = embedding
    
    def _get_from_cache(self, content: str) -> Optional[List[float]]:
        """从嵌入缓存获取"""
        return self.embedding_cache.get(content)
    
    def _calculate_similarity(self, vec1: List[float], vec2: List[float]) -> float:
        """
        计算两个向量之间的余弦相似性
        
        Args:
            vec1: 第一个向量
            vec2: 第二个向量
            
        Returns:
            相似性分数 (0-1之间)
        """
        if not vec1 or not vec2:
            return 0.0
            
        dot_product = sum(a * b for a, b in zip(vec1, vec2))
        magnitude1 = sum(a * a for a in vec1) ** 0.5
        magnitude2 = sum(b * b for b in vec2) ** 0.5
        
        if magnitude1 == 0 or magnitude2 == 0:
            return 0.0
        
        return dot_product / (magnitude1 * magnitude2)
    
    def _generate_embedding_fallback(self, content: str) -> Optional[List[float]]:
        """
        降级模式：使用简单关键词向量化
        """
        # 简单的TF-IDF风格向量化（降级实现）
        import re
        from collections import Counter
        
        # 简单分词
        words = re.findall(r'\w+', content.lower())
        word_count = Counter(words)
        
        # 使用哈希来生成固定长度的向量
        vector_size = 384  # 与百度Embedding输出维度一致
        vector = [0.0] * vector_size
        
        for word, count in word_count.items():
            hash_val = hash(word) % vector_size
            vector[hash_val] += count
        
        # 归一化
        magnitude = sum(v*v for v in vector) ** 0.5
        if magnitude > 0:
            vector = [v/magnitude for v in vector]
        
        return vector
    
    @with_fallback(lambda self, content, tags=None, metadata=None: self._add_memory_fallback(content, tags, metadata))
    def add_memory(self, content: str, tags: List[str] = None, metadata: Dict = None) -> bool:
        """
        添加记忆到数据库
        
        Args:
            content: 记忆内容
            tags: 标签列表
            metadata: 元数据
            
        Returns:
            是否添加成功
        """
        try:
            # 输入验证
            if not content or not isinstance(content, str):
                print("❌ 错误: 内容不能为空且必须是字符串")
                return False
            
            if len(content) > 10000:  # 限制内容长度
                print("❌ 错误: 内容过长，请保持在10000字符以内")
                return False
                
            if tags is not None and not isinstance(tags, list):
                print("❌ 错误: 标签必须是字符串列表")
                return False
                
            if metadata is not None and not isinstance(metadata, dict):
                print("❌ 错误: 元数据必须是字典类型")
                return False

            # 检查内容是否已存在
            content_hash = self._get_content_hash(content)
            if self._content_exists(content_hash):
                print(f"⚠️  内容已存在，跳过重复添加: {content[:50]}...")
                return True

            # 生成向量表示
            cached_embedding = self._get_from_cache(content)
            if cached_embedding:
                embedding = cached_embedding
                print("🔄 使用缓存的嵌入向量")
            else:
                if self.client:
                    embedding = self.client.get_embedding_vector(content, model="embedding-v1")
                    if not embedding:
                        print(f"❌ 无法为内容生成向量，尝试降级模式: {content[:50]}...")
                        embedding = self._generate_embedding_fallback(content)
                        if not embedding:
                            print("❌ 降级模式也无法生成向量")
                            return False
                else:
                    # 降级模式
                    embedding = self._generate_embedding_fallback(content)
                    if not embedding:
                        print("❌ 降级模式也无法生成向量")
                        return False
                
                # 添加到缓存
                self._add_to_cache(content, embedding)
        
            # 转换为JSON字符串
            try:
                embedding_json = json.dumps(embedding) if embedding else None
                tags_str = ",".join(tags) if tags else ""
                metadata_json = json.dumps(metadata) if metadata else "{}"
            except TypeError as e:
                print(f"❌ 数据序列化错误: {str(e)}")
                return False
        
            # 插入数据库
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            try:
                cursor.execute('''
                    INSERT OR IGNORE INTO memories (content, embedding_json, tags, metadata_json, content_hash)
                    VALUES (?, ?, ?, ?, ?)
                ''', (content, embedding_json, tags_str, metadata_json, content_hash))
                
                conn.commit()
                
                if cursor.rowcount > 0:
                    print(f"✅ 已添加记忆: {content[:50]}{'...' if len(content) > 50 else ''}")
                    return True
                else:
                    print(f"⚠️  内容已存在: {content[:50]}...")
                    return True
            except sqlite3.Error as e:
                print(f"❌ 数据库插入错误: {str(e)}")
                print("   可能原因: 数据库权限不足、磁盘空间不足或数据库损坏")
                return False
            finally:
                conn.close()
                
        except Exception as e:
            print(f"❌ 添加记忆时发生未知错误: {str(e)}")
            print("   详细错误信息:")
            traceback.print_exc()
            return False
    
    def _add_memory_fallback(self, content: str, tags: List[str] = None, metadata: Dict = None) -> bool:
        """降级模式：只存储内容，不生成向量"""
        try:
            # 输入验证
            if not content or not isinstance(content, str):
                print("❌ 降级模式 - 内容不能为空且必须是字符串")
                return False
            
            if len(content) > 10000:  # 限制内容长度
                print("❌ 降级模式 - 内容过长，请保持在10000字符以内")
                return False

            # 检查内容是否已存在
            content_hash = self._get_content_hash(content)
            if self._content_exists(content_hash):
                print(f"⚠️  降级模式 - 内容已存在，跳过重复添加: {content[:50]}...")
                return True

            # 不生成向量，只存储内容
            try:
                tags_str = ",".join(tags) if tags else ""
                metadata_json = json.dumps(metadata) if metadata else "{}"
            except TypeError as e:
                print(f"❌ 降级模式 - 数据序化错误: {str(e)}")
                return False
        
            # 插入数据库（embedding_json为NULL）
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            try:
                cursor.execute('''
                    INSERT OR IGNORE INTO memories (content, embedding_json, tags, metadata_json, content_hash)
                    VALUES (?, NULL, ?, ?, ?)
                ''', (content, tags_str, metadata_json, content_hash))
                
                conn.commit()
                
                if cursor.rowcount > 0:
                    print(f"✅ 降级模式 - 已添加记忆: {content[:50]}{'...' if len(content) > 50 else ''}")
                    return True
                else:
                    print(f"⚠️  降级模式 - 内容已存在: {content[:50]}...")
                    return True
            except sqlite3.Error as e:
                print(f"❌ 降级模式 - 数据库插入错误: {str(e)}")
                return False
            finally:
                conn.close()
                
        except Exception as e:
            print(f"❌ 降级模式 - 添加记忆时发生未知错误: {str(e)}")
            return False
    
    def _content_exists(self, content_hash: str) -> bool:
        """检查内容是否已存在"""
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('SELECT 1 FROM memories WHERE content_hash = ?', (content_hash,))
            exists = cursor.fetchone() is not None
            
            conn.close()
            return exists
        except:
            return False
    
    @with_fallback(lambda self, query, limit=5, tags=None: self._search_memories_fallback(query, limit, tags))
    def search_memories(self, query: str, limit: int = 5, tags: List[str] = None) -> List[Dict]:
        """
        通过语义搜索相关记忆
        
        Args:
            query: 搜索查询
            limit: 返回结果数量限制
            tags: 标签过滤条件
            
        Returns:
            相关记忆列表
        """
        try:
            # 输入验证
            if not query or not isinstance(query, str):
                print("❌ 错误: 查询不能为空且必须是字符串")
                return []
                
            if limit <= 0 or limit > 100:
                print("❌ 错误: 结果数量限制必须在1-100之间")
                return []
                
            if tags is not None and not isinstance(tags, list):
                print("❌ 错误: 标签必须是字符串列表")
                return []

            # 生成查询向量
            query_embedding = self._get_from_cache(query)
            if not query_embedding:
                if self.client:
                    query_embedding = self.client.get_embedding_vector(query, model="embedding-v1")
                    if query_embedding:
                        self._add_to_cache(query, query_embedding)
                else:
                    # 降级模式
                    query_embedding = self._generate_embedding_fallback(query)
                
                if not query_embedding:
                    print("❌ 无法为查询生成向量，使用关键词匹配")
                    return self._keyword_search(query, limit, tags)

            # 从数据库获取所有有向量的记忆
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            try:
                # 构建查询条件
                where_clause = "WHERE embedding_json IS NOT NULL"  # 只搜索有向量的记忆
                params = []
                
                if tags:
                    # 为每个标签构建OR条件
                    tag_conditions = []
                    for tag in tags:
                        tag_conditions.extend(["tags LIKE ?", "tags LIKE ?", "tags LIKE ?"])
                        params.extend([f'%{tag}%', f'{tag},%', f'%,{tag}%'])
                    
                    if tag_conditions:
                        where_clause += f" AND ({' OR '.join(tag_conditions)})"
                
                cursor.execute(f'''
                    SELECT id, content, embedding_json, timestamp, tags, metadata_json
                    FROM memories
                    {where_clause}
                    ORDER BY timestamp DESC
                ''', params)
                
                rows = cursor.fetchall()
            except sqlite3.Error as e:
                print(f"❌ 数据库查询错误: {str(e)}")
                print("   可能原因: 数据库损坏、权限问题或SQL语法错误")
                return []
            finally:
                conn.close()
            
            # 计算与查询向量的相似性
            results = []
            for row in rows:
                try:
                    embedding = json.loads(row[2])  # embedding_json
                    similarity = self._calculate_similarity(query_embedding, embedding)
                    
                    results.append({
                        "id": row[0],
                        "content": row[1],
                        "similarity": similarity,
                        "timestamp": row[3],
                        "tags": row[4],
                        "metadata": json.loads(row[5]) if row[5] else {},
                    })
                except json.JSONDecodeError:
                    print(f"⚠️ 警告: 无法解析记忆ID {row[0]} 的嵌入向量，跳过该记录")
                    continue
                except Exception as e:
                    print(f"⚠️ 警告: 处理记忆ID {row[0]} 时出错: {str(e)}，跳过该记录")
                    continue
            
            # 按相似性排序并返回前N个结果
            results.sort(key=lambda x: x["similarity"], reverse=True)
            return results[:limit]
            
        except Exception as e:
            print(f"❌ 搜索记忆时发生未知错误: {str(e)}")
            print("   详细错误信息:")
            traceback.print_exc()
            return []
    
    def _search_memories_fallback(self, query: str, limit: int = 5, tags: List[str] = None) -> List[Dict]:
        """降级模式：使用关键词匹配搜索"""
        return self._keyword_search(query, limit, tags)
    
    def _keyword_search(self, query: str, limit: int = 5, tags: List[str] = None) -> List[Dict]:
        """关键词匹配搜索（用于降级模式）"""
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            # 构建查询条件
            where_clause = "WHERE 1=1"
            params = []
            
            if tags:
                # 为每个标签构建OR条件
                tag_conditions = []
                for tag in tags:
                    tag_conditions.extend(["tags LIKE ?", "tags LIKE ?", "tags LIKE ?"])
                    params.extend([f'%{tag}%', f'{tag},%', f'%,{tag}%'])
                
                if tag_conditions:
                    where_clause += f" AND ({' OR '.join(tag_conditions)})"
            
            cursor.execute(f'''
                SELECT id, content, timestamp, tags, metadata_json
                FROM memories
                {where_clause}
                ORDER BY timestamp DESC
            ''', params)
            
            rows = cursor.fetchall()
            conn.close()
            
            # 使用关键词匹配计算相似性
            query_lower = query.lower()
            results = []
            
            for row in rows:
                content_lower = row[1].lower()
                # 简单的关键词匹配得分
                score = 0
                for word in query_lower.split():
                    if word in content_lower:
                        score += 1
                
                if score > 0:  # 只返回匹配的
                    results.append({
                        "id": row[0],
                        "content": row[1],
                        "similarity": score / (len(query_lower.split()) + 1),  # 归一化得分
                        "timestamp": row[2],
                        "tags": row[3],
                        "metadata": json.loads(row[4]) if row[4] else {},
                    })
            
            # 按匹配度排序
            results.sort(key=lambda x: x["similarity"], reverse=True)
            return results[:limit]
            
        except Exception as e:
            print(f"❌ 关键词搜索时发生错误: {str(e)}")
            return []
    
    def get_all_memories(self) -> List[Dict]:
        """
        获取所有记忆（不分页）
        
        Returns:
            所有记忆列表
        """
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            cursor.execute('''
                SELECT id, content, embedding_json, timestamp, tags, metadata_json
                FROM memories
                ORDER BY timestamp DESC
            ''')
            
            rows = cursor.fetchall()
            conn.close()
            
            results = []
            for row in rows:
                try:
                    results.append({
                        "id": row[0],
                        "content": row[1],
                        "has_embedding": row[2] is not None,  # 仅返回是否有嵌入，不返回实际向量
                        "timestamp": row[3],
                        "tags": row[4],
                        "metadata": json.loads(row[5]) if row[5] else {},
                    })
                except json.JSONDecodeError:
                    print(f"⚠️ 警告: 无法解析记忆ID {row[0]} 的元数据，使用空字典")
                    results.append({
                        "id": row[0],
                        "content": row[1],
                        "has_embedding": row[2] is not None,
                        "timestamp": row[3],
                        "tags": row[4],
                        "metadata": {},
                    })
                except Exception as e:
                    print(f"⚠️ 警告: 处理记忆ID {row[0]} 时出错: {str(e)}，跳过该记录")
                    continue
            
            return results
            
        except sqlite3.Error as e:
            print(f"❌ 获取所有记忆时数据库错误: {str(e)}")
            print("   可能原因: 数据库损坏、权限问题或连接失败")
            return []
        except Exception as e:
            print(f"❌ 获取所有记忆时发生未知错误: {str(e)}")
            print("   详细错误信息:")
            traceback.print_exc()
            return []
    
    def get_statistics(self) -> Dict:
        """
        获取数据库统计信息
        
        Returns:
            统计信息字典
        """
        try:
            conn = sqlite3.connect(self.db_path)
            cursor = conn.cursor()
            
            # 总记忆数
            cursor.execute('SELECT COUNT(*) FROM memories')
            total_memories = cursor.fetchone()[0]
            
            # 有嵌入的记忆数
            cursor.execute('SELECT COUNT(*) FROM memories WHERE embedding_json IS NOT NULL')
            memories_with_embeddings = cursor.fetchone()[0]
            
            # 按标签分组统计
            cursor.execute('SELECT tags, COUNT(*) FROM memories GROUP BY tags')
            tag_rows = cursor.fetchall()
            tag_counts = dict(tag_rows) if tag_rows else {}
            
            # 最早和最新的记忆时间
            cursor.execute('SELECT MIN(timestamp), MAX(timestamp) FROM memories')
            min_max_result = cursor.fetchone()
            min_time, max_time = min_max_result if min_max_result else (None, None)
            
            conn.close()
            
            return {
                "total_memories": total_memories,
                "memories_with_embeddings": memories_with_embeddings,
                "memories_without_embeddings": total_memories - memories_with_embeddings,
                "tag_distribution": tag_counts,
                "earliest_memory": min_time,
                "latest_memory": max_time,
                "fallback_mode": self.fallback_mode
            }
            
        except sqlite3.Error as e:
            print(f"❌ 获取统计数据时数据库错误: {str(e)}")
            print("   可能原因: 数据库损坏、权限问题或连接失败")
            return {
                "total_memories": 0,
                "memories_with_embeddings": 0,
                "memories_without_embeddings": 0,
                "tag_distribution": {},
                "earliest_memory": None,
                "latest_memory": None,
                "fallback_mode": True
            }
        except Exception as e:
            print(f"❌ 获取统计数据时发生未知错误: {str(e)}")
            print("   详细错误信息:")
            traceback.print_exc()
            return {
                "total_memories": 0,
                "memories_with_embeddings": 0,
                "memories_without_embeddings": 0,
                "tag_distribution": {},
                "earliest_memory": None,
                "latest_memory": None,
                "fallback_mode": True
            }


def main():
    """
    主函数 - 演示增强版百度Embedding内存数据库功能
    """
    print("🤖 增强版百度Embedding内存数据库")
    print("="*60)
    
    try:
        # 创建内存数据库实例
        mem_db = EnhancedMemoryBaiduEmbeddingDB()
        
        print("\n📊 数据库统计信息:")
        stats = mem_db.get_statistics()
        print(f"  总记忆数: {stats['total_memories']}")
        print(f"  有嵌入的记忆数: {stats['memories_with_embeddings']}")
        print(f"  无嵌入的记忆数: {stats['memories_without_embeddings']}")
        print(f"  降级模式: {'是' if stats['fallback_mode'] else '否'}")
        print(f"  标签分布: {dict(list(stats['tag_distribution'].items())[:5])}")  # 只显示前5个
        print(f"  最早记忆: {stats['earliest_memory']}")
        print(f"  最新记忆: {stats['latest_memory']}")
        
        print("\n📝 添加记忆示例:")
        # 添加一些示例记忆
        examples = [
            {
                "content": "用户喜欢健身，特别关注胸肌和背肌训练，不喜欢练斜方肌",
                "tags": ["user-preference", "fitness"],
                "metadata": {"user": "九十", "date": "2026-01-30"}
            },
            {
                "content": "今天的天气很好，适合户外运动",
                "tags": ["weather", "activity"],
                "metadata": {"date": "2026-01-30"}
            },
            {
                "content": "用户的目标是读书500本、观影2000部、创作20首歌、储蓄50万、学一门外语",
                "tags": ["user-goal", "long-term"],
                "metadata": {"user": "九十", "priority": "high"}
            }
        ]
        
        for example in examples:
            success = mem_db.add_memory(
                example["content"],
                example["tags"],
                example["metadata"]
            )
            print(f"  添加记忆: {'✅' if success else '❌'} - {example['content'][:30]}...")
        
        print("\n🔍 语义搜索示例:")
        # 搜索相关记忆
        search_queries = [
            "用户健身偏好",
            "读书和学习目标",
            "今天的活动建议"
        ]
        
        for query in search_queries:
            print(f"\n  搜索: '{query}'")
            results = mem_db.search_memories(query, limit=2)
            if results:
                for i, result in enumerate(results, 1):
                    print(f"    {i}. 相似度: {result['similarity']:.3f} - {result['content'][:50]}...")
            else:
                print("    未找到相关记忆")
        
        print(f"\n🎉 增强版百度Embedding内存数据库演示完成！")
        print(f"  工作模式: {'完整功能' if not stats['fallback_mode'] else '降级模式'}")
        print("已成功实现基于向量相似性的智能记忆管理功能")
        
    except Exception as e:
        print(f"\n❌ 演示过程中发生错误: {str(e)}")
        print("   详细错误信息:")
        traceback.print_exc()
        return 1
    
    return 0


if __name__ == "__main__":
    main()
