# Memory Baidu Embedding DB - Complete Tutorial

This tutorial will guide you through setting up and using the semantic memory system for Clawdbot.

## Table of Contents
1. [Introduction](#introduction)
2. [Prerequisites](#prerequisites)
3. [Setup](#setup)
4. [Basic Usage](#basic-usage)
5. [Advanced Features](#advanced-features)
6. [Integration Examples](#integration-examples)
7. [Troubleshooting](#troubleshooting)

## Introduction

The Memory Baidu Embedding DB is a semantic memory system that stores and retrieves information based on meaning rather than exact keywords. Instead of searching for specific words, it understands concepts and finds related information using vector similarity.

### Why Semantic Memory?
Traditional memory systems only find exact matches:
- Query: "What does the user like?"
- Stored: "The user enjoys hiking"
- Result: Only matches if you search for "hiking"

Our semantic system understands relationships:
- Query: "What does the user enjoy?"
- Stored: "The user enjoys hiking"
- Result: Matches because "enjoy" and "like" are semantically related

## Prerequisites

Before starting, you'll need:
- A Baidu Qianfan API account with access to Embedding-V1
- Python 3.8+
- Basic knowledge of environment variables

## Setup

### Step 1: Get Baidu API Credentials

1. Go to the [Baidu Qianfan Console](https://console.bce.baidu.com/qianfan/)
2. Create an account or sign in
3. Navigate to the API management section
4. Create an API Key and Secret Key
5. Ensure you have access to the Embedding-V1 model

### Step 2: Configure Environment Variables

Open your terminal and set the environment variables:

```bash
# Replace with your actual API credentials
export BAIDU_API_STRING='${BAIDU_API_STRING}'
export BAIDU_SECRET_KEY='${BAIDU_SECRET_KEY}'

# To make these permanent, add to your ~/.bashrc or ~/.zshrc:
echo 'export BAIDU_API_STRING="${BAIDU_API_STRING}"' >> ~/.bashrc
echo 'export BAIDU_SECRET_KEY="${BAIDU_SECRET_KEY}"' >> ~/.bashrc
source ~/.bashrc
```

### Step 3: Test the Installation

Navigate to the skill directory and run the test:

```bash
cd /root/clawd/skills/memory-baidu-embedding-db
python3 memory_baidu_embedding_db.py
```

You should see output showing:
- Database statistics
- Memory additions
- Semantic searches
- Similarity calculations

## Basic Usage

### Creating a Memory Instance

```python
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

# Create a new memory database instance
memory_db = MemoryBaiduEmbeddingDB()

# Or specify a custom database path
memory_db = MemoryBaiduEmbeddingDB(db_path='/path/to/custom/memory.db')
```

### Adding Memories

Add individual memories with content, tags, and metadata:

```python
# Simple memory addition
success = memory_db.add_memory("The user likes Python programming")
print(f"Memory added: {success}")

# Memory with tags and metadata
success = memory_db.add_memory(
    content="The user prefers concise responses and enjoys technical discussions",
    tags=["user-preference", "communication-style"],
    metadata={
        "importance": "high",
        "source": "conversation-2026-01-30",
        "confidence": 0.95
    }
)
print(f"Detailed memory added: {success}")
```

### Searching Memories

Search for memories using semantic similarity:

```python
# Basic search
results = memory_db.search_memories("What does the user like?")
for result in results:
    print(f"Content: {result['content']}")
    print(f"Similarity: {result['similarity']:.3f}")
    print(f"Timestamp: {result['timestamp']}")

# Search with tag filtering
results = memory_db.search_memories(
    query="programming preferences",
    tags=["tech-preference"],
    limit=3
)
```

### Retrieving Similar Memories

Find memories similar to a given piece of content:

```python
# Find memories similar to a statement
similar_memories = memory_db.retrieve_similar_memories(
    content="The user enjoys coding",
    limit=5
)

for memory in similar_memories:
    print(f"- {memory['content']} (similarity: {memory['similarity']:.3f})")
```

## Advanced Features

### Working with Tags

Tags help organize and filter memories:

```python
# Add memories with multiple tags
memory_db.add_memory(
    content="The user works as a software engineer",
    tags=["professional-info", "job", "background"]
)

memory_db.add_memory(
    content="The user lives in Beijing, China",
    tags=["personal-info", "location", "background"]
)

# Search within specific tags
professional_info = memory_db.search_memories(
    query="job details",
    tags=["professional-info"],
    limit=5
)
```

### Metadata Management

Store additional information with each memory:

```python
# Add memory with rich metadata
memory_db.add_memory(
    content="The user prefers morning meetings",
    tags=["schedule-preference"],
    metadata={
        "source_conversation_id": "conv_12345",
        "confidence_score": 0.89,
        "verification_status": "confirmed",
        "last_updated": "2026-01-30T10:00:00Z",
        "related_topics": ["schedule", "preferences", "meetings"]
    }
)

# Later, you can filter based on metadata properties
# (though direct metadata filtering requires custom implementation)
```

### Batch Operations

Add multiple memories efficiently:

```python
# Batch add memories
memories_to_add = [
    {
        "content": "The user prefers emails over phone calls",
        "tags": ["communication", "preference"],
        "metadata": {"importance": "medium"}
    },
    {
        "content": "The user drinks coffee every morning",
        "tags": ["habit", "lifestyle"],
        "metadata": {"importance": "low"}
    },
    {
        "content": "The user has a dog named Max",
        "tags": ["pets", "personal-info"],
        "metadata": {"importance": "high"}
    }
]

added_count = 0
for memory in memories_to_add:
    success = memory_db.add_memory(
        memory["content"],
        memory["tags"],
        memory["metadata"]
    )
    if success:
        added_count += 1

print(f"Added {added_count}/{len(memories_to_add)} memories")
```

### Getting Statistics

Monitor your memory system:

```python
stats = memory_db.get_statistics()
print(f"Total memories: {stats['total_memories']}")
print(f"Earliest memory: {stats['earliest_memory']}")
print(f"Latest memory: {stats['latest_memory']}")
print("Tag distribution:")
for tag, count in stats['tag_distribution'].items():
    if tag:  # Empty tags appear as empty string
        print(f"  {tag}: {count}")
```

### Managing Memories

Delete or clear memories when needed:

```python
# Get all memories to identify which to delete
all_memories = memory_db.get_all_memories()
print(f"Current memories: {len(all_memories)}")

# Delete a specific memory by ID
if all_memories:
    memory_id_to_delete = all_memories[0]['id']
    success = memory_db.delete_memory(memory_id_to_delete)
    print(f"Deleted memory {memory_id_to_delete}: {success}")

# Clear all memories (use with caution!)
# success = memory_db.clear_all_memories()
# print(f"All memories cleared: {success}")
```

## Integration Examples

### Example 1: Chatbot Context Memory

```python
class SmartChatbot:
    def __init__(self):
        self.memory_db = MemoryBaiduEmbeddingDB()
    
    def process_message(self, user_input, user_id):
        # Search for relevant context
        context_memories = self.memory_db.search_memories(
            query=user_input,
            limit=3
        )
        
        # Format context for the response
        context = self.format_context(context_memories)
        
        # Generate response with context
        response = self.generate_response(user_input, context)
        
        # Store important information from this exchange
        if self.is_important_info(user_input):
            self.memory_db.add_memory(
                content=user_input,
                tags=["conversation", f"user-{user_id}"],
                metadata={"timestamp": self.get_timestamp()}
            )
        
        return response
    
    def format_context(self, memories):
        if not memories:
            return ""
        
        context_parts = ["Relevant context from our conversation history:"]
        for memory in memories:
            if memory['similarity'] > 0.5:  # Only include highly relevant memories
                context_parts.append(f"- {memory['content']}")
        
        return "\n".join(context_parts)
    
    def generate_response(self, input_text, context):
        # This would connect to your LLM
        # For this example, we'll return a simple response
        return f"I understand you said: {input_text}\nContext: {context}"
    
    def is_important_info(self, text):
        # Define what constitutes important information to remember
        important_keywords = ["prefer", "like", "dislike", "need", "want", "goal", "important"]
        return any(keyword in text.lower() for keyword in important_keywords)
    
    def get_timestamp(self):
        from datetime import datetime
        return datetime.now().isoformat()

# Usage
bot = SmartChatbot()
response = bot.process_message("What are my preferences again?", "user123")
print(response)
```

### Example 2: Personal Assistant Memory

```python
class PersonalAssistantMemory:
    def __init__(self):
        self.memory_db = MemoryBaiduEmbeddingDB()
    
    def remember_user_fact(self, user_id, fact):
        """Remember a fact about a user"""
        return self.memory_db.add_memory(
            content=fact,
            tags=["user-fact", f"user-{user_id}", "preference"],
            metadata={
                "user_id": user_id,
                "fact_type": "preference",
                "remembered_at": self._get_timestamp()
            }
        )
    
    def get_user_preferences(self, user_id, topic_query="preferences"):
        """Retrieve user preferences related to a topic"""
        query = f"preferences for user {user_id} about {topic_query}"
        results = self.memory_db.search_memories(
            query=query,
            tags=[f"user-{user_id}", "preference"],
            limit=10
        )
        return [r['content'] for r in results if r['similarity'] > 0.3]
    
    def suggest_based_on_memory(self, user_id, current_context):
        """Suggest relevant information based on current context"""
        user_memories = self.memory_db.search_memories(
            query=current_context,
            tags=[f"user-{user_id}"],
            limit=5
        )
        
        suggestions = []
        for memory in user_memories:
            if memory['similarity'] > 0.4:
                suggestions.append({
                    'suggestion': memory['content'],
                    'relevance': memory['similarity']
                })
        
        return sorted(suggestions, key=lambda x: x['relevance'], reverse=True)
    
    def _get_timestamp(self):
        from datetime import datetime
        return datetime.now().isoformat()

# Usage example
assistant = PersonalAssistantMemory()

# Remember user preferences
assistant.remember_user_fact("john_doe", "prefers dark mode in applications")
assistant.remember_user_fact("john_doe", "works as a software developer")
assistant.remember_user_fact("john_doe", "likes to drink coffee in the morning")

# Later, retrieve preferences
preferences = assistant.get_user_preferences("john_doe", "work")
print("Work-related preferences:", preferences)

# Get suggestions based on current context
suggestions = assistant.suggest_based_on_memory("john_doe", "working on computer")
for suggestion in suggestions:
    print(f"Suggested: {suggestion['suggestion']} (relevance: {suggestion['relevance']:.2f})")
```

## Troubleshooting

### Common Issues and Solutions

#### Issue: API Authentication Failed
**Symptoms**: Error messages mentioning authentication or invalid credentials
**Solution**: 
1. Verify your API credentials are correct
2. Check that you've exported them as environment variables
3. Confirm your Baidu Qianfan account has access to Embedding-V1

```bash
# Verify environment variables are set
echo $BAIDU_API_STRING
echo $BAIDU_SECRET_KEY
```

#### Issue: Slow Search Performance
**Symptoms**: Searches taking several seconds
**Solution**:
1. This is expected for large memory sets without indexing
2. Consider periodically archiving old memories
3. Use tags to narrow down search scope
4. Limit the number of results returned

#### Issue: Similarity Scores Always Low
**Symptoms**: Search results with very low similarity scores (< 0.1)
**Solution**:
1. This could be normal depending on your data
2. Try increasing the search limit to see if more results appear
3. Consider the nature of your queries and stored content

#### Issue: Database File Permissions
**Symptoms**: Errors when adding memories
**Solution**: Check that the directory has write permissions
```bash
ls -la ~/.clawd/
chmod 755 ~/.clawd/  # if needed
```

### Debugging Tips

1. **Check API connectivity**:
   ```python
   # Test basic API functionality
   from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB
   db = MemoryBaiduEmbeddingDB()
   # Try adding a simple memory
   success = db.add_memory("test connection")
   print(f"Connection test: {success}")
   ```

2. **Verify database creation**:
   ```python
   import os
   db_path = os.path.expanduser("~/.clawd/memory_baidu.db")
   print(f"Database exists: {os.path.exists(db_path)}")
   print(f"Database size: {os.path.getsize(db_path) if os.path.exists(db_path) else 0} bytes")
   ```

3. **Test search functionality**:
   ```python
   # Add a known memory
   db.add_memory("The user likes programming", tags=["interest"])
   
   # Search for similar concepts
   results = db.search_memories("coding interests", tags=["interest"])
   print(f"Found {len(results)} results for 'coding interests'")
   ```

## Best Practices

### Memory Organization
- Use consistent tagging conventions
- Include metadata for important memories
- Regularly review and clean up obsolete memories

### Performance Optimization
- Use tags to narrow search scope
- Limit result sets for better performance
- Consider archiving old memories that are rarely accessed

### Security
- Never hardcode API credentials in code
- Use environment variables or secure credential management
- Regularly rotate API keys

### Data Quality
- Validate content before storing
- Consider implementing content filtering
- Monitor for duplicate memories

## Conclusion

The Memory Baidu Embedding DB provides a powerful semantic memory system for your Clawdbot applications. By leveraging Baidu's Embedding-V1 technology, it enables sophisticated memory retrieval based on meaning rather than exact keyword matches.

Remember to:
- Keep your API credentials secure
- Monitor performance as your memory grows
- Regularly maintain and organize your stored memories
- Test thoroughly in your specific use case

Happy coding!