# Memory Baidu Embedding DB - 错误处理最佳实践

## 概述
本文档介绍了Memory Baidu Embedding DB技能中的错误处理机制以及如何在集成时遵循最佳实践。

## 错误处理原则

### 1. 预防优于修复
- 在执行任何操作之前验证输入参数
- 检查必需的环境配置
- 验证API凭据的有效性

### 2. 用户友好的错误信息
- 提供清晰、具体的错误描述
- 指明问题的根本原因
- 提供解决问题的具体步骤

### 3. 优雅降级
- 当某个功能不可用时，提供备选方案
- 不因单一错误导致整个系统崩溃
- 保持核心功能的可用性

## 常见错误场景及处理

### 1. API凭据缺失或无效
```python
# 不推荐
if not api_string or not secret_key:
    raise Exception("Invalid credentials")

# 推荐
if not api_string or not secret_key:
    print("❌ 错误: 缺少必要的API凭据!")
    print("   请设置以下环境变量:")
    print("   export BAIDU_API_STRING='your_bce_v3_api_string'")
    print("   export BAIDU_SECRET_KEY='${BAIDU_SECRET_KEY}'")
    print("   您可以从 https://console.bce.baidu.com/qianfan/ 获取API凭据")
    raise ValueError("缺少百度API凭据")
```

### 2. 输入参数验证
```python
# 不推荐
def add_memory(self, content, tags=None):
    # 直接使用参数，不验证

# 推荐
def add_memory(self, content: str, tags: List[str] = None, metadata: Dict = None) -> bool:
    try:
        # 输入验证
        if not content or not isinstance(content, str):
            print("❌ 错误: 内容不能为空且必须是字符串")
            return False
        
        if len(content) > 10000:  # 限制内容长度
            print("❌ 错误: 内容过长，请保持在10000字符以内")
            return False
            
        if tags is not None and not isinstance(tags, list):
            print("❌ 错误: 标签必须是字符串列表")
            return False
            
        if metadata is not None and not isinstance(metadata, dict):
            print("❌ 错误: 元数据必须是字典类型")
            return False
        
        # 继续执行业务逻辑
        # ...
        
    except Exception as e:
        print(f"❌ 添加记忆时发生未知错误: {str(e)}")
        print("   详细错误信息:")
        traceback.print_exc()
        return False
```

### 3. 数据库操作错误处理
```python
# 推荐的数据库操作模式
def get_all_memories(self) -> List[Dict]:
    try:
        conn = sqlite3.connect(self.db_path)
        cursor = conn.cursor()
        
        cursor.execute('''
            SELECT id, content, embedding_json, timestamp, tags, metadata_json
            FROM memories
            ORDER BY timestamp DESC
        ''')
        
        rows = cursor.fetchall()
        conn.close()
        
        results = []
        for row in rows:
            try:
                results.append({
                    "id": row[0],
                    "content": row[1],
                    "embedding_json": row[2],
                    "timestamp": row[3],
                    "tags": row[4],
                    "metadata": json.loads(row[5]) if row[5] else {},
                })
            except json.JSONDecodeError:
                print(f"⚠️ 警告: 无法解析记忆ID {row[0]} 的元数据，使用空字典")
                results.append({
                    "id": row[0],
                    "content": row[1],
                    "embedding_json": row[2],
                    "timestamp": row[3],
                    "tags": row[4],
                    "metadata": {},
                })
            except Exception as e:
                print(f"⚠️ 警告: 处理记忆ID {row[0]} 时出错: {str(e)}，跳过该记录")
                continue
        
        return results
        
    except sqlite3.Error as e:
        print(f"❌ 获取所有记忆时数据库错误: {str(e)}")
        print("   可能原因: 数据库损坏、权限问题或连接失败")
        return []
    except Exception as e:
        print(f"❌ 获取所有记忆时发生未知错误: {str(e)}")
        print("   详细错误信息:")
        traceback.print_exc()
        return []
```

## 集成时的最佳实践

### 1. 在调用方进行错误处理
```python
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

def safe_memory_operation():
    try:
        mem_db = MemoryBaiduEmbeddingDB()
        
        # 添加记忆
        success = mem_db.add_memory("用户偏好信息")
        if not success:
            print("警告: 记忆添加失败，使用默认设置")
            # 实现备用逻辑
        
        # 搜索记忆
        results = mem_db.search_memories("用户偏好", limit=3)
        if not results:
            print("信息: 未找到相关记忆，使用默认行为")
            # 实现默认行为
        
        return results
        
    except ValueError as e:
        print(f"配置错误: {e}")
        # 实现降级策略
        return []
    except Exception as e:
        print(f"系统错误: {e}")
        # 实现全面降级
        return []
```

### 2. 日志记录
```python
import logging

def enhanced_memory_operation():
    logger = logging.getLogger(__name__)
    
    try:
        mem_db = MemoryBaiduEmbeddingDB()
        results = mem_db.search_memories("查询内容")
        
        if results:
            logger.info(f"找到 {len(results)} 个相关记忆")
        else:
            logger.warning("未找到相关记忆")
            
        return results
        
    except ValueError as e:
        logger.error(f"配置错误: {e}")
        raise
    except Exception as e:
        logger.error(f"搜索失败: {e}", exc_info=True)
        return []
```

### 3. 重试机制
```python
import time
from typing import Callable, TypeVar, Optional

T = TypeVar('T')

def retry_on_failure(func: Callable[[], T], max_retries: int = 3, delay: float = 1.0) -> Optional[T]:
    """
    在API调用失败时重试
    """
    for attempt in range(max_retries):
        try:
            return func()
        except Exception as e:
            if attempt == max_retries - 1:  # 最后一次尝试
                print(f"❌ 经过 {max_retries} 次尝试后仍失败: {e}")
                return None
            
            print(f"⚠️ 第 {attempt + 1} 次尝试失败，{delay}秒后重试...")
            time.sleep(delay)
    
    return None

# 使用示例
def safe_search(query: str):
    def search_func():
        mem_db = MemoryBaiduEmbeddingDB()
        return mem_db.search_memories(query, limit=5)
    
    return retry_on_failure(search_func)
```

## 错误分类

### 1. 配置错误 (Configuration Errors)
- 缺少API凭据
- 无效的环境变量
- 数据库路径不可写

### 2. 输入错误 (Input Errors)
- 无效的数据类型
- 超出长度限制的内容
- 格式错误的参数

### 3. 系统错误 (System Errors)
- 数据库连接失败
- 网络连接问题
- API调用超时

### 4. 逻辑错误 (Logic Errors)
- 试图删除不存在的记录
- 操作参数冲突

## 用户体验考虑

### 1. 清晰的错误级别标识
- ❌ 表示严重错误，操作失败
- ⚠️ 表示警告，操作继续但可能不完整
- ✅ 表示成功操作

### 2. 提供解决建议
每个错误信息都应该包括如何解决的建议。

### 3. 保持静默的非关键错误
对于不影响核心功能的错误（如单个数据项解析失败），使用警告而非错误。

## 总结

通过实施这些错误处理最佳实践，Memory Baidu Embedding DB确保了：
1. 系统的稳定性
2. 用户友好的错误反馈
3. 易于调试和维护
4. 优雅的故障降级

记住，良好的错误处理不仅能防止系统崩溃，还能显著改善用户体验。