# Memory Baidu Embedding DB - API Reference

This document provides detailed information about the classes, methods, and parameters available in the Memory Baidu Embedding DB skill.

## Table of Contents
1. [MemoryBaiduEmbeddingDB Class](#memorybaiduembeddingdb-class)
2. [Methods](#methods)
3. [Parameters](#parameters)
4. [Return Types](#return-types)
5. [Error Handling](#error-handling)

## MemoryBaiduEmbeddingDB Class

The main class for interacting with the semantic memory system.

### Constructor

```python
MemoryBaiduEmbeddingDB(db_path: str = None)
```

#### Parameters
- `db_path` (str, optional): Path to the SQLite database file. If not provided, defaults to `~/.clawd/memory_baidu.db`

#### Environment Variables Required
- `BAIDU_API_STRING`: Your Baidu API string in the format `bce-v3/ALTAK-<api_key_part>/<signature>`
- `BAIDU_SECRET_KEY`: Your Baidu secret key

## Methods

### add_memory()

Adds a new memory to the database with semantic vector representation.

```python
def add_memory(self, content: str, tags: List[str] = None, metadata: Dict = None) -> bool
```

#### Parameters
- `content` (str): The text content to be stored as a memory
- `tags` (List[str], optional): List of tags to categorize the memory
- `metadata` (Dict, optional): Additional metadata to store with the memory

#### Returns
- `bool`: True if the memory was added successfully, False otherwise

#### Example
```python
memory_db = MemoryBaiduEmbeddingDB()

# Simple addition
success = memory_db.add_memory("The user likes Python programming")

# With tags and metadata
success = memory_db.add_memory(
    content="The user prefers concise responses",
    tags=["communication", "preference"],
    metadata={"importance": "high", "source": "conversation-2026-01-30"}
)
```

### search_memories()

Searches for memories semantically similar to the query.

```python
def search_memories(self, query: str, limit: int = 5, tags: List[str] = None) -> List[Dict]
```

#### Parameters
- `query` (str): The search query to find similar memories
- `limit` (int, optional): Maximum number of results to return (default: 5)
- `tags` (List[str], optional): List of tags to filter results

#### Returns
- `List[Dict]`: List of memory dictionaries with the following structure:
  ```python
  [
      {
          "id": int,              # Unique identifier for the memory
          "content": str,         # The stored content
          "similarity": float,    # Similarity score (0.0 to 1.0)
          "timestamp": str,       # Timestamp of when it was added
          "tags": str,            # Comma-separated tags string
          "metadata": Dict        # Stored metadata dictionary
      },
      ...
  ]
  ```

#### Example
```python
# Basic search
results = memory_db.search_memories("What does the user like?")

# Search with limit and tags
results = memory_db.search_memories(
    query="programming preferences",
    limit=10,
    tags=["tech-preference"]
)
```

### retrieve_similar_memories()

Retrieves memories similar to the provided content.

```python
def retrieve_similar_memories(self, content: str, limit: int = 5) -> List[Dict]
```

#### Parameters
- `content` (str): Content to find similar memories to
- `limit` (int, optional): Maximum number of results to return (default: 5)

#### Returns
- `List[Dict]`: Same structure as `search_memories()`

#### Example
```python
similar = memory_db.retrieve_similar_memories(
    content="The user enjoys coding",
    limit=3
)
```

### get_all_memories()

Retrieves all memories from the database.

```python
def get_all_memories(self) -> List[Dict]
```

#### Returns
- `List[Dict]`: List of all memory dictionaries with the following structure:
  ```python
  [
      {
          "id": int,                  # Unique identifier for the memory
          "content": str,             # The stored content
          "embedding_json": str,      # JSON string of the vector representation
          "timestamp": str,           # Timestamp of when it was added
          "tags": str,                # Comma-separated tags string
          "metadata": Dict            # Stored metadata dictionary
      },
      ...
  ]
  ```

#### Example
```python
all_memories = memory_db.get_all_memories()
print(f"Total memories: {len(all_memories)}")
```

### delete_memory()

Deletes a specific memory by its ID.

```python
def delete_memory(self, memory_id: int) -> bool
```

#### Parameters
- `memory_id` (int): The ID of the memory to delete

#### Returns
- `bool`: True if the memory was deleted successfully, False otherwise

#### Example
```python
success = memory_db.delete_memory(123)
if success:
    print("Memory deleted successfully")
else:
    print("Memory not found or deletion failed")
```

### clear_all_memories()

Removes all memories from the database.

```python
def clear_all_memories(self) -> bool
```

#### Returns
- `bool`: True if all memories were cleared successfully, False otherwise

⚠️ **Warning**: This operation cannot be undone. All memories will be permanently deleted.

#### Example
```python
success = memory_db.clear_all_memories()
if success:
    print("All memories cleared")
```

### get_statistics()

Retrieves statistics about the memory database.

```python
def get_statistics(self) -> Dict
```

#### Returns
- `Dict`: Statistics dictionary with the following structure:
  ```python
  {
      "total_memories": int,        # Total number of stored memories
      "tag_distribution": Dict,     # Count of memories per tag
      "earliest_memory": str,       # Timestamp of earliest memory
      "latest_memory": str          # Timestamp of latest memory
  }
  ```

#### Example
```python
stats = memory_db.get_statistics()
print(f"Total memories: {stats['total_memories']}")
print(f"Tag distribution: {stats['tag_distribution']}")
```

## Parameters

### Common Parameter Types

- `str`: Standard string parameter
- `List[str]`: List of strings
- `Dict`: Dictionary with string keys and any JSON-serializable values
- `int`: Integer number
- `float`: Floating-point number

### Parameter Validation

- `content` (str): Must be non-empty, maximum 10,000 characters
- `limit` (int): Must be between 1 and 100
- `memory_id` (int): Must be a positive integer
- `tags` (List[str]): Each tag must be 1-50 characters, maximum 10 tags

## Return Types

### Success Indicators

Most methods return either:
- `bool`: For operations that succeed or fail (True/False)
- `List[Dict]`: For query operations returning multiple results
- `Dict`: For operations returning a single object or statistics

### Error Handling

If API credentials are invalid or network issues occur, methods may return:
- Empty results for query methods
- False for boolean-returning methods
- Appropriate error messages printed to console

## Error Handling

### Common Errors

1. **Authentication Error**:
   - Cause: Invalid or missing API credentials
   - Solution: Verify `BAIDU_API_STRING` and `BAIDU_SECRET_KEY` environment variables

2. **Network Error**:
   - Cause: Unable to reach Baidu API servers
   - Solution: Check internet connectivity

3. **Database Error**:
   - Cause: Issues with SQLite database access
   - Solution: Check file permissions for database path

4. **Input Validation Error**:
   - Cause: Invalid parameters passed to methods
   - Solution: Verify parameter types and constraints

### Error Prevention

```python
import os

# Verify environment variables are set
if not os.getenv('BAIDU_API_STRING') or not os.getenv('BAIDU_SECRET_KEY'):
    raise ValueError("Baidu API credentials not set in environment variables")

# Validate inputs before calling methods
def safe_add_memory(memory_db, content, tags=None, metadata=None):
    if not content or len(content.strip()) == 0:
        print("Error: Content cannot be empty")
        return False
    
    if len(content) > 10000:
        print("Error: Content exceeds 10,000 character limit")
        return False
    
    if tags and len(tags) > 10:
        print("Error: Maximum 10 tags allowed")
        return False
    
    return memory_db.add_memory(content, tags, metadata)
```

## Examples

### Complete Usage Example

```python
from memory_baidu_embedding_db import MemoryBaiduEmbeddingDB

def demonstrate_memory_system():
    # Initialize the memory system
    memory_db = MemoryBaiduEmbeddingDB()
    
    # Add some sample memories
    samples = [
        {
            "content": "The user prefers concise responses and enjoys technical discussions",
            "tags": ["communication", "preference"],
            "metadata": {"importance": "high"}
        },
        {
            "content": "The user works as a software engineer with 5 years of experience",
            "tags": ["professional", "background"],
            "metadata": {"verified": True}
        },
        {
            "content": "The user likes to drink coffee every morning",
            "tags": ["habit", "lifestyle"],
            "metadata": {"importance": "low"}
        }
    ]
    
    # Add the samples
    for sample in samples:
        success = memory_db.add_memory(
            sample["content"],
            sample["tags"],
            sample["metadata"]
        )
        print(f"Added memory: {'Success' if success else 'Failed'}")
    
    # Search for related memories
    print("\nSearching for 'technical' related memories:")
    results = memory_db.search_memories("technical topics", limit=5)
    for result in results:
        print(f"- {result['content']} (similarity: {result['similarity']:.3f})")
    
    # Get statistics
    stats = memory_db.get_statistics()
    print(f"\nTotal memories: {stats['total_memories']}")
    
    return memory_db

# Run the demonstration
memory_system = demonstrate_memory_system()
```

This API reference provides comprehensive information for developers looking to integrate the Memory Baidu Embedding DB skill into their Clawdbot applications.