#!/bin/bash
# meeting-prep/scripts/prep.sh — Research attendees for a specific meeting (on-demand)

set -euo pipefail

# --- CONFIG & ARGS ---
PREP_DIR="${PREP_DIR:-$HOME/.config/meeting-prep}"
CONFIG_FILE="$PREP_DIR/config.json"
HISTORY_FILE="$PREP_DIR/brief-history.json"
LOG_FILE="$PREP_DIR/prep-log.json"
BRIEFS_DIR="$PREP_DIR/briefs"

TARGET="$1"
DRY_RUN=false
FORMAT="markdown"
SEND=false

# Parse flags: --dry-run, --format <fmt>, --send
shift
while [ "$#" -gt 0 ]; do
  case "$1" in
    --dry-run) DRY_RUN=true; shift ;;
    --format) FORMAT="$2"; shift; shift ;;
    --send) SEND=true; shift ;;
    *) echo "Unknown parameter: $1"; exit 1 ;;
  esac
done

# --- LOGGING ---
log() {
  echo "[$(date +'%Y-%m-%d %H:%M:%S')] $1"
  if [ "$DRY_RUN" = "false" ]; then
    jq --arg msg "$1" '.preps += [{"timestamp": (now | todate), "message": $msg}]' "$LOG_FILE" > "$LOG_FILE.tmp" && mv "$LOG_FILE.tmp" "$LOG_FILE"
  fi
}

# --- FUNCTIONS ---

# research_person <email> <name>
# Simulates researching a person. In a real implementation, this would
# use `web_search` and `web_fetch` to find LinkedIn, news, etc.
research_person() {
  local email="$1"
  local name="$2"
  log "Researching person: $name <$email>"

  # In a real script, you'd use web_search here:
  # LINKEDIN_URL=$(web_search --query "$name linkedin profile" | jq -r '.[0].url')
  # And then web_fetch to get details.
  
  # For this placeholder, we'll generate mock data.
  COMPANY=$(echo "$email" | cut -d'@' -f2 | cut -d'.' -f1)
  ROLE="VP of Something"
  RECENT_ACTIVITY="Posted on LinkedIn about industry trends."
  HOOK="Shares an interest in AI automation."

  cat <<-EOF
### $name — $ROLE, ${COMPANY^}
- **Background:** Experienced professional with a strong background in their field.
- **Recent:** $RECENT_ACTIVITY
- **Hook:** $HOOK
- **Source:** Mock data (LinkedIn search placeholder)
EOF
}

# research_company <domain>
research_company() {
  local domain="$1"
  log "Researching company: $domain"
  
  # Real implementation: web_search "$domain news" and fetch their site
  
  cat <<-EOF
### Company Context: ${domain^}
- **Recent News:** Recently launched a new product line to great acclaim.
- **Industry:** A leading player in the B2B SaaS space.
- **Source:** Mock data (Web search placeholder)
EOF
}

# --- MAIN LOGIC ---
log "Starting prep for target: $TARGET"
if [ "$DRY_RUN" = "true" ]; then log "Running in DRY-RUN mode"; fi

MEETING_TITLE="Ad-hoc Briefing"
MEETING_TIME=$(date --iso-8601=minutes)
ATTENDEES=()
COMPANIES=()

# Detect if target is a calendar event ID or emails
if [[ "$TARGET" == *@* ]]; then
  log "Target is email(s). Parsing..."
  IFS=',' read -ra EMAILS <<< "$TARGET"
  for email in "${EMAILS[@]}"; do
    name=$(echo "$email" | cut -d'@' -f1 | sed 's/[._]/\ /g' | awk '{for(i=1;i<=NF;i++) $i=toupper(substr($i,1,1)) substr($i,2)} 1')
    ATTENDEES+=("$email|$name")
    COMPANIES+=($(echo "$email" | cut -d'@' -f2))
  done
  MEETING_TITLE="Briefing for ${EMAILS[0]}"
else
  log "Target is a meeting ID. Fetching details from calendar..."
  # In a real script, use gog to get event details:
  # EVENT_JSON=$(gog calendar get "$TARGET")
  # MEETING_TITLE=$(echo "$EVENT_JSON" | jq -r '.summary')
  # MEETING_TIME=$(echo "$EVENT_JSON" | jq -r '.start')
  # ATTENDEES=... parse from attendees list
  MEETING_TITLE="Meeting: $TARGET"
  log "NOTE: Calendar integration is a placeholder. Using mock attendee."
  ATTENDEES+=("john.doe@example.com|John Doe")
  COMPANIES+=("example.com")
fi

UNIQUE_COMPANIES=($(echo "${COMPANIES[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
log "Found ${#ATTENDEES[@]} attendees from ${#UNIQUE_COMPANIES[@]} companies."

# --- GENERATE BRIEF ---
BRIEF_CONTENT=""
BRIEF_CONTENT+="# 🎯 Meeting Brief: $MEETING_TITLE\n"
BRIEF_CONTENT+="**When:** $(date -R -d "$MEETING_TIME")\n\n"
BRIEF_CONTENT+="## 👥 Attendees\n\n"

for attendee_data in "${ATTENDEES[@]}"; do
  IFS='|' read -r email name <<< "$attendee_data"
  BRIEF_CONTENT+=$(research_person "$email" "$name")
  BRIEF_CONTENT+="\n\n"
done

BRIEF_CONTENT+="## 🏢 Company Context\n\n"
for company_domain in "${UNIQUE_COMPANIES[@]}"; do
  BRIEF_CONTENT+=$(research_company "$company_domain")
  BRIEF_CONTENT+="\n\n"
done

BRIEF_CONTENT+="---\n*Brief generated by Meeting Prep skill.*"

# --- OUTPUT ---
MEETING_SLUG=$(echo "$MEETING_TITLE" | tr '[:upper:]' '[:lower:]' | tr -s ' ' '-' | tr -cd 'a-z0-9-')
FILENAME="$(date -d "$MEETING_TIME" +'%Y-%m-%d')-$MEETING_SLUG.md"
FILEPATH="$BRIEFS_DIR/$FILENAME"

if [ "$DRY_RUN" = "true" ]; then
  echo "--- DRY RUN BRIEF ---"
  echo -e "$BRIEF_CONTENT"
  echo "---------------------"
  log "Dry run complete. Brief would be saved to $FILEPATH"
else
  echo -e "$BRIEF_CONTENT" > "$FILEPATH"
  log "Brief saved to $FILEPATH"
  
  # Update history
  jq --arg key "$TARGET" --arg path "$FILEPATH" '.events[$key] = $path' "$HISTORY_FILE" > "$HISTORY_FILE.tmp" && mv "$HISTORY_FILE.tmp" "$HISTORY_FILE"

  if [ "$SEND" = "true" ]; then
    log "Sending brief via configured channel..."
    # Placeholder for `message send` call
    # In a real script:
    # `message send --channel <channel_from_config> --file "$FILEPATH"`
    echo "Message sending placeholder: Brief for '$MEETING_TITLE' sent."
  fi
fi

log "Prep script finished successfully."
