#!/usr/bin/env -S uv run --script
# /// script
# requires-python = ">=3.10"
# dependencies = [
#     "reportlab",
# ]
# ///
"""
Markdown to PDF Converter v2.0
Converts markdown files to clean, formatted PDFs with full feature support.
"""

import argparse
import re
import sys
from pathlib import Path
from typing import List, Tuple, Optional

from reportlab.lib.pagesizes import letter
from reportlab.lib.styles import getSampleStyleSheet, ParagraphStyle
from reportlab.lib.units import inch
from reportlab.lib.enums import TA_LEFT, TA_CENTER
from reportlab.platypus import (
    SimpleDocTemplate, Paragraph, Spacer, Preformatted,
    Table, TableStyle, HRFlowable, ListFlowable, ListItem, Image
)
from reportlab.lib import colors


# Block types
BLOCK_H1 = 'h1'
BLOCK_H2 = 'h2'
BLOCK_H3 = 'h3'
BLOCK_H4 = 'h4'
BLOCK_H5 = 'h5'
BLOCK_H6 = 'h6'
BLOCK_PARA = 'para'
BLOCK_CODE = 'code'
BLOCK_TABLE = 'table'
BLOCK_HR = 'hr'
BLOCK_BULLET_LIST = 'bullet_list'
BLOCK_NUMBER_LIST = 'number_list'
BLOCK_QUOTE = 'quote'
BLOCK_IMAGE = 'image'


def sanitize_text(text: str) -> str:
    """Escape XML special characters and normalize unicode."""
    # Escape XML entities
    text = text.replace('&', '&amp;')
    text = text.replace('<', '&lt;')
    text = text.replace('>', '&gt;')

    # Normalize common unicode characters
    replacements = {
        '\u2013': '-',   # en-dash
        '\u2014': '--',  # em-dash
        '\u2018': "'",   # left single quote
        '\u2019': "'",   # right single quote
        '\u201c': '"',   # left double quote
        '\u201d': '"',   # right double quote
        '\u2026': '...', # ellipsis
        '\u00a0': ' ',   # non-breaking space
    }
    for old, new in replacements.items():
        text = text.replace(old, new)

    return text


def process_inline_formatting(text: str) -> str:
    """Convert inline markdown to reportlab XML tags."""
    # First sanitize
    text = sanitize_text(text)

    # Inline code (must be before bold/italic to avoid conflicts)
    text = re.sub(r'`([^`]+)`', r'<font face="Courier" size="9" color="#c7254e">\1</font>', text)

    # Bold + italic (***text*** or ___text___)
    text = re.sub(r'\*\*\*(.+?)\*\*\*', r'<b><i>\1</i></b>', text)
    text = re.sub(r'___(.+?)___', r'<b><i>\1</i></b>', text)

    # Bold (**text** or __text__)
    text = re.sub(r'\*\*(.+?)\*\*', r'<b>\1</b>', text)
    text = re.sub(r'__(.+?)__', r'<b>\1</b>', text)

    # Italic (*text* or _text_) - be careful not to match **
    text = re.sub(r'(?<!\*)\*(?!\*)(.+?)(?<!\*)\*(?!\*)', r'<i>\1</i>', text)
    text = re.sub(r'(?<!_)_(?!_)(.+?)(?<!_)_(?!_)', r'<i>\1</i>', text)

    # Links [text](url)
    text = re.sub(r'\[([^\]]+)\]\(([^)]+)\)', r'<link href="\2"><u><font color="blue">\1</font></u></link>', text)

    # Strikethrough ~~text~~
    text = re.sub(r'~~(.+?)~~', r'<strike>\1</strike>', text)

    return text


def skip_frontmatter(content: str) -> str:
    """Remove YAML frontmatter from content."""
    if content.startswith('---'):
        # Find the closing ---
        end = content.find('---', 3)
        if end != -1:
            return content[end + 3:].lstrip('\n')
    return content


def parse_table(lines: List[str]) -> Tuple[List[str], List[List[str]]]:
    """Parse markdown table into headers and rows."""
    headers = []
    rows = []

    for i, line in enumerate(lines):
        cells = [c.strip() for c in line.strip('|').split('|')]

        if i == 0:
            headers = cells
        elif i == 1:
            # Separator row (|---|---|), skip it
            continue
        else:
            rows.append(cells)

    return headers, rows


def parse_markdown(content: str, verbose: bool = False) -> List[Tuple[str, any]]:
    """Parse markdown and return list of (type, content) tuples."""
    # Skip frontmatter
    content = skip_frontmatter(content)

    lines = content.split('\n')
    blocks = []
    i = 0

    while i < len(lines):
        line = lines[i]
        stripped = line.strip()

        # Empty line
        if not stripped:
            i += 1
            continue

        # Headers
        if stripped.startswith('#'):
            level = len(stripped) - len(stripped.lstrip('#'))
            if 1 <= level <= 6 and (len(stripped) == level or stripped[level] == ' '):
                header_text = stripped[level:].strip()
                block_type = [BLOCK_H1, BLOCK_H2, BLOCK_H3, BLOCK_H4, BLOCK_H5, BLOCK_H6][level - 1]
                blocks.append((block_type, header_text))
                if verbose:
                    print(f"  [Header {level}] {header_text[:50]}...")
                i += 1
                continue

        # Horizontal rule
        if re.match(r'^(-{3,}|\*{3,}|_{3,})$', stripped):
            blocks.append((BLOCK_HR, None))
            if verbose:
                print("  [Horizontal Rule]")
            i += 1
            continue

        # Code block
        if stripped.startswith('```'):
            code_lines = []
            lang = stripped[3:].strip()
            i += 1
            while i < len(lines) and not lines[i].strip().startswith('```'):
                code_lines.append(lines[i])
                i += 1
            blocks.append((BLOCK_CODE, {'lang': lang, 'code': '\n'.join(code_lines)}))
            if verbose:
                print(f"  [Code Block] {len(code_lines)} lines ({lang or 'plain'})")
            i += 1  # Skip closing ```
            continue

        # Table
        if '|' in stripped and stripped.startswith('|'):
            table_lines = [line]
            i += 1
            while i < len(lines) and '|' in lines[i].strip() and lines[i].strip():
                table_lines.append(lines[i])
                i += 1

            if len(table_lines) >= 2:  # Need at least header + separator
                headers, rows = parse_table(table_lines)
                blocks.append((BLOCK_TABLE, {'headers': headers, 'rows': rows}))
                if verbose:
                    print(f"  [Table] {len(headers)} cols, {len(rows)} rows")
            continue

        # Blockquote
        if stripped.startswith('>'):
            quote_lines = []
            while i < len(lines) and lines[i].strip().startswith('>'):
                quote_lines.append(lines[i].strip()[1:].strip())
                i += 1
            blocks.append((BLOCK_QUOTE, '\n'.join(quote_lines)))
            if verbose:
                print(f"  [Blockquote] {len(quote_lines)} lines")
            continue

        # Bullet list
        if re.match(r'^[-*+]\s', stripped):
            list_items = []
            while i < len(lines):
                item_line = lines[i].strip()
                if re.match(r'^[-*+]\s', item_line):
                    # Check for task list item
                    task_match = re.match(r'^[-*+]\s+\[([ xX])\]\s*(.*)$', item_line)
                    if task_match:
                        checked = task_match.group(1).lower() == 'x'
                        item_text = task_match.group(2)
                        prefix = '[x] ' if checked else '[ ] '
                        list_items.append(prefix + item_text)
                    else:
                        list_items.append(item_line[2:].strip())
                    i += 1
                elif item_line and not re.match(r'^\d+\.\s', item_line) and not item_line.startswith('#'):
                    # Continuation of list item (indented)
                    if list_items:
                        list_items[-1] += ' ' + item_line
                    i += 1
                else:
                    break
            blocks.append((BLOCK_BULLET_LIST, list_items))
            if verbose:
                print(f"  [Bullet List] {len(list_items)} items")
            continue

        # Numbered list
        if re.match(r'^\d+\.\s', stripped):
            list_items = []
            while i < len(lines):
                item_line = lines[i].strip()
                if re.match(r'^\d+\.\s', item_line):
                    list_items.append(re.sub(r'^\d+\.\s*', '', item_line))
                    i += 1
                elif item_line and not re.match(r'^[-*+]\s', item_line) and not item_line.startswith('#'):
                    # Continuation of list item
                    if list_items:
                        list_items[-1] += ' ' + item_line
                    i += 1
                else:
                    break
            blocks.append((BLOCK_NUMBER_LIST, list_items))
            if verbose:
                print(f"  [Numbered List] {len(list_items)} items")
            continue

        # Image ![alt](path)
        image_match = re.match(r'^!\[([^\]]*)\]\(([^)]+)\)$', stripped)
        if image_match:
            alt_text = image_match.group(1)
            img_path = image_match.group(2)
            blocks.append((BLOCK_IMAGE, {'alt': alt_text, 'path': img_path}))
            if verbose:
                print(f"  [Image] {img_path} ({alt_text})")
            i += 1
            continue

        # Regular paragraph - collect until empty line or special element
        para_lines = []
        while i < len(lines):
            current = lines[i]
            current_stripped = current.strip()

            # Stop at empty line
            if not current_stripped:
                break
            # Stop at header
            if current_stripped.startswith('#'):
                break
            # Stop at code block
            if current_stripped.startswith('```'):
                break
            # Stop at horizontal rule
            if re.match(r'^(-{3,}|\*{3,}|_{3,})$', current_stripped):
                break
            # Stop at table
            if '|' in current_stripped and current_stripped.startswith('|'):
                break
            # Stop at blockquote
            if current_stripped.startswith('>'):
                break
            # Stop at list
            if re.match(r'^[-*+]\s', current_stripped) or re.match(r'^\d+\.\s', current_stripped):
                break

            para_lines.append(current_stripped)
            i += 1

        if para_lines:
            blocks.append((BLOCK_PARA, ' '.join(para_lines)))
            if verbose:
                preview = ' '.join(para_lines)[:50]
                print(f"  [Paragraph] {preview}...")

    return blocks


def create_styles() -> dict:
    """Create custom paragraph styles."""
    styles = getSampleStyleSheet()

    # Heading styles with distinct colors
    heading_colors = ['#1a1a2e', '#16213e', '#1f4287', '#2d6187', '#4a7c87', '#5f8a8a']
    heading_sizes = [24, 20, 16, 14, 12, 11]

    for i, (color, size) in enumerate(zip(heading_colors, heading_sizes), 1):
        styles.add(ParagraphStyle(
            name=f'Heading{i}Custom',
            parent=styles['Heading1'],
            fontSize=size,
            textColor=colors.HexColor(color),
            spaceAfter=8 if i >= 4 else 12,
            spaceBefore=12 if i <= 2 else 8,
            fontName='Helvetica-Bold',
        ))

    # Code style
    styles.add(ParagraphStyle(
        name='CodeBlock',
        parent=styles['Normal'],
        fontName='Courier',
        fontSize=8,
        textColor=colors.HexColor('#333333'),
        backColor=colors.HexColor('#f5f5f5'),
        borderPadding=8,
        leftIndent=10,
        rightIndent=10,
    ))

    # Blockquote style
    styles.add(ParagraphStyle(
        name='BlockQuote',
        parent=styles['Normal'],
        fontSize=10,
        textColor=colors.HexColor('#555555'),
        leftIndent=20,
        borderColor=colors.HexColor('#cccccc'),
        borderWidth=2,
        borderPadding=10,
    ))

    # List item style
    styles.add(ParagraphStyle(
        name='ListItem',
        parent=styles['Normal'],
        fontSize=10,
        leftIndent=20,
        spaceBefore=2,
        spaceAfter=2,
    ))

    # Table cell style
    styles.add(ParagraphStyle(
        name='TableCell',
        parent=styles['Normal'],
        fontSize=9,
        alignment=TA_LEFT,
    ))

    # Table header style
    styles.add(ParagraphStyle(
        name='TableHeader',
        parent=styles['Normal'],
        fontSize=9,
        fontName='Helvetica-Bold',
        alignment=TA_LEFT,
    ))

    return styles


def add_page_number(canvas, doc):
    """Add page numbers to the footer."""
    page_num = canvas.getPageNumber()
    text = f"Page {page_num}"
    canvas.saveState()
    canvas.setFont('Helvetica', 9)
    canvas.setFillColor(colors.HexColor('#888888'))
    canvas.drawCentredString(letter[0] / 2, 0.5 * inch, text)
    canvas.restoreState()


def create_pdf(markdown_file: str, output_pdf: Optional[str] = None, verbose: bool = False):
    """Convert markdown to PDF."""

    md_path = Path(markdown_file)
    if not md_path.exists():
        print(f"Error: File not found: {markdown_file}")
        sys.exit(1)

    if output_pdf is None:
        output_pdf = str(md_path.stem) + '.pdf'

    if verbose:
        print(f"Input: {markdown_file}")
        print(f"Output: {output_pdf}")
        print("Parsing markdown...")

    # Read markdown
    with open(markdown_file, 'r', encoding='utf-8') as f:
        content = f.read()

    # Parse markdown
    blocks = parse_markdown(content, verbose)

    if verbose:
        print(f"Found {len(blocks)} blocks")
        print("Building PDF...")

    # Create PDF document
    doc = SimpleDocTemplate(
        output_pdf,
        pagesize=letter,
        rightMargin=0.75 * inch,
        leftMargin=0.75 * inch,
        topMargin=0.75 * inch,
        bottomMargin=0.75 * inch
    )

    styles = create_styles()
    story = []

    # Build story from blocks
    for block_type, content in blocks:
        try:
            if block_type in (BLOCK_H1, BLOCK_H2, BLOCK_H3, BLOCK_H4, BLOCK_H5, BLOCK_H6):
                level = int(block_type[1])
                style_name = f'Heading{level}Custom'
                text = process_inline_formatting(content)
                story.append(Paragraph(text, styles[style_name]))
                story.append(Spacer(1, 0.1 * inch))

            elif block_type == BLOCK_PARA:
                text = process_inline_formatting(content)
                story.append(Paragraph(text, styles['Normal']))
                story.append(Spacer(1, 0.08 * inch))

            elif block_type == BLOCK_CODE:
                lang = content.get('lang', '')
                code = content.get('code', '')
                # Sanitize code (escape XML but keep newlines)
                code = sanitize_text(code)
                code = code.replace('\n', '<br/>')

                # Create code block with background
                code_para = Paragraph(
                    f'<font face="Courier" size="8">{code}</font>',
                    styles['CodeBlock']
                )
                story.append(code_para)
                story.append(Spacer(1, 0.1 * inch))

            elif block_type == BLOCK_TABLE:
                headers = content.get('headers', [])
                rows = content.get('rows', [])

                # Process inline formatting for all cells
                header_paras = [Paragraph(process_inline_formatting(h), styles['TableHeader']) for h in headers]
                row_paras = [[Paragraph(process_inline_formatting(cell), styles['TableCell']) for cell in row] for row in rows]

                table_data = [header_paras] + row_paras

                # Calculate column widths
                num_cols = len(headers)
                available_width = letter[0] - 1.5 * inch
                col_width = available_width / num_cols

                table = Table(table_data, colWidths=[col_width] * num_cols)
                table.setStyle(TableStyle([
                    ('BACKGROUND', (0, 0), (-1, 0), colors.HexColor('#f0f0f0')),
                    ('TEXTCOLOR', (0, 0), (-1, 0), colors.HexColor('#333333')),
                    ('ALIGN', (0, 0), (-1, -1), 'LEFT'),
                    ('FONTNAME', (0, 0), (-1, 0), 'Helvetica-Bold'),
                    ('FONTSIZE', (0, 0), (-1, -1), 9),
                    ('BOTTOMPADDING', (0, 0), (-1, 0), 8),
                    ('TOPPADDING', (0, 0), (-1, 0), 8),
                    ('BACKGROUND', (0, 1), (-1, -1), colors.white),
                    ('GRID', (0, 0), (-1, -1), 0.5, colors.HexColor('#cccccc')),
                    ('ROWBACKGROUNDS', (0, 1), (-1, -1), [colors.white, colors.HexColor('#fafafa')]),
                    ('LEFTPADDING', (0, 0), (-1, -1), 6),
                    ('RIGHTPADDING', (0, 0), (-1, -1), 6),
                    ('TOPPADDING', (0, 1), (-1, -1), 4),
                    ('BOTTOMPADDING', (0, 1), (-1, -1), 4),
                ]))
                story.append(table)
                story.append(Spacer(1, 0.15 * inch))

            elif block_type == BLOCK_HR:
                story.append(Spacer(1, 0.1 * inch))
                story.append(HRFlowable(
                    width="100%",
                    thickness=1,
                    color=colors.HexColor('#cccccc'),
                    spaceBefore=5,
                    spaceAfter=5
                ))
                story.append(Spacer(1, 0.1 * inch))

            elif block_type == BLOCK_BULLET_LIST:
                items = []
                for item_text in content:
                    text = process_inline_formatting(item_text)
                    items.append(ListItem(Paragraph(text, styles['ListItem'])))

                bullet_list = ListFlowable(
                    items,
                    bulletType='bullet',
                    start=None,
                    bulletFontSize=8,
                    leftIndent=15,
                )
                story.append(bullet_list)
                story.append(Spacer(1, 0.08 * inch))

            elif block_type == BLOCK_NUMBER_LIST:
                items = []
                for item_text in content:
                    text = process_inline_formatting(item_text)
                    items.append(ListItem(Paragraph(text, styles['ListItem'])))

                num_list = ListFlowable(
                    items,
                    bulletType='1',
                    bulletFontSize=10,
                    leftIndent=15,
                )
                story.append(num_list)
                story.append(Spacer(1, 0.08 * inch))

            elif block_type == BLOCK_QUOTE:
                text = process_inline_formatting(content)
                # Add a left border effect with indentation
                quote_text = f'<font color="#666666"><i>{text}</i></font>'
                story.append(Paragraph(quote_text, styles['BlockQuote']))
                story.append(Spacer(1, 0.1 * inch))

            elif block_type == BLOCK_IMAGE:
                img_path = content.get('path', '')
                alt_text = content.get('alt', '')

                # Check if image exists relative to markdown file or absolute
                full_img_path = Path(md_path.parent) / img_path
                if not full_img_path.exists():
                    full_img_path = Path(img_path)

                if full_img_path.exists():
                    try:
                        # Load image and scale it to fit page width
                        img = Image(str(full_img_path))
                        available_width = letter[0] - 1.5 * inch
                        available_height = letter[1] - 2.0 * inch

                        # Scale proportionally
                        img_width, img_height = img.drawWidth, img.drawHeight
                        aspect = img_height / float(img_width)

                        if img_width > available_width:
                            img_width = available_width
                            img_height = img_width * aspect

                        if img_height > available_height:
                            img_height = available_height
                            img_width = img_height / aspect

                        img.drawWidth = img_width
                        img.drawHeight = img_height

                        story.append(img)
                        if alt_text:
                            story.append(Paragraph(f'<font size="8" color="#666666"><i>{alt_text}</i></font>', styles['Normal']))
                        story.append(Spacer(1, 0.15 * inch))
                    except Exception as img_e:
                        if verbose:
                            print(f"  Warning: Could not render image {img_path}: {img_e}")
                else:
                    if verbose:
                        print(f"  Warning: Image not found: {img_path}")
                    story.append(Paragraph(f'<font color="red">[Image not found: {img_path}]</font>', styles['Normal']))

        except Exception as e:
            if verbose:
                print(f"  Warning: Failed to render {block_type}: {e}")
            # Continue with other blocks
            continue

    # Build PDF with page numbers
    try:
        doc.build(story, onFirstPage=add_page_number, onLaterPages=add_page_number)
        print(f"PDF created: {output_pdf}")
        return output_pdf
    except Exception as e:
        print(f"Error creating PDF: {e}")
        sys.exit(1)


def main():
    parser = argparse.ArgumentParser(
        description='Convert markdown to PDF with full formatting support',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s report.md                    # Creates report.pdf
  %(prog)s report.md -o final.pdf       # Custom output name
  %(prog)s report.md -v                 # Verbose processing info
"""
    )
    parser.add_argument('markdown_file', help='Input markdown file')
    parser.add_argument('-o', '--output', help='Output PDF file path')
    parser.add_argument('-v', '--verbose', action='store_true', help='Print detailed processing info')

    args = parser.parse_args()
    create_pdf(args.markdown_file, args.output, args.verbose)


if __name__ == '__main__':
    main()
