import http from 'http';

const server = http.createServer((req, res) => {
    // CORS headers
    res.setHeader('Access-Control-Allow-Origin', '*');
    res.setHeader('Access-Control-Allow-Methods', 'POST, GET, OPTIONS');
    res.setHeader('Access-Control-Allow-Headers', 'Content-Type, mcp-session-id');

    if (req.method === 'OPTIONS') {
        res.writeHead(204);
        res.end();
        return;
    }

    console.log(`[Server] ${req.method} ${req.url}`);

    if (req.url === '/mcp' && req.method === 'POST') {
        let body = '';
        req.on('data', chunk => {
            body += chunk.toString();
        });
        req.on('end', () => {
            try {
                const { method, params, id } = JSON.parse(body);
                console.log(`[Server] Received method: ${method}`);

                const sendJson = (data) => {
                    res.writeHead(200, { 'Content-Type': 'application/json' });
                    res.end(JSON.stringify(data));
                };

                if (method === 'tools/list') {
                    sendJson({
                        jsonrpc: '2.0',
                        id: id,
                        result: {
                            tools: [
                                {
                                    name: 'echo',
                                    description: 'Echo back a message',
                                    inputSchema: {
                                        type: 'object',
                                        properties: {
                                            message: { type: 'string' }
                                        }
                                    }
                                },
                                {
                                    name: 'add',
                                    description: 'Add two numbers',
                                    inputSchema: {
                                        type: 'object',
                                        properties: {
                                            a: { type: 'number' },
                                            b: { type: 'number' }
                                        },
                                        required: ['a', 'b']
                                    }
                                }
                            ]
                        }
                    });
                } else if (method === 'tools/call') {
                    const { name, arguments: args } = params;

                    if (name === 'echo') {
                        sendJson({
                            jsonrpc: '2.0',
                            id: id,
                            result: {
                                content: [
                                    { type: 'text', text: `Echo: ${args.message}` }
                                ]
                            }
                        });
                    } else if (name === 'add') {
                        sendJson({
                            jsonrpc: '2.0',
                            id: id,
                            result: {
                                content: [
                                    { type: 'text', text: String(args.a + args.b) }
                                ]
                            }
                        });
                    } else {
                        res.writeHead(404, { 'Content-Type': 'application/json' });
                        res.end(JSON.stringify({
                            jsonrpc: '2.0',
                            id: id,
                            error: { code: -32601, message: 'Method not found' }
                        }));
                    }
                } else {
                    // Other methods
                    sendJson({
                        jsonrpc: '2.0',
                        id: id,
                        result: {}
                    });
                }
            } catch (e) {
                console.error('[Server] Error processing request:', e.message);
                res.writeHead(400);
                res.end('Invalid JSON');
            }
        });
    } else {
        res.writeHead(404);
        res.end('Not Found');
    }
});

const PORT = 3005;
server.listen(PORT, () => {
    console.log(`Test MCP Server running on http://localhost:${PORT}/mcp`);
});
