import Plugin from '../src/index.js';

// Mock OpenClaw API
const mockApi = {
    logger: {
        info: (msg) => console.log('[Plugin Info]', msg),
        error: (msg) => console.error('[Plugin Error]', msg)
    },
    config: {
        plugins: {
            entries: {
                'mcp-integration': {
                    config: {
                        servers: {
                            test: {
                                enabled: true,
                                url: 'http://localhost:3005/mcp'
                            }
                        }
                    }
                }
            }
        }
    },
    registerService: (service) => {
        console.log('[Mock] Service registered:', service.id);
        // Auto-start for testing
        service.start().then(() => {
            console.log('[Mock] Service started');
        });

        // Store stop function for cleanup
        global.serviceStop = service.stop;
    },
    registerTool: (tool) => {
        console.log('[Mock] Tool registered:', tool.name);
        global.toolExecutor = tool;
    }
};

async function runTest() {
    console.log('--- Starting Test Client ---');

    // Initialize plugin
    Plugin(mockApi);

    // Wait for connection
    await new Promise(resolve => setTimeout(resolve, 1000));

    if (!global.toolExecutor) {
        console.error('Tool not registered!');
        process.exit(1);
    }

    try {
        // Test 1: List tools
        console.log('\n--- Testing Action: List ---');
        const listResult = await global.toolExecutor.execute('test-id', { action: 'list' });
        console.log('List Result:', JSON.stringify(listResult, null, 2));

        // Test 2: Call Echo
        console.log('\n--- Testing Action: Call (echo) ---');
        const echoResult = await global.toolExecutor.execute('test-id', {
            action: 'call',
            server: 'test',
            tool: 'echo',
            args: { message: 'Hello World!' }
        });
        console.log('Echo Result:', JSON.stringify(echoResult, null, 2));

        // Test 3: Call Add
        console.log('\n--- Testing Action: Call (add) ---');
        const addResult = await global.toolExecutor.execute('test-id', {
            action: 'call',
            server: 'test',
            tool: 'add',
            args: { a: 10, b: 32 }
        });
        console.log('Add Result:', JSON.stringify(addResult, null, 2));

    } catch (err) {
        console.error('Test Failed:', err);
    } finally {
        // Cleanup
        if (global.serviceStop) {
            console.log('\n--- Stopping Service ---');
            await global.serviceStop();
        }
        process.exit(0);
    }
}

runTest();
