/**
 * Streamable HTTP Transport for MCP SDK
 * Implements the MCP Streamable HTTP transport specification.
 * Server sends responses via GET SSE stream, POST responses may be empty.
 */

export class StreamableHTTPClientTransport {
  constructor(url, options = {}) {
    this._url = url instanceof URL ? url : new URL(url);
    this._sessionId = options.sessionId || null;
    this._abortController = null;
    this._sseAbortController = null;
    this._debug = options.debug || false;
  }

  async start() {
    if (this._abortController) {
      throw new Error('StreamableHTTPClientTransport already started');
    }
    this._abortController = new AbortController();
    if (this._debug) console.log('[Transport] Started');
  }

  /**
   * Open a GET SSE stream to receive server-initiated messages
   */
  _openSseStream() {
    if (this._debug) console.log('[Transport] Opening GET SSE stream');

    this._sseAbortController = new AbortController();

    fetch(this._url, {
      method: 'GET',
      headers: {
        'Accept': 'text/event-stream',
        ...(this._sessionId && { 'mcp-session-id': this._sessionId })
      },
      signal: this._sseAbortController.signal
    }).then(async (response) => {
      if (!response.ok) {
        if (response.status === 405) {
          if (this._debug) console.log('[Transport] Server does not support GET SSE (405)');
          return;
        }
        throw new Error(`GET SSE failed: HTTP ${response.status}`);
      }

      if (this._debug) console.log('[Transport] GET SSE stream opened');

      // Read SSE stream
      const reader = response.body.getReader();
      const decoder = new TextDecoder();
      let buffer = '';

      try {
        while (true) {
          const { value, done } = await reader.read();
          if (done) {
            if (this._debug) console.log('[Transport] GET SSE stream ended');
            break;
          }

          buffer += decoder.decode(value, { stream: true });

          // Parse SSE events from buffer
          const events = this._parseSseEvents(buffer);
          buffer = events.remaining;

          for (const event of events.parsed) {
            if (event.data) {
              try {
                const message = JSON.parse(event.data);
                if (this._debug) console.log('[Transport] SSE message:', JSON.stringify(message).substring(0, 200));
                if (this.onmessage) {
                  this.onmessage(message);
                }
              } catch (e) {
                // Not valid JSON
              }
            }
          }
        }
      } catch (error) {
        if (error.name !== 'AbortError') {
          if (this._debug) console.error('[Transport] SSE stream error:', error.message);
          if (this.onerror) this.onerror(error);
        }
      }
    }).catch((error) => {
      if (error.name !== 'AbortError') {
        if (this._debug) console.error('[Transport] GET SSE error:', error.message);
        if (this.onerror) this.onerror(error);
      }
    });
  }

  /**
   * Parse SSE events from a text buffer.
   * SSE format: fields separated by \n, events separated by \n\n
   */
  _parseSseEvents(buffer) {
    const parsed = [];
    
    // Split on double newline (event boundary)
    const parts = buffer.split(/\n\n|\r\n\r\n/);
    
    // Last part might be incomplete
    const remaining = parts.pop() || '';
    
    for (const part of parts) {
      if (!part.trim()) continue;
      
      const event = { event: null, data: '', id: null };
      const lines = part.split(/\n|\r\n/);
      
      for (const line of lines) {
        const trimmed = line.replace(/\r$/, '');
        if (trimmed.startsWith('event:')) {
          event.event = trimmed.slice(6).trim();
        } else if (trimmed.startsWith('data:')) {
          const data = trimmed.slice(5).trimStart();
          event.data = event.data ? event.data + '\n' + data : data;
        } else if (trimmed.startsWith('id:')) {
          event.id = trimmed.slice(3).trim();
        }
      }
      
      if (event.data) {
        parsed.push(event);
      }
    }

    return { parsed, remaining };
  }

  async send(message) {
    if (this._debug) console.log('[Transport] Sending:', JSON.stringify(message).substring(0, 200));

    try {
      const headers = {
        'Content-Type': 'application/json',
        'Accept': 'application/json, text/event-stream',
      };

      if (this._sessionId) {
        headers['mcp-session-id'] = this._sessionId;
      }

      const response = await fetch(this._url, {
        method: 'POST',
        headers,
        body: JSON.stringify(message),
        signal: this._abortController?.signal
      });

      // Capture session ID
      const sessionId = response.headers.get('mcp-session-id');
      if (sessionId) {
        this._sessionId = sessionId;
        if (this._debug) console.log('[Transport] Session ID:', sessionId);
      }

      if (!response.ok) {
        const text = await response.text().catch(() => '');
        throw new Error(`HTTP ${response.status}: ${text || response.statusText}`);
      }

      // 202 Accepted = notification accepted
      if (response.status === 202) {
        if (response.body) await response.body.cancel().catch(() => {});
        if (this._debug) console.log('[Transport] 202 Accepted');

        // After initialized notification, open GET SSE stream
        if (message.method === 'notifications/initialized') {
          this._openSseStream();
        }
        return;
      }

      const contentType = response.headers.get('content-type') || '';

      if (contentType.includes('text/event-stream')) {
        // POST response is SSE - parse it
        await this._readSseResponse(response.body);
      } else if (contentType.includes('application/json')) {
        const data = await response.json();
        if (this._debug) console.log('[Transport] JSON response');
        const messages = Array.isArray(data) ? data : [data];
        for (const msg of messages) {
          if (this.onmessage) this.onmessage(msg);
        }
      } else {
        // Try to consume and release
        if (response.body) await response.body.cancel().catch(() => {});
      }
    } catch (error) {
      if (error.name === 'AbortError') return;
      if (this._debug) console.error('[Transport] Send error:', error.message);
      if (this.onerror) this.onerror(error);
      throw error;
    }
  }

  async _readSseResponse(body) {
    if (!body) return;

    const reader = body.getReader();
    const decoder = new TextDecoder();
    let buffer = '';

    try {
      while (true) {
        const { value, done } = await reader.read();
        if (done) break;

        buffer += decoder.decode(value, { stream: true });
        const events = this._parseSseEvents(buffer);
        buffer = events.remaining;

        for (const event of events.parsed) {
          if (event.data) {
            try {
              const message = JSON.parse(event.data);
              if (this.onmessage) this.onmessage(message);
            } catch { /* skip */ }
          }
        }
      }
    } catch (error) {
      if (error.name !== 'AbortError') {
        if (this.onerror) this.onerror(error);
      }
    }
  }

  async close() {
    if (this._debug) console.log('[Transport] Closing');
    this._sseAbortController?.abort();
    this._abortController?.abort();
    this._sseAbortController = null;
    this._abortController = null;
    if (this.onclose) this.onclose();
  }

  get sessionId() {
    return this._sessionId;
  }
}
