# Contributing to MCP Integration Plugin

Thank you for your interest in contributing! This document provides guidelines for contributing to the MCP Integration plugin.

## 🚀 Getting Started

### Prerequisites

- Node.js 18.0.0 or higher
- npm or yarn
- OpenClaw 2026.1.0 or higher
- Git

### Development Setup

```bash
# Clone repository
git clone https://github.com/yourorg/mcp-integration.git
cd mcp-integration

# Install dependencies
npm install

# Link to OpenClaw (for testing)
ln -s $(pwd) ~/.openclaw/extensions/mcp-integration

# Restart OpenClaw
openclaw gateway restart
```

## 📝 Development Workflow

### 1. Create a Branch

```bash
git checkout -b feature/your-feature-name
# or
git checkout -b fix/bug-description
```

### 2. Make Changes

- Write clean, readable code
- Follow existing code style
- Add comments for complex logic
- Update documentation if needed

### 3. Test Your Changes

```bash
# Restart OpenClaw to load changes
openclaw gateway restart

# Check logs for errors
openclaw logs | grep MCP

# Test in OpenClaw chat
# "List MCP tools"
# "Call a tool..."
```

### 4. Commit Changes

```bash
# Add files
git add .

# Commit with descriptive message
git commit -m "Add feature: description of what you added"
```

**Commit Message Format:**
```
<type>: <subject>

<body>

<footer>
```

**Types:**
- `feat`: New feature
- `fix`: Bug fix
- `docs`: Documentation only
- `style`: Code style changes (formatting, etc.)
- `refactor`: Code refactoring
- `test`: Adding tests
- `chore`: Maintenance tasks

**Examples:**
```
feat: Add stdio transport support

Implements stdio transport for MCP servers that don't support HTTP.
Adds StdioTransport class with spawn and IPC handling.

Closes #123
```

```
fix: Handle connection timeout correctly

Previously timeouts would cause unhandled promise rejection.
Now properly catches and logs timeout errors.

Fixes #456
```

### 5. Push and Create PR

```bash
# Push to your fork
git push origin feature/your-feature-name

# Create Pull Request on GitHub
# - Clear description of changes
# - Link to related issues
# - Screenshots if UI changes
```

## 🧪 Testing

### Manual Testing

```bash
# Start test MCP server
node test-server.js

# In OpenClaw chat, test:
# 1. List tools
# 2. Call each tool
# 3. Error cases
# 4. Edge cases
```

### Test Checklist

- [ ] Plugin loads without errors
- [ ] Server connection succeeds
- [ ] Tools are listed correctly
- [ ] Tool calls work
- [ ] Errors are handled gracefully
- [ ] Configuration is validated
- [ ] Logs are helpful
- [ ] Documentation is updated

## 📚 Documentation

### When to Update Docs

- Adding new features
- Changing configuration options
- Fixing bugs that affect usage
- Adding examples

### Which Files to Update

| Change Type | Files to Update |
|-------------|-----------------|
| New feature | README.md, API.md, EXAMPLES.md |
| Configuration | CONFIGURATION.md |
| Bug fix | TROUBLESHOOTING.md, CHANGELOG.md |
| Example | EXAMPLES.md |

## 🎨 Code Style

### JavaScript/Node.js

- Use ES6+ features (async/await, arrow functions, etc.)
- Prefer `const` over `let`, avoid `var`
- Use meaningful variable names
- Add JSDoc comments for functions

```javascript
/**
 * Connect to an MCP server
 * @param {string} name - Server name
 * @param {ServerConfig} config - Server configuration
 * @returns {Promise<Tool[]>} Array of available tools
 * @throws {Error} If connection fails
 */
async connectServer(name, config) {
  // Implementation
}
```

### Error Handling

```javascript
// ✅ Good - specific error messages
try {
  await connectServer(name, config);
} catch (error) {
  this.logger.error(`Failed to connect to ${name}: ${error.message}`);
  throw new Error(`Connection failed: ${error.message}`);
}

// ❌ Bad - generic errors
try {
  await connectServer(name, config);
} catch (error) {
  throw new Error('Error');
}
```

### Logging

```javascript
// ✅ Good - structured logging with context
this.logger.info(`[MCP] Connected to ${name}: ${tools.length} tools available`);
this.logger.error(`[MCP] Failed to connect to ${name}:`, {
  url: config.url,
  error: error.message
});

// ❌ Bad - unstructured logging
console.log('connected');
console.log(error);
```

## 🐛 Reporting Bugs

### Bug Report Template

```markdown
**Describe the bug**
Clear description of what the bug is.

**To Reproduce**
Steps to reproduce:
1. Configure plugin with...
2. Connect to server...
3. Call tool...
4. See error

**Expected behavior**
What you expected to happen.

**Actual behavior**
What actually happened.

**Environment:**
- OpenClaw version: [e.g., 2026.1.0]
- Plugin version: [e.g., 0.1.0]
- Node.js version: [e.g., 20.0.0]
- OS: [e.g., Ubuntu 22.04]

**Configuration:**
```json
{
  // Redacted configuration
}
```

**Logs:**
```
[Paste relevant logs here]
```

**Additional context**
Any other information.
```

## ✨ Feature Requests

### Feature Request Template

```markdown
**Is your feature request related to a problem?**
Clear description of the problem.

**Describe the solution you'd like**
What you want to happen.

**Describe alternatives you've considered**
Other solutions you've thought about.

**Additional context**
Mockups, examples, etc.
```

## 🔄 Pull Request Process

1. **Fork** the repository
2. **Create** a feature branch
3. **Make** your changes
4. **Test** thoroughly
5. **Update** documentation
6. **Commit** with clear messages
7. **Push** to your fork
8. **Create** Pull Request

### PR Requirements

- [ ] Code follows style guidelines
- [ ] Changes are tested
- [ ] Documentation is updated
- [ ] Commit messages are clear
- [ ] No merge conflicts
- [ ] Passes CI checks (when available)

### PR Review Process

1. Maintainer reviews code
2. Feedback provided
3. Changes requested (if needed)
4. Approval given
5. PR merged

## 📜 License

By contributing, you agree that your contributions will be licensed under the MIT License.

## 🤝 Code of Conduct

### Our Pledge

We are committed to providing a welcoming and inspiring community for all.

### Our Standards

**Positive behavior:**
- Using welcoming and inclusive language
- Being respectful of differing viewpoints
- Gracefully accepting constructive criticism
- Focusing on what is best for the community

**Unacceptable behavior:**
- Harassment or discriminatory language
- Trolling or inflammatory comments
- Public or private harassment
- Publishing others' private information

### Enforcement

Violations may result in temporary or permanent ban from the project.

## 📞 Getting Help

- **Discord**: Join #plugins channel
- **GitHub**: Open an issue
- **Email**: support@openclaw.ai

## 🙏 Recognition

Contributors will be recognized in:
- CHANGELOG.md
- README.md contributors section
- GitHub contributors page

## 📚 Resources

- [OpenClaw Documentation](https://docs.openclaw.ai)
- [MCP Specification](https://spec.modelcontextprotocol.io)
- [TypeScript Handbook](https://www.typescriptlang.org/docs/)
- [Node.js Best Practices](https://github.com/goldbergyoni/nodebestpractices)

---

**Thank you for contributing!** 🦞
