#!/usr/bin/env python3
"""
mail.py - OpenClaw mail-client skill
IMAP read/search + SMTP send using Python stdlib only.
Credentials: ~/.openclaw/secrets/mail_creds (chmod 600)
Config:       ~/.openclaw/config/mail-client/config.json (generated by setup.py)
"""

import argparse
import email
import email.header
import imaplib
import json
import mimetypes
import os
import pathlib
import smtplib
import sys
from email.mime.base import MIMEBase
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate, make_msgid
from email import encoders

sys.path.insert(0, str(pathlib.Path(__file__).resolve().parent))
from _retry import with_retry

# ---------------------------------------------------------------------------
# Paths
# ---------------------------------------------------------------------------

SKILL_DIR           = pathlib.Path(__file__).resolve().parent.parent
_CONFIG_DIR         = pathlib.Path.home() / ".openclaw" / "config" / "mail-client"
CONFIG_PATH         = _CONFIG_DIR / "config.json"
CONFIG_EXAMPLE_PATH = SKILL_DIR / "config.example.json"
CREDS_PATH          = pathlib.Path.home() / ".openclaw" / "secrets" / "mail_creds"

# ---------------------------------------------------------------------------
# Defaults
# ---------------------------------------------------------------------------

_DEFAULT_CONFIG = {
    "allow_send": False,
    "allow_read": False,
    "allow_search": False,
    "allow_delete": False,
    "smtp_port": 587,
    "imap_port": 993,
    "mail_from": "",
    "default_folder": "INBOX",
    "max_results": 20,
    "max_attachment_bytes": 25_000_000,  # 25 MB total
}

# Extensions considered safe for attachment (lowercase, with dot).
_ALLOWED_EXTENSIONS = {
    ".txt", ".csv", ".json", ".xml", ".yaml", ".yml", ".toml",
    ".pdf", ".doc", ".docx", ".xls", ".xlsx", ".ppt", ".pptx", ".odt", ".ods", ".odp",
    ".png", ".jpg", ".jpeg", ".gif", ".webp", ".svg", ".bmp", ".ico",
    ".zip", ".tar", ".gz", ".bz2", ".7z",
    ".html", ".htm", ".css", ".js", ".ts", ".py", ".md", ".rst",
    ".eml", ".ics", ".vcf",
    ".mp3", ".wav", ".ogg", ".mp4", ".webm",
}

# ---------------------------------------------------------------------------
# Exceptions
# ---------------------------------------------------------------------------


class MailError(RuntimeError):
    """Base exception for mail-client errors."""


class PermissionDeniedError(MailError):
    """Raised when a capability is disabled in config.json."""


# ---------------------------------------------------------------------------
# Credential loader
# ---------------------------------------------------------------------------


def _load_creds(path: pathlib.Path = CREDS_PATH) -> dict:
    """Load credentials from file then override with environment variables.

    Precedence: env vars > creds file.
    Required vars: MAIL_USER, MAIL_APP_KEY, MAIL_SMTP_HOST, MAIL_IMAP_HOST.
    """
    creds: dict = {}
    if path.exists():
        with open(path, "r", encoding="utf-8") as fh:
            for line in fh:
                line = line.strip()
                if not line or line.startswith("#"):
                    continue
                if "=" not in line:
                    continue
                key, _, value = line.partition("=")
                creds[key.strip()] = value.strip()
    # Environment variables override file values
    for key in ("MAIL_USER", "MAIL_APP_KEY", "MAIL_SMTP_HOST", "MAIL_IMAP_HOST"):
        if key in os.environ:
            creds[key] = os.environ[key]
    if not creds.get("MAIL_USER") or not creds.get("MAIL_APP_KEY"):
        raise MailError(
            f"Credentials missing. Provide MAIL_USER / MAIL_APP_KEY via env vars "
            f"or run scripts/setup.py to create {path}."
        )
    return creds


def _load_config(path: pathlib.Path = CONFIG_PATH) -> dict:
    """Load config.json; fall back to defaults if not present."""
    if not path.exists():
        return dict(_DEFAULT_CONFIG)
    with open(path, "r", encoding="utf-8") as fh:
        data = json.load(fh)
    # Merge with defaults so missing keys use safe defaults
    merged = dict(_DEFAULT_CONFIG)
    merged.update(data)
    return merged


# ---------------------------------------------------------------------------
# MailClient
# ---------------------------------------------------------------------------


class MailClient:
    """IMAP + SMTP client for OpenClaw mail-client skill."""

    def __init__(
        self,
        creds: dict | None = None,
        config: dict | None = None,
    ) -> None:
        self._creds = creds if creds is not None else _load_creds()
        self._config = config if config is not None else _load_config()

        self._smtp_host = self._creds.get("MAIL_SMTP_HOST", "")
        self._imap_host = self._creds.get("MAIL_IMAP_HOST", "")
        self._user = self._creds.get("MAIL_USER", "")
        self._app_key = self._creds.get("MAIL_APP_KEY", "")

        self._smtp_port = int(self._config.get("smtp_port", _DEFAULT_CONFIG["smtp_port"]))
        self._imap_port = int(self._config.get("imap_port", _DEFAULT_CONFIG["imap_port"]))
        self._from = self._config.get("mail_from", "") or self._user

        self._default_folder: str = self._config.get(
            "default_folder", _DEFAULT_CONFIG["default_folder"]
        )
        self._max_results: int = int(
            self._config.get("max_results", _DEFAULT_CONFIG["max_results"])
        )

    # -----------------------------------------------------------------------
    # Permission guards
    # -----------------------------------------------------------------------

    def _check_send(self) -> None:
        if not self._config.get("allow_send", False):
            raise PermissionDeniedError(
                "allow_send is false in config.json. "
                "Edit ~/.openclaw/config/mail-client/config.json to enable sending."
            )

    def _check_read(self) -> None:
        if not self._config.get("allow_read", False):
            raise PermissionDeniedError(
                "allow_read is false in config.json. "
                "Edit ~/.openclaw/config/mail-client/config.json to enable reading."
            )

    def _check_search(self) -> None:
        if not self._config.get("allow_search", False):
            raise PermissionDeniedError(
                "allow_search is false in config.json. "
                "Edit ~/.openclaw/config/mail-client/config.json to enable searching."
            )

    def _check_delete(self) -> None:
        if not self._config.get("allow_delete", False):
            raise PermissionDeniedError(
                "allow_delete is false in config.json. "
                "Edit ~/.openclaw/config/mail-client/config.json to enable delete/move."
            )

    # -----------------------------------------------------------------------
    # IMAP connection helper
    # -----------------------------------------------------------------------

    def _imap_connect(self) -> imaplib.IMAP4_SSL:
        """Return an authenticated IMAP4_SSL connection."""
        import ssl as _ssl
        ctx = _ssl.create_default_context()
        def _do():
            imap = imaplib.IMAP4_SSL(self._imap_host, self._imap_port, ssl_context=ctx)
            imap.login(self._user, self._app_key)
            return imap
        return with_retry(_do)

    # -----------------------------------------------------------------------
    # Helpers
    # -----------------------------------------------------------------------

    @staticmethod
    def _decode_header(raw: str | None) -> str:
        """Decode RFC 2047-encoded header value."""
        if raw is None:
            return ""
        parts = email.header.decode_header(raw)
        decoded_parts: list[str] = []
        for fragment, charset in parts:
            if isinstance(fragment, bytes):
                decoded_parts.append(
                    fragment.decode(charset or "utf-8", errors="replace")
                )
            else:
                decoded_parts.append(fragment)
        return "".join(decoded_parts)

    def _folder(self, folder: str | None) -> str:
        return folder if folder is not None else self._default_folder

    @staticmethod
    def _uid_list(imap: imaplib.IMAP4_SSL, criteria: str) -> list[bytes]:
        """Run UID SEARCH and return list of UID byte-strings."""
        _, data = imap.uid("search", None, criteria)
        if not data or not data[0]:
            return []
        return data[0].split()

    @staticmethod
    def _fetch_envelope(imap: imaplib.IMAP4_SSL, uid: bytes) -> dict:
        """Fetch lightweight envelope fields for a UID."""
        _, data = imap.uid("fetch", uid, "(ENVELOPE)")
        if not data or data[0] is None:
            return {}
        # Use RFC822 header fetch to extract key fields
        _, hdr_data = imap.uid(
            "fetch", uid, "(BODY.PEEK[HEADER.FIELDS (FROM TO SUBJECT DATE MESSAGE-ID)])"
        )
        if not hdr_data or hdr_data[0] is None:
            return {}
        raw_headers = hdr_data[0][1] if isinstance(hdr_data[0], tuple) else b""
        msg = email.message_from_bytes(raw_headers)
        return {
            "uid": uid.decode(),
            "from": MailClient._decode_header(msg.get("From", "")),
            "to": MailClient._decode_header(msg.get("To", "")),
            "subject": MailClient._decode_header(msg.get("Subject", "")),
            "date": msg.get("Date", ""),
            "message_id": msg.get("Message-ID", ""),
        }

    # -----------------------------------------------------------------------
    # Attachment helpers
    # -----------------------------------------------------------------------

    def _validate_attachment(self, path: pathlib.Path) -> None:
        """Raise MailError if the file cannot be attached."""
        if not path.exists():
            raise MailError(f"Attachment not found: {path}")
        if not path.is_file():
            raise MailError(f"Attachment is not a regular file: {path}")
        ext = path.suffix.lower()
        if ext not in _ALLOWED_EXTENSIONS:
            raise MailError(
                f"Attachment extension '{ext}' is not allowed. "
                f"Allowed: {', '.join(sorted(_ALLOWED_EXTENSIONS))}"
            )

    @staticmethod
    def _make_attachment(path: pathlib.Path) -> MIMEBase:
        """Build a MIMEBase part for a file."""
        mime_type, _ = mimetypes.guess_type(str(path))
        if mime_type is None:
            mime_type = "application/octet-stream"
        maintype, subtype = mime_type.split("/", 1)
        part = MIMEBase(maintype, subtype)
        part.set_payload(path.read_bytes())
        encoders.encode_base64(part)
        part.add_header(
            "Content-Disposition", "attachment", filename=path.name
        )
        return part

    # -----------------------------------------------------------------------
    # Public API
    # -----------------------------------------------------------------------

    def send(
        self,
        to: str | list[str],
        subject: str,
        body: str,
        html: str | None = None,
        cc: str | list[str] | None = None,
        bcc: str | list[str] | None = None,
        attachments: list[str] | None = None,
    ) -> dict:
        """Send an email via SMTP. Supports optional file attachments."""
        self._check_send()

        if isinstance(to, str):
            to = [to]
        if isinstance(cc, str):
            cc = [cc]
        if isinstance(bcc, str):
            bcc = [bcc]

        recipients = list(to)
        if cc:
            recipients += cc
        if bcc:
            recipients += bcc

        # Validate attachments upfront
        att_paths: list[pathlib.Path] = []
        if attachments:
            max_bytes = int(self._config.get(
                "max_attachment_bytes", _DEFAULT_CONFIG["max_attachment_bytes"]
            ))
            total = 0
            for fpath in attachments:
                p = pathlib.Path(fpath)
                self._validate_attachment(p)
                total += p.stat().st_size
                if total > max_bytes:
                    raise MailError(
                        f"Total attachment size ({total:,} bytes) exceeds "
                        f"limit ({max_bytes:,} bytes)."
                    )
                att_paths.append(p)

        # Build message
        has_attachments = bool(att_paths)
        if has_attachments:
            # mixed envelope: text/html alternatives + file parts
            msg = MIMEMultipart("mixed")
            alt_part = MIMEMultipart("alternative")
            alt_part.attach(MIMEText(body, "plain", "utf-8"))
            if html:
                alt_part.attach(MIMEText(html, "html", "utf-8"))
            msg.attach(alt_part)
            for p in att_paths:
                msg.attach(self._make_attachment(p))
        else:
            msg = MIMEMultipart("alternative")
            msg.attach(MIMEText(body, "plain", "utf-8"))
            if html:
                msg.attach(MIMEText(html, "html", "utf-8"))

        msg["From"] = self._from
        msg["To"] = ", ".join(to)
        if cc:
            msg["Cc"] = ", ".join(cc)
        msg["Subject"] = subject
        msg["Date"] = formatdate(localtime=True)
        msg["Message-ID"] = make_msgid()

        import ssl as _ssl
        ctx = _ssl.create_default_context()
        msg_bytes = msg.as_bytes()
        def _do_send():
            if self._smtp_port == 465:
                with smtplib.SMTP_SSL(self._smtp_host, self._smtp_port, context=ctx) as srv:
                    srv.login(self._user, self._app_key)
                    srv.sendmail(self._from, recipients, msg_bytes)
            else:
                with smtplib.SMTP(self._smtp_host, self._smtp_port) as srv:
                    srv.ehlo()
                    srv.starttls(context=ctx)
                    srv.ehlo()
                    srv.login(self._user, self._app_key)
                    srv.sendmail(self._from, recipients, msg_bytes)
        with_retry(_do_send)

        result = {
            "ok": True,
            "to": to,
            "subject": subject,
            "message_id": msg["Message-ID"],
        }
        if att_paths:
            result["attachments"] = [p.name for p in att_paths]
        return result

    def list_messages(
        self,
        folder: str | None = None,
        limit: int | None = None,
        unseen_only: bool = False,
    ) -> list[dict]:
        """List messages in a folder (newest first)."""
        self._check_read()
        folder = self._folder(folder)
        limit = limit if limit is not None else self._max_results

        imap = self._imap_connect()
        try:
            imap.select(folder, readonly=True)
            criteria = "UNSEEN" if unseen_only else "ALL"
            uids = self._uid_list(imap, criteria)
            uids = list(reversed(uids))[:limit]
            results: list[dict] = []
            for uid in uids:
                envelope = self._fetch_envelope(imap, uid)
                if envelope:
                    results.append(envelope)
        finally:
            imap.logout()

        return results

    def read_message(self, uid: str, folder: str | None = None) -> dict:
        """Fetch a complete message by UID."""
        self._check_read()
        folder = self._folder(folder)

        imap = self._imap_connect()
        try:
            imap.select(folder, readonly=True)
            _, data = imap.uid("fetch", uid.encode(), "(RFC822)")
        finally:
            imap.logout()

        if not data or data[0] is None:
            raise MailError(f"Message UID {uid} not found in {folder}")

        raw = data[0][1] if isinstance(data[0], tuple) else data[0]
        msg = email.message_from_bytes(raw)

        body_plain = ""
        body_html = ""

        if msg.is_multipart():
            for part in msg.walk():
                ct = part.get_content_type()
                if ct == "text/plain" and not body_plain:
                    payload = part.get_payload(decode=True)
                    charset = part.get_content_charset() or "utf-8"
                    body_plain = payload.decode(charset, errors="replace") if payload else ""
                elif ct == "text/html" and not body_html:
                    payload = part.get_payload(decode=True)
                    charset = part.get_content_charset() or "utf-8"
                    body_html = payload.decode(charset, errors="replace") if payload else ""
        else:
            payload = msg.get_payload(decode=True)
            charset = msg.get_content_charset() or "utf-8"
            if msg.get_content_type() == "text/html":
                body_html = payload.decode(charset, errors="replace") if payload else ""
            else:
                body_plain = payload.decode(charset, errors="replace") if payload else ""

        return {
            "uid": uid,
            "from": self._decode_header(msg.get("From", "")),
            "to": self._decode_header(msg.get("To", "")),
            "cc": self._decode_header(msg.get("Cc", "")),
            "subject": self._decode_header(msg.get("Subject", "")),
            "date": msg.get("Date", ""),
            "message_id": msg.get("Message-ID", ""),
            "body": body_plain,
            "body_html": body_html,
        }

    def search_messages(
        self,
        query: str | None = None,
        from_addr: str | None = None,
        subject: str | None = None,
        since: str | None = None,
        before: str | None = None,
        unseen_only: bool = False,
        folder: str | None = None,
        limit: int | None = None,
    ) -> list[dict]:
        """Search messages using IMAP SEARCH criteria."""
        self._check_search()
        folder = self._folder(folder)
        limit = limit if limit is not None else self._max_results

        criteria_parts: list[str] = []

        if unseen_only:
            criteria_parts.append("UNSEEN")
        if from_addr:
            criteria_parts.append(f'FROM "{from_addr}"')
        if subject:
            criteria_parts.append(f'SUBJECT "{subject}"')
        if since:
            criteria_parts.append(f"SINCE {since}")
        if before:
            criteria_parts.append(f"BEFORE {before}")
        if query:
            criteria_parts.append(f'TEXT "{query}"')

        criteria = " ".join(criteria_parts) if criteria_parts else "ALL"

        imap = self._imap_connect()
        try:
            imap.select(folder, readonly=True)
            uids = self._uid_list(imap, criteria)
            uids = list(reversed(uids))[:limit]
            results: list[dict] = []
            for uid in uids:
                envelope = self._fetch_envelope(imap, uid)
                if envelope:
                    results.append(envelope)
        finally:
            imap.logout()

        return results

    def move_message(
        self, uid: str, target_folder: str, folder: str | None = None
    ) -> bool:
        """Move a message from folder to target_folder (requires allow_delete)."""
        self._check_delete()
        folder = self._folder(folder)

        imap = self._imap_connect()
        try:
            imap.select(folder)
            # Try MOVE extension first
            if "MOVE" in imap.capabilities:
                imap.uid("move", uid.encode(), target_folder)
            else:
                # Fallback: COPY then delete
                imap.uid("copy", uid.encode(), target_folder)
                imap.uid("store", uid.encode(), "+FLAGS", r"(\Deleted)")
                imap.expunge()
        finally:
            imap.logout()

        return True

    def mark_read(self, uid: str, folder: str | None = None) -> bool:
        """Mark a message as read."""
        self._check_read()
        folder = self._folder(folder)

        imap = self._imap_connect()
        try:
            imap.select(folder)
            imap.uid("store", uid.encode(), "+FLAGS", r"(\Seen)")
        finally:
            imap.logout()

        return True

    def mark_unread(self, uid: str, folder: str | None = None) -> bool:
        """Mark a message as unread."""
        self._check_read()
        folder = self._folder(folder)

        imap = self._imap_connect()
        try:
            imap.select(folder)
            imap.uid("store", uid.encode(), "-FLAGS", r"(\Seen)")
        finally:
            imap.logout()

        return True

    def delete_message(self, uid: str, folder: str | None = None) -> bool:
        """Permanently delete a message (requires allow_delete)."""
        self._check_delete()
        folder = self._folder(folder)

        imap = self._imap_connect()
        try:
            imap.select(folder)
            imap.uid("store", uid.encode(), "+FLAGS", r"(\Deleted)")
            imap.expunge()
        finally:
            imap.logout()

        return True

    def list_folders(self) -> list[str]:
        """List all IMAP folders."""
        self._check_read()

        imap = self._imap_connect()
        try:
            _, folders_raw = imap.list()
        finally:
            imap.logout()

        folders: list[str] = []
        for item in folders_raw:
            if item is None:
                continue
            if isinstance(item, bytes):
                decoded = item.decode("utf-8", errors="replace")
            else:
                decoded = str(item)
            # Extract folder name after the last delimiter
            parts = decoded.rsplit('"."', 1)
            if len(parts) == 2:
                folders.append(parts[1].strip().strip('"'))
            else:
                # Try space delimiter
                parts2 = decoded.rsplit(" ", 1)
                folders.append(parts2[-1].strip().strip('"'))

        return folders

    def get_quota(self) -> dict:
        """Get IMAP quota if server supports GETQUOTAROOT."""
        imap = self._imap_connect()
        try:
            # Check for QUOTA capability
            if "QUOTA" not in (cap.upper() for cap in imap.capabilities):
                return {"supported": False, "message": "Server does not advertise QUOTA capability"}
            typ, quota_data = imap.getquotaroot("INBOX")
            if typ != "OK":
                return {"supported": False, "message": "GETQUOTAROOT failed"}
            # Parse quota response
            result: dict = {"supported": True, "raw": []}
            for item in quota_data:
                if item:
                    result["raw"].append(
                        item.decode("utf-8", errors="replace")
                        if isinstance(item, bytes)
                        else str(item)
                    )
            return result
        except (imaplib.IMAP4.error, AttributeError) as exc:
            return {"supported": False, "message": str(exc)}
        finally:
            imap.logout()


# ---------------------------------------------------------------------------
# CLI
# ---------------------------------------------------------------------------


def _print_json(data: object) -> None:
    print(json.dumps(data, indent=2, ensure_ascii=False))


def cmd_send(client: MailClient, args: argparse.Namespace) -> None:
    result = client.send(
        to=args.to,
        subject=args.subject,
        body=args.body,
        html=args.html,
        cc=args.cc,
        bcc=args.bcc,
        attachments=args.attachment,
    )
    _print_json(result)


def cmd_list(client: MailClient, args: argparse.Namespace) -> None:
    msgs = client.list_messages(
        folder=args.folder,
        limit=args.limit,
        unseen_only=args.unseen,
    )
    _print_json(msgs)


def cmd_read(client: MailClient, args: argparse.Namespace) -> None:
    msg = client.read_message(uid=args.uid, folder=args.folder)
    _print_json(msg)


def cmd_search(client: MailClient, args: argparse.Namespace) -> None:
    msgs = client.search_messages(
        query=args.query,
        from_addr=args.from_addr,
        subject=args.subject,
        since=args.since,
        before=args.before,
        unseen_only=args.unseen,
        folder=args.folder,
        limit=args.limit,
    )
    _print_json(msgs)


def cmd_move(client: MailClient, args: argparse.Namespace) -> None:
    ok = client.move_message(
        uid=args.uid, target_folder=args.target, folder=args.folder
    )
    _print_json({"ok": ok, "uid": args.uid, "target": args.target})


def cmd_mark_read(client: MailClient, args: argparse.Namespace) -> None:
    ok = client.mark_read(uid=args.uid, folder=args.folder)
    _print_json({"ok": ok, "uid": args.uid, "flag": "\\Seen"})


def cmd_mark_unread(client: MailClient, args: argparse.Namespace) -> None:
    ok = client.mark_unread(uid=args.uid, folder=args.folder)
    _print_json({"ok": ok, "uid": args.uid, "flag": "-\\Seen"})


def cmd_delete(client: MailClient, args: argparse.Namespace) -> None:
    ok = client.delete_message(uid=args.uid, folder=args.folder)
    _print_json({"ok": ok, "uid": args.uid})


def cmd_folders(client: MailClient, _args: argparse.Namespace) -> None:
    _print_json(client.list_folders())


def cmd_quota(client: MailClient, _args: argparse.Namespace) -> None:
    _print_json(client.get_quota())


def cmd_config(_client: MailClient, _args: argparse.Namespace) -> None:
    """Print current effective configuration (no secrets)."""
    cfg = _load_config()
    _print_json(cfg)


def build_parser() -> argparse.ArgumentParser:
    parser = argparse.ArgumentParser(
        prog="mail",
        description="OpenClaw mail-client: IMAP + SMTP via Python stdlib",
    )
    sub = parser.add_subparsers(dest="command", required=True)

    # send
    p_send = sub.add_parser("send", help="Send an email")
    p_send.add_argument("--to", nargs="+", required=True, help="Recipient(s)")
    p_send.add_argument("--subject", required=True)
    p_send.add_argument("--body", required=True, help="Plain-text body")
    p_send.add_argument("--html", default=None, help="HTML body (optional)")
    p_send.add_argument("--cc", nargs="+", default=None)
    p_send.add_argument("--bcc", nargs="+", default=None)
    p_send.add_argument("--attachment", "-a", nargs="+", default=None,
                        help="File path(s) to attach")
    p_send.set_defaults(func=cmd_send)

    # list
    p_list = sub.add_parser("list", help="List messages")
    p_list.add_argument("--folder", default=None)
    p_list.add_argument("--limit", type=int, default=None)
    p_list.add_argument("--unseen", action="store_true", help="Unseen only")
    p_list.set_defaults(func=cmd_list)

    # read
    p_read = sub.add_parser("read", help="Read a message by UID")
    p_read.add_argument("uid", help="Message UID")
    p_read.add_argument("--folder", default=None)
    p_read.set_defaults(func=cmd_read)

    # search
    p_search = sub.add_parser("search", help="Search messages")
    p_search.add_argument("--query", default=None, help="Full-text search")
    p_search.add_argument("--from-addr", dest="from_addr", default=None)
    p_search.add_argument("--subject", default=None)
    p_search.add_argument("--since", default=None, help="Date: DD-Mon-YYYY")
    p_search.add_argument("--before", default=None, help="Date: DD-Mon-YYYY")
    p_search.add_argument("--unseen", action="store_true")
    p_search.add_argument("--folder", default=None)
    p_search.add_argument("--limit", type=int, default=None)
    p_search.set_defaults(func=cmd_search)

    # move
    p_move = sub.add_parser("move", help="Move message to another folder")
    p_move.add_argument("uid", help="Message UID")
    p_move.add_argument("target", help="Target folder name")
    p_move.add_argument("--folder", default=None, help="Source folder")
    p_move.set_defaults(func=cmd_move)

    # mark-read
    p_mr = sub.add_parser("mark-read", help="Mark message as read")
    p_mr.add_argument("uid")
    p_mr.add_argument("--folder", default=None)
    p_mr.set_defaults(func=cmd_mark_read)

    # mark-unread
    p_mu = sub.add_parser("mark-unread", help="Mark message as unread")
    p_mu.add_argument("uid")
    p_mu.add_argument("--folder", default=None)
    p_mu.set_defaults(func=cmd_mark_unread)

    # delete
    p_del = sub.add_parser("delete", help="Delete a message")
    p_del.add_argument("uid")
    p_del.add_argument("--folder", default=None)
    p_del.set_defaults(func=cmd_delete)

    # folders
    p_folders = sub.add_parser("folders", help="List IMAP folders")
    p_folders.set_defaults(func=cmd_folders)

    # quota
    p_quota = sub.add_parser("quota", help="Get mailbox quota")
    p_quota.set_defaults(func=cmd_quota)

    # config
    p_cfg = sub.add_parser("config", help="Show current config (no secrets)")
    p_cfg.set_defaults(func=cmd_config)

    return parser


def main() -> None:
    parser = build_parser()
    args = parser.parse_args()

    try:
        client = MailClient()
        args.func(client, args)
    except PermissionDeniedError as exc:
        print(f"[permission denied] {exc}", file=sys.stderr)
        sys.exit(2)
    except MailError as exc:
        print(f"[mail error] {exc}", file=sys.stderr)
        sys.exit(1)
    except KeyboardInterrupt:
        print("\nAborted.", file=sys.stderr)
        sys.exit(130)


if __name__ == "__main__":
    main()
