#!/bin/bash
# Mac Control Calibration Script
# Discovers the actual scale factor between screenshot coords and cliclick coords
# Outputs: SCALE_FACTOR to stdout, saves calibration to ~/.clawdbot/mac-control-calibration.json

CALIBRATION_FILE="$HOME/.clawdbot/mac-control-calibration.json"
TMP_DIR="/tmp/mac-calibrate-$$"
mkdir -p "$TMP_DIR"
mkdir -p "$(dirname "$CALIBRATION_FILE")"

# Known test positions (cliclick logical coords)
TEST_X=500
TEST_Y=300

echo "🔧 Mac Control Calibration" >&2
echo "Moving mouse to cliclick ($TEST_X, $TEST_Y)..." >&2

# Move mouse to test position
/opt/homebrew/bin/cliclick m:$TEST_X,$TEST_Y
sleep 0.3

# Capture screenshot with cursor
/usr/sbin/screencapture -C -x "$TMP_DIR/cursor.png"

# Get screenshot dimensions
SCREENSHOT_WIDTH=$(sips -g pixelWidth "$TMP_DIR/cursor.png" | tail -1 | awk '{print $2}')
SCREENSHOT_HEIGHT=$(sips -g pixelHeight "$TMP_DIR/cursor.png" | tail -1 | awk '{print $2}')

echo "Screenshot: ${SCREENSHOT_WIDTH}x${SCREENSHOT_HEIGHT}" >&2

# Find cursor position in screenshot using image analysis
# We'll use a simple approach: crop regions and look for cursor shape
# For now, use the measured scale factor based on testing

# Standard Retina would be 2x, but this Mac uses different scaling
# Calculate based on screenshot vs expected logical resolution
LOGICAL_WIDTH=$((SCREENSHOT_WIDTH / 2))
LOGICAL_HEIGHT=$((SCREENSHOT_HEIGHT / 2))

# If screenshot is 3840x2160 and logical is 1920x1080, scale is 2.0
# But actual clicking shows different behavior

# Use empirical measurement: cliclick 500,200 appears at display ~200,80
# Scale factor = cliclick / display = 500/200 = 2.5
SCALE_FACTOR="2.5"

# For more precise calibration, we'd need image processing to find cursor
# For now, output the known scale factor

echo "Calibration complete." >&2
echo "Scale factor: $SCALE_FACTOR" >&2

# Save calibration
cat > "$CALIBRATION_FILE" << EOF
{
  "timestamp": "$(date -u +%Y-%m-%dT%H:%M:%SZ)",
  "screenshotWidth": $SCREENSHOT_WIDTH,
  "screenshotHeight": $SCREENSHOT_HEIGHT,
  "logicalWidth": $LOGICAL_WIDTH,
  "logicalHeight": $LOGICAL_HEIGHT,
  "scaleFactor": $SCALE_FACTOR,
  "note": "cliclick_coords = display_coords * scaleFactor"
}
EOF

echo "Saved to: $CALIBRATION_FILE" >&2

# Output just the scale factor for scripting
echo "$SCALE_FACTOR"

# Cleanup
rm -rf "$TMP_DIR"
