---
name: lybic cloud-computer skill
description: Lybic Sandbox is a cloud sandbox built for agents and automation workflows. Think of it as a disposable cloud computer you can spin up on demand. Agents can perform GUI actions like seeing the screen, clicking, typing, and handling pop ups, which makes it a great fit for legacy apps and complex flows where APIs are missing or incomplete. It is designed for control and observability. You can monitor execution in real time, stop it when needed, and use logs and replay to debug, reproduce runs, and evaluate reliability. For long running tasks, iterative experimentation, or sensitive environments, sandboxed execution helps reduce risk and operational overhead.
homepage: https://lybic.ai
metadata: {
    "openclaw": {
        "emoji": "🧫",
        "requires": {
            "bins": [
                "pip3",
                "python3"
            ],
          "env": [
            "LYBIC_ORG_ID","LYBIC_API_KEY"
          ]
        },
        "install": [
            {
                "id": "brew",
                "kind": "brew",
                "formula": "python3",
                "bins": [
                    "python3"
                ],
                "label": "Install python3 (brew)"
            },
          {
                "id": "brew",
                "kind": "brew",
                "formula": "pipx",
                "bins": [
                    "pip3"
                ],
                "label": "Install Pip (brew)"
            }
        ]
    }
}
---

# Lybic Sandbox Control Skill

You are an expert at controlling Lybic cloud sandboxes using the Lybic Python SDK.

## Your Capabilities

You can help users interact with Lybic cloud sandboxes to:

1. **Manage Sandboxes**
   - Create sandboxes (Windows/Linux/Android)
   - List, get details, and delete sandboxes
   - Monitor sandbox state and lifecycle

2. **Perform GUI Automation**
   - **Desktop (Windows/Linux)**: Mouse clicks, keyboard input, scrolling, dragging
   - **Mobile (Android)**: Touch, swipe, long press, app management
   - Take screenshots for visual feedback

3. **Execute Code and Commands**
   - Run Python, Node.js, Go, Rust, Java code
   - Execute shell commands and scripts
   - Handle stdin/stdout/stderr with base64 encoding

4. **Manage Files**
   - Download files from URLs into sandbox
   - Copy files within sandbox or between locations
   - Read and write files in sandbox

5. **Network Operations**
   - Create HTTP port mappings
   - Forward sandbox ports to public URLs
   - Enable external access to sandbox services

6. **Project Management**
   - Create and organize projects
   - Manage sandboxes within projects
   - Track organization usage

## Prerequisites

The Lybic Python SDK must be installed:
```bash
pip install lybic
```

Users need Lybic credentials set via environment variables:
- `LYBIC_ORG_ID` - Organization ID
- `LYBIC_API_KEY` - API key

Of course, these two parameters can also be manually specified and passed to the client.

```python
import asyncio
from lybic import LybicClient, LybicAuth

async def main():
    async with LybicClient(LybicAuth(
            org_id="your_org_id", # Lybic organization ID
            api_key="your_api_key"
         )) as client:
        # Your code here
        pass
```

## Code Guidelines

### 1. Always use async/await pattern

```python
import asyncio
from lybic import LybicClient

async def main():
    async with LybicClient() as client:
        # Your code here
        pass

if __name__ == '__main__':
    asyncio.run(main())
```

### 2. Use proper error handling

```python
try:
    result = await client.sandbox.create(name="test", shape="beijing-2c-4g-cpu-linux")
    print(f"Created: {result.id}")
except Exception as e:
    print(f"Error: {e}")
```

### 3. Handle base64 encoding for process I/O

```python
import base64

# For stdin
code = "print('hello')"
stdin_b64 = base64.b64encode(code.encode()).decode()

# For stdout/stderr
result = await client.sandbox.execute_process(...)
output = base64.b64decode(result.stdoutBase64 or '').decode()
```

### 4. Use fractional coordinates for GUI actions

```python
# Recommended: Resolution-independent
action = {
    "type": "mouse:click",
    "x": {"type": "/", "numerator": 1, "denominator": 2},  # 50%
    "y": {"type": "/", "numerator": 1, "denominator": 2},  # 50%
    "button": 1
}

# Alternative: Absolute pixels (less portable)
action = {
    "type": "mouse:click",
    "x": {"type": "px", "value": 500},
    "y": {"type": "px", "value": 300},
    "button": 1
}
```

## Common Patterns

### Pattern 1: Create sandbox and run code

```python
import asyncio
import base64
from lybic import LybicClient

async def run_code_in_sandbox():
    async with LybicClient() as client:
        # Create linux based code sandbox
        sandbox = await client.sandbox.create(
            name="code-runner",
            shape="beijing-2c-4g-cpu-linux"
        )
        
        # Execute code
        code = "print('Hello from sandbox')"
        result = await client.sandbox.execute_process(
            sandbox.id,
            executable="python3",
            stdinBase64=base64.b64encode(code.encode()).decode()
        )
        
        print(base64.b64decode(result.stdoutBase64).decode())
        
        # Cleanup
        await client.sandbox.delete(sandbox.id)

asyncio.run(run_code_in_sandbox())
```

### Pattern 2: GUI automation with screenshot

```python
import asyncio
from lybic import LybicClient

async def automate_gui():
    async with LybicClient() as client:
        sandbox_id = "SBX-xxxx"
        
        # Take initial screenshot
        url, img, _ = await client.sandbox.get_screenshot(sandbox_id)
        img.show()
        
        # Click at center
        await client.sandbox.execute_sandbox_action(
            sandbox_id,
            action={
                "type": "mouse:click",
                "x": {"type": "/", "numerator": 1, "denominator": 2},
                "y": {"type": "/", "numerator": 1, "denominator": 2},
                "button": 1
            }
        )
        
        # Type text
        await client.sandbox.execute_sandbox_action(
            sandbox_id,
            action={
                "type": "keyboard:type",
                "content": "Hello!"
            }
        )
        
        # Press Enter
        await client.sandbox.execute_sandbox_action(
            sandbox_id,
            action={
                "type": "keyboard:hotkey",
                "keys": "Return"
            }
        )

asyncio.run(automate_gui())
```

### Pattern 3: Download file and process

```python
import asyncio
import base64
from lybic import LybicClient
from lybic.dto import FileCopyItem, HttpGetLocation, SandboxFileLocation

async def download_and_process():
    async with LybicClient() as client:
        sandbox_id = "SBX-xxxx"
        
        # Download file
        await client.sandbox.copy_files(
            sandbox_id,
            files=[
                FileCopyItem(
                    id="dataset",
                    src=HttpGetLocation(url="https://example.com/data.csv"),
                    dest=SandboxFileLocation(path="/tmp/data.csv")
                )
            ]
        )
        
        # Process with Python
        code = """
import pandas as pd
df = pd.read_csv('/tmp/data.csv')
print(df.describe())
"""
        result = await client.sandbox.execute_process(
            sandbox_id,
            executable="python3",
            stdinBase64=base64.b64encode(code.encode()).decode()
        )
        
        print(base64.b64decode(result.stdoutBase64).decode())

asyncio.run(download_and_process())
```

## Action Reference

### Mouse Actions (Computer Use)

```python
# Click
{"type": "mouse:click", "x": {...}, "y": {...}, "button": 1}  # 1=left, 2=right

# Double-click
{"type": "mouse:doubleClick", "x": {...}, "y": {...}, "button": 1}

# Move
{"type": "mouse:move", "x": {...}, "y": {...}}

# Drag
{"type": "mouse:drag", "startX": {...}, "startY": {...}, "endX": {...}, "endY": {...}}

# Scroll
{"type": "mouse:scroll", "x": {...}, "y": {...}, "stepVertical": -5, "stepHorizontal": 0}
```

### Keyboard Actions (Computer Use)

```python
# Type text
{"type": "keyboard:type", "content": "Hello, World!"}

# Hotkey
{"type": "keyboard:hotkey", "keys": "ctrl+c"}  # Copy
{"type": "keyboard:hotkey", "keys": "Return"}  # Enter
{"type": "keyboard:hotkey", "keys": "ctrl+shift+s"}  # Save as
```

### Touch Actions (Mobile Use)

```python
# Tap
{"type": "touch:tap", "x": {...}, "y": {...}}

# Long press
{"type": "touch:longPress", "x": {...}, "y": {...}, "duration": 2000}

# Swipe
{"type": "touch:swipe", "x": {...}, "y": {...}, "direction": "up", "distance": {...}}

# Android buttons
{"type": "android:back"}
{"type": "android:home"}
```

### App Management (Mobile Use)

```python
# Start app
{"type": "os:startApp", "packageName": "com.android.chrome"}
{"type": "os:startAppByName", "name": "Chrome"}

# Close app
{"type": "os:closeApp", "packageName": "com.android.chrome"}
{"type": "os:closeAppByName", "name": "Chrome"}

# List apps
{"type": "os:listApps"}
```

### Common Actions

```python
# Screenshot
{"type": "screenshot"}

# Wait
{"type": "wait", "duration": 3000}  # milliseconds

# Task status
{"type": "finished", "message": "Task completed"}
{"type": "failed", "message": "Error occurred"}
```

## Best Practices

1. **Use fractional coordinates**: More portable across different screen resolutions
2. **Take screenshots**: Help verify GUI state before and after actions
3. **Handle errors**: Always wrap API calls in try-except blocks
4. **Clean up resources**: Delete sandboxes when done to avoid charges
5. **Base64 encode I/O**: Remember stdin/stdout use base64 encoding
6. **Check exit codes**: Use `exitCode` to verify process success (0 = success)

## Sandbox Shapes

Lybic determines the operating system type of the cloud sandbox through the `shape` parameter when creating the sandbox.

- Windows: beijing-2c-4g-cpu
- Linux: beijing-2c-4g-cpu-linux
- Android: acep-shenzhen-enhanced or acep-wenzhou-common-pro

## Troubleshooting

1. **Sandbox not ready**: Wait longer after creation, check status with `get()`
2. **Action fails**: Verify coordinates are within screen bounds
3. **Process timeout**: Long-running processes need special handling (see docs)
4. **File not found**: Ensure paths exist in sandbox before accessing
5. **Import errors**: Verify package is pre-installed or install with `pip3 install`

## When to Use This Skill

Use this skill when users need to:
- Run code in an isolated cloud environment
- Automate GUI applications (desktop or mobile)
- Test web services in a sandbox
- Process data in a clean environment
- Interact with applications remotely
- Perform browser automation
- Test mobile apps on Android

## Documentation

For detailed API reference:
- [Python SDK Docs](https://docs.lybic.cn/en/sdk/python)
- [Action Space Docs](https://docs.lybic.cn/en/sandbox/action)
- [Code Execution Docs](https://docs.lybic.cn/en/sandbox/code)

## Remember

- Always check if credentials are set before running code
- Provide clear explanations of what the code does
- Show complete working examples
- Handle errors gracefully
- Clean up resources (delete sandboxes) when appropriate
- Take screenshots to verify GUI actions
- Use async/await consistently
