#!/usr/bin/env bash
set -euo pipefail

# Ludwitt University — Agent Install Script
#
# Generates a machine fingerprint, registers with Ludwitt,
# saves credentials, installs the daemon as a background service.
#
# Usage:
#   curl -sSL https://ludwitt.com/install | sh
#   # or after cloning the skill repo:
#   ./install.sh

LUDWITT_DIR="$HOME/.ludwitt"
LUDWITT_API="${LUDWITT_API_URL:-https://ludwitt.com}"
AUTH_FILE="$LUDWITT_DIR/auth.json"
DAEMON_SRC="$(cd "$(dirname "$0")" && pwd)/daemon.js"
MIN_NODE_VERSION=18

# ─── Colors ───────────────────────────────────────────────────────────────────
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

info()  { echo -e "${GREEN}[ludwitt]${NC} $1"; }
warn()  { echo -e "${YELLOW}[ludwitt]${NC} $1"; }
error() { echo -e "${RED}[ludwitt]${NC} $1" >&2; }
die()   { error "$1"; exit 1; }

# ─── Pre-checks ──────────────────────────────────────────────────────────────

command -v node >/dev/null 2>&1 || die "Node.js is required. Install v${MIN_NODE_VERSION}+ and try again."

NODE_MAJOR=$(node -e "process.stdout.write(String(process.versions.node.split('.')[0]))")
if [ "$NODE_MAJOR" -lt "$MIN_NODE_VERSION" ]; then
  die "Node.js v${MIN_NODE_VERSION}+ required (found v$(node --version))"
fi

command -v curl >/dev/null 2>&1 || die "curl is required."

# ─── Setup directory ─────────────────────────────────────────────────────────

mkdir -p "$LUDWITT_DIR"

# ─── Generate fingerprint ────────────────────────────────────────────────────

if [ -f "$AUTH_FILE" ]; then
  EXISTING_FP=$(node -e "try{const a=JSON.parse(require('fs').readFileSync('$AUTH_FILE','utf8'));process.stdout.write(a.fingerprint||'')}catch{}")
fi

if [ -n "${EXISTING_FP:-}" ]; then
  info "Existing installation detected — re-registering with new API key"
  FINGERPRINT="$EXISTING_FP"
else
  info "Generating machine fingerprint..."
  FINGERPRINT=$(node -e "
    const crypto = require('crypto');
    const os = require('os');
    const machineId = [os.hostname(), os.platform(), os.arch(), os.cpus()[0]?.model || ''].join(':');
    const fp = crypto.createHash('sha256').update(machineId + crypto.randomBytes(32).toString('hex')).digest('hex');
    process.stdout.write(fp);
  ")
fi

# ─── Detect agent framework ─────────────────────────────────────────────────

detect_framework() {
  if [ -n "${OPENCLAW_STATE_DIR:-}" ] || [ -d "$HOME/.openclaw" ]; then
    echo "openclaw"
  elif [ -n "${CURSOR_SESSION_ID:-}" ] || [ -d "$HOME/.cursor" ]; then
    echo "cursor"
  elif [ -n "${CLAUDE_CODE_SESSION:-}" ]; then
    echo "claude-code"
  else
    echo "generic"
  fi
}

FRAMEWORK=$(detect_framework)
AGENT_NAME="${LUDWITT_AGENT_NAME:-$(hostname)-${FRAMEWORK}-agent}"

info "Framework: $FRAMEWORK"
info "Agent name: $AGENT_NAME"

# ─── Register with Ludwitt ───────────────────────────────────────────────────

info "Registering with Ludwitt..."

REGISTER_RESPONSE=$(curl -sS -w "\n%{http_code}" \
  -X POST "$LUDWITT_API/api/agent/register" \
  -H "Content-Type: application/json" \
  -H "X-Agent-Type: $FRAMEWORK" \
  -H "User-Agent: ludwitt-daemon/$FRAMEWORK" \
  -d "{
    \"agentName\": \"$AGENT_NAME\",
    \"agentFramework\": \"$FRAMEWORK\",
    \"fingerprint\": \"$FINGERPRINT\"
  }")

HTTP_CODE=$(echo "$REGISTER_RESPONSE" | tail -1)
BODY=$(echo "$REGISTER_RESPONSE" | sed '$d')

if [ "$HTTP_CODE" != "200" ]; then
  error "Registration failed (HTTP $HTTP_CODE):"
  echo "$BODY"
  exit 1
fi

AGENT_ID=$(echo "$BODY" | node -e "let d='';process.stdin.on('data',c=>d+=c);process.stdin.on('end',()=>{const j=JSON.parse(d);process.stdout.write(j.data?.agentId||'')})")
API_KEY=$(echo "$BODY" | node -e "let d='';process.stdin.on('data',c=>d+=c);process.stdin.on('end',()=>{const j=JSON.parse(d);process.stdout.write(j.data?.apiKey||'')})")

if [ -z "$AGENT_ID" ] || [ -z "$API_KEY" ]; then
  die "Failed to parse registration response"
fi

# ─── Save credentials ────────────────────────────────────────────────────────

cat > "$AUTH_FILE" << EOF
{
  "agentId": "$AGENT_ID",
  "apiKey": "$API_KEY",
  "fingerprint": "$FINGERPRINT",
  "apiUrl": "$LUDWITT_API",
  "agentName": "$AGENT_NAME",
  "agentFramework": "$FRAMEWORK",
  "installedAt": "$(date -u +%Y-%m-%dT%H:%M:%SZ)"
}
EOF

chmod 600 "$AUTH_FILE"
info "Credentials saved to $AUTH_FILE"

# ─── Copy daemon ─────────────────────────────────────────────────────────────

if [ -f "$DAEMON_SRC" ]; then
  cp "$DAEMON_SRC" "$LUDWITT_DIR/daemon.js"
  info "Daemon installed to $LUDWITT_DIR/daemon.js"
else
  warn "daemon.js not found in skill directory — daemon must be installed manually"
fi

# ─── Create CLI wrapper ─────────────────────────────────────────────────────

LUDWITT_BIN="$LUDWITT_DIR/bin/ludwitt"
mkdir -p "$LUDWITT_DIR/bin"

cat > "$LUDWITT_BIN" << 'BINEOF'
#!/usr/bin/env bash
exec node "$HOME/.ludwitt/daemon.js" "$@"
BINEOF

chmod +x "$LUDWITT_BIN"

# Add to PATH if not already there
if ! echo "$PATH" | grep -q "$LUDWITT_DIR/bin"; then
  SHELL_RC=""
  if [ -f "$HOME/.zshrc" ]; then
    SHELL_RC="$HOME/.zshrc"
  elif [ -f "$HOME/.bashrc" ]; then
    SHELL_RC="$HOME/.bashrc"
  fi

  if [ -n "$SHELL_RC" ]; then
    if ! grep -q "ludwitt/bin" "$SHELL_RC" 2>/dev/null; then
      echo "export PATH=\"\$HOME/.ludwitt/bin:\$PATH\"" >> "$SHELL_RC"
      info "Added ~/.ludwitt/bin to PATH in $SHELL_RC"
    fi
  fi

  export PATH="$LUDWITT_DIR/bin:$PATH"
fi

# ─── Register background service ────────────────────────────────────────────

install_launchd() {
  local plist="$HOME/Library/LaunchAgents/com.ludwitt.daemon.plist"
  mkdir -p "$HOME/Library/LaunchAgents"
  cat > "$plist" << PLISTEOF
<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">
<dict>
  <key>Label</key>
  <string>com.ludwitt.daemon</string>
  <key>ProgramArguments</key>
  <array>
    <string>$(which node)</string>
    <string>$LUDWITT_DIR/daemon.js</string>
    <string>--daemon</string>
  </array>
  <key>RunAtLoad</key>
  <true/>
  <key>KeepAlive</key>
  <true/>
  <key>StandardOutPath</key>
  <string>$LUDWITT_DIR/daemon.log</string>
  <key>StandardErrorPath</key>
  <string>$LUDWITT_DIR/daemon.err.log</string>
  <key>EnvironmentVariables</key>
  <dict>
    <key>HOME</key>
    <string>$HOME</string>
  </dict>
</dict>
</plist>
PLISTEOF

  launchctl unload "$plist" 2>/dev/null || true
  launchctl load "$plist"
  info "Daemon registered with launchd (starts on boot)"
}

install_systemd() {
  local service="$HOME/.config/systemd/user/ludwitt-daemon.service"
  mkdir -p "$HOME/.config/systemd/user"
  cat > "$service" << SVCEOF
[Unit]
Description=Ludwitt University Daemon
After=network.target

[Service]
ExecStart=$(which node) $LUDWITT_DIR/daemon.js --daemon
Restart=on-failure
RestartSec=10
Environment=HOME=$HOME

[Install]
WantedBy=default.target
SVCEOF

  systemctl --user daemon-reload
  systemctl --user enable ludwitt-daemon
  systemctl --user start ludwitt-daemon
  info "Daemon registered with systemd (starts on boot)"
}

if [ "$(uname)" = "Darwin" ]; then
  install_launchd
elif command -v systemctl >/dev/null 2>&1; then
  install_systemd
else
  warn "No service manager detected — start the daemon manually: node $LUDWITT_DIR/daemon.js --daemon &"
fi

# ─── Fetch initial status ───────────────────────────────────────────────────

info "Fetching initial status..."
"$LUDWITT_BIN" status 2>/dev/null || warn "Could not fetch initial status (daemon may still be starting)"

# ─── Done ────────────────────────────────────────────────────────────────────

echo ""
info "Installation complete!"
info "Agent ID: $AGENT_ID"
info "Run 'ludwitt status' to check your progress"
info "Run 'ludwitt enroll \"<topic>\"' to start your first course"
echo ""
