"""
Longbridge OpenAPI Skill - 长桥证券开放接口技能

这是一个集成 Longbridge OpenAPI SDK 的 Claude Skill，提供港美股交易、实时行情和账户管理功能。

主要功能：
1. 实时行情订阅（港股、美股、A股）
2. 获取股票报价、K线数据
3. 提交、修改、撤销交易订单
4. 查询账户资金和持仓
5. 查询历史订单和成交记录

支持的市场：
- 港股 (HK)
- 美股 (US)
- A股 (SH/SZ)
"""

import os
from typing import List, Optional, Dict, Any
from longbridge.openapi import QuoteContext, TradeContext, Config

class LongbridgeSkill:
    def __init__(self):
        """
        初始化 Longbridge SDK
        需要设置环境变量：
        - LONGBRIDGE_APP_KEY
        - LONGBRIDGE_APP_SECRET
        - LONGBRIDGE_ACCESS_TOKEN
        """
        self.config = Config(
            app_key=os.environ.get("LONGBRIDGE_APP_KEY"),
            app_secret=os.environ.get("LONGBRIDGE_APP_SECRET"),
            access_token=os.environ.get("LONGBRIDGE_ACCESS_TOKEN")
        )
        self.quote_ctx = None
        self.trade_ctx = None

    def _ensure_quote_context(self):
        """确保行情上下文已初始化"""
        if self.quote_ctx is None:
            self.quote_ctx = QuoteContext(self.config)

    def _ensure_trade_context(self):
        """确保交易上下文已初始化"""
        if self.trade_ctx is None:
            self.trade_ctx = TradeContext(self.config)

    async def quote_subscribe(self, symbols: List[str], sub_types: List[str]) -> Dict[str, Any]:
        """
        订阅实时行情数据

        Args:
            symbols: 股票代码列表，如 ['700.HK', 'AAPL.US']
            sub_types: 订阅类型列表，如 ['quote', 'depth']

        Returns:
            订阅结果
        """
        self._ensure_quote_context()

        # 构建订阅类型
        from longbridge.openapi import SubType
        sub_flags = []
        for sub_type in sub_types:
            if sub_type == "quote":
                sub_flags.append(SubType.Quote)
            elif sub_type == "depth":
                sub_flags.append(SubType.Depth)
            elif sub_type == "brokers":
                sub_flags.append(SubType.Brokers)
            elif sub_type == "trade":
                sub_flags.append(SubType.Trade)

        # 订阅
        combined_flags = sub_flags[0]
        for flag in sub_flags[1:]:
            combined_flags |= flag

        self.quote_ctx.subscribe(symbols, combined_flags, is_first_push=True)

        return {
            "status": "success",
            "message": f"已订阅 {len(symbols)} 个标的的行情数据",
            "symbols": symbols,
            "sub_types": sub_types
        }

    async def get_realtime_quote(self, symbols: List[str]) -> List[Dict[str, Any]]:
        """
        获取实时股票报价

        Args:
            symbols: 股票代码列表

        Returns:
            报价信息列表
        """
        self._ensure_quote_context()
        quotes = self.quote_ctx.quote(symbols)

        result = []
        for quote in quotes:
            result.append({
                "symbol": quote.symbol,
                "last_done": float(quote.last_done),
                "open": float(quote.open),
                "high": float(quote.high),
                "low": float(quote.low),
                "volume": quote.volume,
                "turnover": float(quote.turnover),
                "timestamp": quote.timestamp
            })

        return result

    async def get_static_info(self, symbols: List[str]) -> List[Dict[str, Any]]:
        """
        获取股票静态信息

        Args:
            symbols: 股票代码列表

        Returns:
            静态信息列表
        """
        self._ensure_quote_context()
        info_list = self.quote_ctx.static_info(symbols)

        result = []
        for info in info_list:
            result.append({
                "symbol": info.symbol,
                "name_cn": info.name_cn,
                "name_en": info.name_en,
                "exchange": info.exchange,
                "currency": info.currency,
                "lot_size": info.lot_size
            })

        return result

    async def submit_order(
        self,
        symbol: str,
        order_type: str,
        side: str,
        quantity: int,
        time_in_force: str,
        price: Optional[float] = None,
        remark: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        提交交易订单

        Args:
            symbol: 股票代码
            order_type: 订单类型 (LO/MO/ELO等)
            side: 买卖方向 (Buy/Sell)
            quantity: 数量
            time_in_force: 有效期 (Day/GTC/GTD)
            price: 价格（限价单必填）
            remark: 备注

        Returns:
            订单提交结果
        """
        self._ensure_trade_context()

        from longbridge.openapi import OrderType, OrderSide, TimeInForceType

        # 转换订单类型
        order_type_map = {
            "LO": OrderType.LO,
            "MO": OrderType.MO,
            "ELO": OrderType.ELO,
            "ALO": OrderType.ALO
        }

        # 转换买卖方向
        side_map = {
            "Buy": OrderSide.Buy,
            "Sell": OrderSide.Sell
        }

        # 转换有效期
        tif_map = {
            "Day": TimeInForceType.Day,
            "GTC": TimeInForceType.GoodTilCanceled,
            "GTD": TimeInForceType.GoodTilDate
        }

        # 提交订单
        order = self.trade_ctx.submit_order(
            symbol=symbol,
            order_type=order_type_map[order_type],
            side=side_map[side],
            submitted_quantity=quantity,
            time_in_force=tif_map[time_in_force],
            submitted_price=price,
            remark=remark
        )

        return {
            "status": "success",
            "order_id": order.order_id,
            "message": f"订单已提交: {symbol} {side} {quantity}股"
        }

    async def cancel_order(self, order_id: str) -> Dict[str, Any]:
        """
        撤销订单

        Args:
            order_id: 订单ID

        Returns:
            撤单结果
        """
        self._ensure_trade_context()
        self.trade_ctx.cancel_order(order_id)

        return {
            "status": "success",
            "order_id": order_id,
            "message": "订单已撤销"
        }

    async def get_today_orders(
        self,
        symbol: Optional[str] = None,
        status: Optional[List[str]] = None
    ) -> List[Dict[str, Any]]:
        """
        获取当日订单

        Args:
            symbol: 股票代码（可选）
            status: 订单状态列表（可选）

        Returns:
            订单列表
        """
        self._ensure_trade_context()
        orders = self.trade_ctx.today_orders(symbol=symbol, status=status)

        result = []
        for order in orders:
            result.append({
                "order_id": order.order_id,
                "symbol": order.symbol,
                "side": str(order.side),
                "order_type": str(order.order_type),
                "status": str(order.status),
                "quantity": order.quantity,
                "executed_quantity": order.executed_quantity,
                "price": float(order.price) if order.price else None,
                "submitted_at": order.submitted_at
            })

        return result

    async def get_history_orders(
        self,
        start_date: str,
        end_date: str,
        symbol: Optional[str] = None
    ) -> List[Dict[str, Any]]:
        """
        获取历史订单

        Args:
            start_date: 开始日期 (YYYY-MM-DD)
            end_date: 结束日期 (YYYY-MM-DD)
            symbol: 股票代码（可选）

        Returns:
            历史订单列表
        """
        self._ensure_trade_context()

        from datetime import datetime
        start = datetime.strptime(start_date, "%Y-%m-%d")
        end = datetime.strptime(end_date, "%Y-%m-%d")

        orders = self.trade_ctx.history_orders(
            symbol=symbol,
            start_at=start,
            end_at=end
        )

        result = []
        for order in orders:
            result.append({
                "order_id": order.order_id,
                "symbol": order.symbol,
                "side": str(order.side),
                "status": str(order.status),
                "quantity": order.quantity,
                "price": float(order.price) if order.price else None
            })

        return result

    async def get_account_balance(self, currency: Optional[str] = None) -> List[Dict[str, Any]]:
        """
        获取账户余额

        Args:
            currency: 货币类型（可选）

        Returns:
            账户余额列表
        """
        self._ensure_trade_context()
        balances = self.trade_ctx.account_balance()

        result = []
        for balance in balances:
            if currency and balance.currency != currency:
                continue
            result.append({
                "currency": balance.currency,
                "cash": float(balance.cash),
                "buying_power": float(balance.buying_power),
                "net_assets": float(balance.net_assets)
            })

        return result

    async def get_stock_positions(self, symbol: Optional[str] = None) -> List[Dict[str, Any]]:
        """
        获取持仓列表

        Args:
            symbol: 股票代码（可选）

        Returns:
            持仓列表
        """
        self._ensure_trade_context()
        positions = self.trade_ctx.stock_positions(symbol=symbol)

        result = []
        for pos in positions:
            result.append({
                "symbol": pos.symbol,
                "quantity": pos.quantity,
                "available_quantity": pos.available_quantity,
                "cost_price": float(pos.cost_price),
                "market_price": float(pos.market_price) if pos.market_price else None
            })

        return result

    async def get_candlesticks(
        self,
        symbol: str,
        period: str,
        count: int,
        adjust_type: str = "NoAdjust"
    ) -> List[Dict[str, Any]]:
        """
        获取历史K线数据

        Args:
            symbol: 股票代码
            period: K线周期 (1m/5m/15m/30m/60m/day/week/month/year)
            count: 获取数量
            adjust_type: 复权类型 (NoAdjust/ForwardAdjust)

        Returns:
            K线数据列表
        """
        self._ensure_quote_context()

        from longbridge.openapi import Period, AdjustType

        # 转换周期
        period_map = {
            "1m": Period.Min_1,
            "5m": Period.Min_5,
            "15m": Period.Min_15,
            "30m": Period.Min_30,
            "60m": Period.Min_60,
            "day": Period.Day,
            "week": Period.Week,
            "month": Period.Month,
            "year": Period.Year
        }

        # 转换复权类型
        adjust_map = {
            "NoAdjust": AdjustType.NoAdjust,
            "ForwardAdjust": AdjustType.ForwardAdjust
        }

        candlesticks = self.quote_ctx.candlesticks(
            symbol=symbol,
            period=period_map[period],
            count=count,
            adjust_type=adjust_map[adjust_type]
        )

        result = []
        for candle in candlesticks:
            result.append({
                "timestamp": candle.timestamp,
                "open": float(candle.open),
                "high": float(candle.high),
                "low": float(candle.low),
                "close": float(candle.close),
                "volume": candle.volume,
                "turnover": float(candle.turnover)
            })

        return result


# 全局实例
_skill = LongbridgeSkill()

# 工具函数映射
async def quote_subscribe(symbols: List[str], sub_types: List[str]):
    return await _skill.quote_subscribe(symbols, sub_types)

async def get_realtime_quote(symbols: List[str]):
    return await _skill.get_realtime_quote(symbols)

async def get_static_info(symbols: List[str]):
    return await _skill.get_static_info(symbols)

async def submit_order(symbol: str, order_type: str, side: str, quantity: int,
                       time_in_force: str, price: Optional[float] = None,
                       remark: Optional[str] = None):
    return await _skill.submit_order(symbol, order_type, side, quantity,
                                     time_in_force, price, remark)

async def cancel_order(order_id: str):
    return await _skill.cancel_order(order_id)

async def get_today_orders(symbol: Optional[str] = None, status: Optional[List[str]] = None):
    return await _skill.get_today_orders(symbol, status)

async def get_history_orders(start_date: str, end_date: str, symbol: Optional[str] = None):
    return await _skill.get_history_orders(start_date, end_date, symbol)

async def get_account_balance(currency: Optional[str] = None):
    return await _skill.get_account_balance(currency)

async def get_stock_positions(symbol: Optional[str] = None):
    return await _skill.get_stock_positions(symbol)

async def get_candlesticks(symbol: str, period: str, count: int, adjust_type: str = "NoAdjust"):
    return await _skill.get_candlesticks(symbol, period, count, adjust_type)
