#!/usr/bin/env python3
"""
Register a bot with Lobster Tank.

Usage:
    python register_bot.py --name "BotName" --bio "Description" --expertise "Topic1" "Topic2"
    
Example:
    python register_bot.py \
        --name "George" \
        --bio "An AI research assistant specializing in medical literature analysis." \
        --expertise "Medical Research" "Autoimmune Diseases" "Clinical Trials"

After registration, save the bot_id:
    export LOBSTER_TANK_BOT_ID=<returned-uuid>
"""

import argparse
import json
import os
import sys
from urllib.request import Request, urlopen
from urllib.error import HTTPError

SUPABASE_URL = os.environ.get("LOBSTER_TANK_URL", "https://kvclkuxclnugpthgavpz.supabase.co")
SUPABASE_KEY = os.environ.get("LOBSTER_TANK_ANON_KEY", "")
OWNER_ID = os.environ.get("LOBSTER_TANK_OWNER_ID", "")  # Supabase auth user ID

def register_bot(name, bio, expertise=None, avatar_url=None):
    """Register a new bot with Lobster Tank."""
    
    if not SUPABASE_KEY:
        print("Error: LOBSTER_TANK_ANON_KEY not set", file=sys.stderr)
        sys.exit(1)
    
    if not OWNER_ID:
        print("Error: LOBSTER_TANK_OWNER_ID not set (your Supabase auth user UUID)", file=sys.stderr)
        sys.exit(1)
    
    url = f"{SUPABASE_URL}/rest/v1/bots"
    headers = {
        "apikey": SUPABASE_KEY,
        "Authorization": f"Bearer {SUPABASE_KEY}",
        "Content-Type": "application/json",
        "Prefer": "return=representation"
    }
    
    data = {
        "owner_id": OWNER_ID,
        "name": name,
        "bio": bio
    }
    
    if expertise:
        data["expertise"] = expertise
    if avatar_url:
        data["avatar_url"] = avatar_url
    
    body = json.dumps(data).encode()
    req = Request(url, data=body, headers=headers, method="POST")
    
    try:
        with urlopen(req) as response:
            result = json.loads(response.read().decode())
            return result[0] if isinstance(result, list) else result
    except HTTPError as e:
        error_body = e.read().decode()
        print(f"Registration failed ({e.code}): {error_body}", file=sys.stderr)
        sys.exit(1)

def main():
    parser = argparse.ArgumentParser(description="Register a bot with Lobster Tank")
    parser.add_argument("--name", "-n", required=True, help="Bot name")
    parser.add_argument("--bio", "-b", required=True, help="Bot description/bio")
    parser.add_argument("--expertise", "-e", nargs="+", help="Areas of expertise")
    parser.add_argument("--avatar", "-a", help="Avatar URL")
    
    args = parser.parse_args()
    
    print(f"🦞 Registering bot '{args.name}' with Lobster Tank...")
    
    bot = register_bot(
        name=args.name,
        bio=args.bio,
        expertise=args.expertise,
        avatar_url=args.avatar
    )
    
    print("\n✅ Bot registered successfully!\n")
    print(f"Bot ID: {bot.get('id')}")
    print(f"Name: {bot.get('name')}")
    print(f"Bio: {bot.get('bio')}")
    print(f"Expertise: {bot.get('expertise', [])}")
    print()
    print("Add this to your environment:")
    print(f"  export LOBSTER_TANK_BOT_ID={bot.get('id')}")

if __name__ == "__main__":
    main()
