#!/usr/bin/env bash
set -euo pipefail

SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
INSTALL_WRAPPER="$SCRIPT_DIR/install-wrapper.sh"

TARGET_DIR=""
SSH_KEY=""
KNOWN_HOSTS=""
MAPS=()
WAKE_MAPS=()
WAKE_BROADCAST="255.255.255.255"
WAKE_PORT="9"
WAKE_WAIT="20"
WAKE_RETRIES="1"

while [[ $# -gt 0 ]]; do
  case "$1" in
    --target-dir) TARGET_DIR="${2:-}"; shift 2 ;;
    --map) MAPS+=("${2:-}"); shift 2 ;;
    --ssh-key) SSH_KEY="${2:-}"; shift 2 ;;
    --known-hosts) KNOWN_HOSTS="${2:-}"; shift 2 ;;
    --wake-map) WAKE_MAPS+=("${2:-}"); shift 2 ;;
    --wake-broadcast) WAKE_BROADCAST="${2:-}"; shift 2 ;;
    --wake-port) WAKE_PORT="${2:-}"; shift 2 ;;
    --wake-wait) WAKE_WAIT="${2:-}"; shift 2 ;;
    --wake-retries) WAKE_RETRIES="${2:-}"; shift 2 ;;
    -h|--help)
      echo "usage: install-business-pack.sh --target-dir DIR --map tool=user@host [--map tool=user@host ...] [--wake-map host-or-user@host=AA:BB:CC:DD:EE:FF ...] [--wake-broadcast IP] [--wake-port PORT] [--wake-wait SECONDS] [--wake-retries N] [--ssh-key KEY] [--known-hosts FILE]"
      exit 0
      ;;
    *)
      echo "unknown argument: $1" >&2
      exit 1
      ;;
  esac
done

[[ -n "$TARGET_DIR" ]] || { echo "missing --target-dir" >&2; exit 1; }
[[ "${#MAPS[@]}" -gt 0 ]] || { echo "missing --map tool=user@host" >&2; exit 1; }
[[ "$WAKE_PORT" =~ ^[0-9]+$ ]] || { echo "wake-port must be an integer" >&2; exit 1; }
[[ "$WAKE_WAIT" =~ ^[0-9]+$ ]] || { echo "wake-wait must be an integer" >&2; exit 1; }
[[ "$WAKE_RETRIES" =~ ^[0-9]+$ ]] || { echo "wake-retries must be an integer" >&2; exit 1; }

lookup_wake_mac() {
  local host="$1"
  local host_only="${host#*@}"
  local mapping key value

  for mapping in "${WAKE_MAPS[@]}"; do
    [[ "$mapping" == *=* ]] || {
      echo "invalid wake-map: $mapping" >&2
      exit 1
    }
    key="${mapping%%=*}"
    value="${mapping#*=}"
    [[ -n "$key" && -n "$value" ]] || {
      echo "invalid wake-map: $mapping" >&2
      exit 1
    }
    if [[ "$key" == "$host" || "$key" == "$host_only" ]]; then
      printf '%s' "$value"
      return 0
    fi
  done

  return 1
}

for mapping in "${MAPS[@]}"; do
  tool="${mapping%%=*}"
  host="${mapping#*=}"
  [[ -n "$tool" && -n "$host" && "$mapping" == *=* ]] || {
    echo "invalid map: $mapping" >&2
    exit 1
  }

  case "$tool" in
    imsg|remindctl|memo|things|peekaboo|brew|gh)
      remote_bin="/opt/homebrew/bin/$tool"
      ;;
    *)
      echo "unsupported tool: $tool" >&2
      exit 1
      ;;
  esac

  wake_mac=""
  if wake_mac="$(lookup_wake_mac "$host")"; then
    :
  fi

  cmd=(
    "$INSTALL_WRAPPER"
    --name "$tool"
    --host "$host"
    --remote-bin "$remote_bin"
    --target-dir "$TARGET_DIR"
  )
  [[ -n "$SSH_KEY" ]] && cmd+=(--ssh-key "$SSH_KEY")
  [[ -n "$KNOWN_HOSTS" ]] && cmd+=(--known-hosts "$KNOWN_HOSTS")
  if [[ -n "$wake_mac" ]]; then
    cmd+=(
      --wake-mac "$wake_mac"
      --wake-broadcast "$WAKE_BROADCAST"
      --wake-port "$WAKE_PORT"
      --wake-wait "$WAKE_WAIT"
      --wake-retries "$WAKE_RETRIES"
    )
  fi
  "${cmd[@]}"
done
