---
name: linux-to-mac-bridge
description: Make a Linux OpenClaw gateway on the same local network ready for Mac-backed tools by installing explicit SSH wrappers for imsg, remindctl, memo, things, peekaboo, brew, gh, and similar node-local CLIs. Use when a Linux gateway and one or more Mac nodes share a trusted LAN or VLAN, you want publishable skills to depend on Linux-side wrapper paths instead of macOS binary paths, and the Mac nodes are expected to stay reachable or have Wake-on-LAN enabled.
---

# Linux to Mac Bridge

Use this skill when a Linux gateway should expose Mac-backed business tools as stable Linux-side commands.

This is the publishable answer to the “gray macOS skill on Linux gateway” problem:

- do not patch bundled skills
- do not pretend Linux can run macOS binaries
- install explicit wrappers on the Linux gateway
- make public skills depend on those wrapper paths instead
- assume the Linux gateway and Mac nodes are on the same trusted local network

## Use This Skill For

- `imsg`, `remindctl`, `memo`, `things`, `peekaboo`
- Homebrew business tools that live on a connected Mac node
- wrapper-backed skills that should show as ready on a Linux gateway
- capability reports and stable tool ownership mapping
- same-LAN Linux gateway to Mac node setups
- Mac nodes that are kept awake or have Wake-on-LAN enabled

## Do Not Use This Skill For

- Linux-native tools that should just be installed locally
- random internet-routed Mac hosts or untrusted WAN hops
- arbitrary remote shell access
- patching OpenClaw internals so bundled macOS skills lie about support

## Requirements

- Linux gateway and Mac nodes share the same trusted local network or VLAN
- Linux gateway can SSH to the owning Mac nodes
- remote binaries exist and have the needed macOS permissions
- you know which Mac owns each tool
- Mac nodes should stay awake for agent work, or at minimum have Wake-on-LAN enabled if you expect them to be resumed remotely

## Local Network Contract

This skill is designed for a homelab or office-local topology:

- Linux gateway on the same LAN as the Macs
- stable RFC1918 or otherwise local addresses for the Mac nodes
- low-latency SSH between gateway and nodes
- no requirement for public internet routing to the Macs

Default assumption:

- if the gateway cannot reach the Mac over the local network, the wrapper-backed workflow is not healthy

Wake-on-LAN note:

- the wrapper install scripts can embed Wake-on-LAN metadata and retry logic
- if a Mac may sleep, configure a wake map so the Linux-side wrapper can send a magic packet and retry SSH automatically

## Design Contract

- Linux gateway holds the wrappers
- Mac nodes hold the real binaries and OS-level permissions
- public skills depend on the wrapper path, not the remote path
- tool ownership stays explicit and auditable
- the bridge is optimized for same-LAN node reachability, not public-host reachability

## Workflow

### 1. Render A Tool Ownership Map

Run:

```bash
scripts/render-tool-map.sh
```

This gives you a repeatable map such as:

- `imsg -> M1`
- `remindctl -> MacBook Pro`
- `gh -> MacBook Pro`

### 2. Install The Business Pack

Example:

```bash
scripts/install-business-pack.sh \
  --target-dir /home/node/.openclaw/bin \
  --map imsg=agent1@192.168.88.13 \
  --map remindctl=agent2@192.168.88.12 \
  --map memo=agent1@192.168.88.13 \
  --map gh=agent2@192.168.88.12 \
  --wake-map 192.168.88.13=AA:BB:CC:DD:EE:FF \
  --wake-map 192.168.88.12=11:22:33:44:55:66 \
  --wake-wait 20 \
  --wake-retries 2
```

This creates wrapper paths on Linux such as:

- `/home/node/.openclaw/bin/imsg`
- `/home/node/.openclaw/bin/remindctl`
- `/home/node/.openclaw/bin/memo`
- `/home/node/.openclaw/bin/gh`

When a wake map is configured, the generated wrapper will:

- attempt SSH normally first
- send a Wake-on-LAN magic packet if the first attempt fails
- wait the configured number of seconds
- retry the remote command

### 3. Verify The Pack

Run:

```bash
scripts/verify-business-pack.sh --target-dir /home/node/.openclaw/bin
```

This verifies the wrapper executables exist and can be resolved from the gateway side.

It also shows whether Wake-on-LAN is embedded in each installed wrapper.

### 4. Build Public Skills On Top

When publishing a community skill:

- require the Linux wrapper path, not the macOS binary path
- document which Mac is expected to own the tool
- treat the wrapper as the stable API surface

Read [references/skill-readiness.md](references/skill-readiness.md).

## Security Rules

- one wrapper per tool
- one explicit owning Mac per tool
- no generic remote shell bridge
- no secrets stored in the skill folder
- no core patching to force a green badge

## Files

- `scripts/install-wrapper.sh`: create one SSH wrapper for a remote binary
- `scripts/install-business-pack.sh`: install a batch of common Mac-backed business tool wrappers with optional Wake-on-LAN maps
- `scripts/verify-business-pack.sh`: verify the wrapper pack on the Linux gateway and show Wake-on-LAN status
- `scripts/render-tool-map.sh`: print the recommended tool ownership map plus Wake-on-LAN map examples
- `references/skill-readiness.md`: how to make public skills stay ready on Linux without patching core
