#!/usr/bin/env node
/**
 * Docker clean-machine smoke test.
 * Exercises all plugin functionality with NO pre-existing data or Ollama.
 */

import os from "node:os";
import path from "node:path";
import fs from "node:fs";
import { randomUUID } from "node:crypto";

import { resolveDbPath, ensureTables, sqliteQuery, sqliteExec, escapeSqlValue } from "../lib/sqlite.js";
import { checkOllamaHealth } from "../lib/embeddings.js";
import { loadEntitiesFromDb, addEntityToDb, mergeConfigEntities } from "../lib/entities.js";
import { isValidEntity } from "../lib/entities.js";
import { consolidateMemories } from "../lib/consolidation.js";
import { buildFtsContext, buildRecallContext } from "../lib/recall.js";
import { captureFromMessages } from "../lib/capture.js";
import { extractTopicSignature, checkStuck, saveTopicHistory } from "../lib/stuck-detection.js";

const DB = path.join(os.tmpdir(), `docker-smoke-${Date.now()}.db`);
const HIST = path.join(os.tmpdir(), `docker-smoke-hist-${Date.now()}.json`);

let pass = 0;
let fail = 0;

function ok(condition, label) {
  if (condition) {
    pass++;
    console.log(`  \x1b[32mPASS\x1b[0m ${label}`);
  } else {
    fail++;
    console.log(`  \x1b[31mFAIL\x1b[0m ${label}`);
  }
}

// ===== 1. Schema creation from scratch =====
console.log("\n1. Fresh DB schema creation");
{
  ok(ensureTables(DB), "Tables created on fresh DB");
  const tables = sqliteQuery(DB, "SELECT name FROM sqlite_master WHERE type='table' ORDER BY name");
  const names = tables.map(t => t.name);
  ok(names.includes("decisions"), "decisions table");
  ok(names.includes("decisions_fts"), "decisions_fts virtual table");
  ok(names.includes("vectors"), "vectors table");
  ok(names.includes("entities"), "entities table");

  // Idempotent
  ok(ensureTables(DB), "ensureTables idempotent (second call succeeds)");
}

// ===== 2. CRUD operations =====
console.log("\n2. CRUD operations");
{
  const id = randomUUID();
  const now = Date.now();
  ok(sqliteExec(DB, `INSERT INTO decisions (id, session_id, timestamp, category, description, rationale, classification, importance, ttl_class, last_accessed_at, entity, fact_key, fact_value, tags) VALUES ('${id}', 'test', ${now}, 'manual', 'TestBot.lang = JavaScript', 'test', 'ARCHIVE', 0.9, 'stable', ${now}, 'TestBot', 'lang', 'JavaScript', '["test"]')`), "INSERT succeeds");

  const rows = sqliteQuery(DB, "SELECT fact_value FROM decisions WHERE entity = 'TestBot' AND fact_key = 'lang'");
  ok(rows.length === 1 && rows[0].fact_value === "JavaScript", "SELECT returns inserted data");

  ok(sqliteExec(DB, `UPDATE decisions SET fact_value = 'TypeScript' WHERE id = '${id}'`), "UPDATE succeeds");
  const updated = sqliteQuery(DB, `SELECT fact_value FROM decisions WHERE id = '${id}'`);
  ok(updated[0].fact_value === "TypeScript", "UPDATE persisted");

  ok(sqliteExec(DB, `DELETE FROM decisions WHERE id = '${id}'`), "DELETE succeeds");
  const deleted = sqliteQuery(DB, `SELECT COUNT(*) as c FROM decisions WHERE id = '${id}'`);
  ok(deleted[0].c === 0, "DELETE removed row");
}

// ===== 3. FTS5 integration =====
console.log("\n3. FTS5 full-text search");
{
  const now = Date.now();
  // Insert some data for FTS
  for (const [entity, key, value] of [
    ["Alice", "language", "Prefers Python for data science and machine learning"],
    ["Bob", "framework", "Uses React and Next.js for frontend development"],
    ["config", "database", "PostgreSQL with pgvector for production deployments"],
  ]) {
    const id = randomUUID();
    sqliteExec(DB, `INSERT INTO decisions (id, session_id, timestamp, category, description, rationale, classification, importance, ttl_class, last_accessed_at, entity, fact_key, fact_value, tags) VALUES ('${id}', 'test', ${now}, 'manual', '${entity}.${key} = ${escapeSqlValue(value)}', 'test', 'ARCHIVE', 0.8, 'stable', ${now}, '${entity}', '${key}', '${escapeSqlValue(value)}', '["test"]')`);
  }

  const fts = sqliteQuery(DB, `SELECT d.entity, d.fact_value FROM decisions d JOIN decisions_fts fts ON d.rowid = fts.rowid WHERE decisions_fts MATCH '"python"' LIMIT 5`);
  ok(fts.length > 0, `FTS5 search for "python" found ${fts.length} result(s)`);
  ok(fts[0].entity === "Alice", "FTS5 matched correct entity");

  const { lines, ftsIds } = buildFtsContext(DB, "python data science", 5);
  ok(lines.length > 0, `buildFtsContext returned ${lines.length} lines`);
  ok(ftsIds.size > 0, `buildFtsContext found ${ftsIds.size} unique IDs`);

  const ctx = buildRecallContext(lines, ftsIds, []);
  ok(typeof ctx === "string" && ctx.length > 0, `buildRecallContext produced ${ctx.length} chars`);
}

// ===== 4. Entity management =====
console.log("\n4. Entity management");
{
  const merged = mergeConfigEntities(["Alice", "Bob", "config"], []);
  ok(merged.has("alice"), "Config entity lowercased");
  ok(merged.has("config"), "Built-in entity included");

  ok(addEntityToDb(DB, "RuntimeBot", "test"), "addEntityToDb succeeds");
  const dbEntities = loadEntitiesFromDb(DB);
  ok(dbEntities.includes("RuntimeBot"), "Entity loaded from DB");

  const fullMerge = mergeConfigEntities(["Alice"], dbEntities);
  ok(fullMerge.has("runtimebot"), "DB entity in merged set");
  ok(fullMerge.has("alice"), "Config entity in merged set");

  // Validation
  ok(isValidEntity("TestBot", fullMerge), "Known entity accepted");
  ok(isValidEntity("PascalCase", fullMerge), "PascalCase auto-accepted");
  ok(!isValidEntity("randomword", fullMerge), "Unknown entity rejected");
}

// ===== 5. Capture =====
console.log("\n5. Auto-capture from messages");
{
  const entities = mergeConfigEntities(["Alice", "Bob", "config"], []);
  const messages = [
    { role: "assistant", content: "Alice: favorite_editor = VSCode with Vim keybindings and custom theme" },
  ];
  const result = captureFromMessages(DB, messages, 5, entities, () => {});
  ok(typeof result.stored === "number", `captureFromMessages returned stored=${result.stored}`);

  const check = sqliteQuery(DB, "SELECT fact_value FROM decisions WHERE entity = 'Alice' AND fact_key = 'favorite_editor'");
  if (result.stored > 0) {
    ok(check.length > 0 && check[0].fact_value.includes("VSCode"), "Captured fact persisted correctly");
  } else {
    ok(true, "Capture pattern didn't match (acceptable for regex variations)");
  }
}

// ===== 6. Stuck detection =====
console.log("\n6. Stuck detection");
{
  const text1 = "The database uses SQLite with FTS5 for persistent full-text search indexing across all stored memory entries";
  const sig = extractTopicSignature(text1);
  ok(sig !== null, `Signature extracted: ${sig}`);

  saveTopicHistory(HIST, []);
  ok(checkStuck(DB, HIST, sig) === null, "Not stuck (1/3)");
  ok(checkStuck(DB, HIST, sig) === null, "Not stuck (2/3)");
  const nudge = checkStuck(DB, HIST, sig);
  ok(nudge !== null && nudge.includes("reflexion"), "Stuck detected (3/3) with Reflexion nudge");

  // Different topic should not trigger
  saveTopicHistory(HIST, []);
  const sig2 = extractTopicSignature("React components with TypeScript interfaces and Next.js server actions for modern frontend architecture");
  checkStuck(DB, HIST, sig);
  checkStuck(DB, HIST, sig2);
  ok(checkStuck(DB, HIST, sig) === null, "Mixed topics not stuck");
}

// ===== 7. Consolidation =====
console.log("\n7. Memory consolidation");
{
  const now = Date.now();
  for (let i = 0; i < 4; i++) {
    sqliteExec(DB, `INSERT INTO decisions (id, session_id, timestamp, category, description, rationale, classification, importance, ttl_class, last_accessed_at, entity, fact_key, fact_value, tags) VALUES ('${randomUUID()}', 'test', ${now - i * 1000}, 'manual', 'DupBot.dup = same', 'test', 'ARCHIVE', 0.5, 'stable', ${now}, 'DupBot', 'dup', 'same', '["test"]')`);
  }
  const before = sqliteQuery(DB, "SELECT COUNT(*) as c FROM decisions WHERE entity = 'DupBot' AND fact_key = 'dup'");
  ok(before[0].c === 4, `Created 4 duplicates`);

  consolidateMemories(DB, () => {});
  const after = sqliteQuery(DB, "SELECT COUNT(*) as c FROM decisions WHERE entity = 'DupBot' AND fact_key = 'dup'");
  ok(after[0].c === 1, `Consolidated to 1`);
}

// ===== 8. SQL safety =====
console.log("\n8. SQL injection safety");
{
  ok(escapeSqlValue("normal") === "normal", "Normal string unchanged");
  ok(escapeSqlValue("it's") === "it''s", "Single quote doubled");
  ok(escapeSqlValue(null) === "", "null → empty string");
  ok(escapeSqlValue(undefined) === "", "undefined → empty string");
  ok(escapeSqlValue("a\0b") === "ab", "Null bytes removed");
  ok(escapeSqlValue("x".repeat(20000)).length === 10000, "Truncated to 10000 chars");
}

// ===== 9. Ollama graceful degradation =====
console.log("\n9. Ollama graceful degradation (no Ollama in Docker)");
{
  const health = await checkOllamaHealth("http://localhost:11434", "nomic-embed-text");
  ok(health.available === false, `Ollama unavailable in Docker (reason: ${health.reason})`);
}

// ===== 10. Path resolution =====
console.log("\n10. Config path resolution");
{
  ok(resolveDbPath(null).endsWith("decisions.db"), "null → default");
  ok(resolveDbPath(undefined).endsWith("decisions.db"), "undefined → default");
  ok(resolveDbPath("~/test.db").startsWith(os.homedir()), "~ expanded");
  ok(resolveDbPath("/abs/path.db") === "/abs/path.db", "Absolute passthrough");
}

// ===== 11. Plugin register (mock API) =====
console.log("\n11. Plugin register with mock API");
{
  const registered = { services: [], tools: [], hooks: [] };
  const mockApi = {
    pluginConfig: {
      dbPath: DB,
      autoRecall: false,
      autoCapture: false,
      vectorSearch: false,
      stuckDetection: false,
      consolidation: false,
      entities: ["MockUser"],
    },
    logger: { info: () => {}, warn: () => {} },
    registerService: (svc) => registered.services.push(svc),
    registerTool: (tool, opts) => registered.tools.push({ tool, opts }),
    on: (event, handler) => registered.hooks.push({ event, handler }),
  };

  const { default: register } = await import("../index.js");
  try {
    register(mockApi);
    ok(true, "register(api) completed without error");
  } catch (e) {
    ok(false, `register(api) threw: ${e.message}`);
  }

  ok(registered.services.length === 1, `1 service registered`);
  ok(registered.services[0].id === "lily-memory", `Service id = lily-memory`);
  ok(registered.tools.length === 5, `5 tools registered`);

  const toolNames = registered.tools.map(t => t.tool.name);
  ok(toolNames.includes("memory_search"), "memory_search tool");
  ok(toolNames.includes("memory_entity"), "memory_entity tool");
  ok(toolNames.includes("memory_store"), "memory_store tool");
  ok(toolNames.includes("memory_semantic_search"), "memory_semantic_search tool");
  ok(toolNames.includes("memory_add_entity"), "memory_add_entity tool");

  ok(registered.hooks.length === 2, `2 hooks registered (compaction only, recall/capture disabled)`);
}

// ===== Cleanup =====
try { fs.unlinkSync(DB); } catch {}
try { fs.unlinkSync(HIST); } catch {}

// ===== Summary =====
console.log(`\n${"=".repeat(50)}`);
console.log(`Docker smoke test: ${pass} passed, ${fail} failed`);
if (fail > 0) {
  console.log("\x1b[31mDOCKER SMOKE TEST FAILED\x1b[0m");
  process.exit(1);
} else {
  console.log("\x1b[32mALL DOCKER SMOKE TESTS PASSED\x1b[0m");
}
