#!/usr/bin/env bash
# Configure the MCP LNC server environment.
#
# Usage:
#   configure.sh                        # Interactive defaults (production mailbox)
#   configure.sh --production           # Production settings
#   configure.sh --dev                  # Development/regtest settings
#   configure.sh --dev --mailbox host   # Custom mailbox server
#   configure.sh --dev --insecure       # Dev mode with TLS verification disabled
#
# Writes to lightning-mcp-server/.env in the lightning-agent-kit repo.

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
MCP_SERVER_DIR="$(cd "$SCRIPT_DIR/../../.." && pwd)/lightning-mcp-server"

MAILBOX=""
DEV_MODE=""
INSECURE=""
TIMEOUT=""
MODE=""

# Parse arguments.
while [[ $# -gt 0 ]]; do
    case $1 in
        --production)
            MODE="production"
            shift
            ;;
        --dev)
            MODE="dev"
            shift
            ;;
        --mailbox)
            MAILBOX="$2"
            shift 2
            ;;
        --insecure)
            INSECURE="true"
            shift
            ;;
        --timeout)
            TIMEOUT="$2"
            shift 2
            ;;
        -h|--help)
            echo "Usage: configure.sh [options]"
            echo ""
            echo "Configure the MCP LNC server environment."
            echo ""
            echo "Modes:"
            echo "  --production      Production settings (default mailbox, TLS enabled)"
            echo "  --dev             Development settings (dev mode, insecure TLS)"
            echo ""
            echo "Options:"
            echo "  --mailbox SERVER  Mailbox server address (host:port)"
            echo "  --insecure        Skip TLS verification (dev only)"
            echo "  --timeout SECS    Connection timeout (default: 30)"
            exit 0
            ;;
        *)
            echo "Unknown option: $1" >&2
            exit 1
            ;;
    esac
done

# Verify lightning-mcp-server directory exists.
if [ ! -d "$MCP_SERVER_DIR" ]; then
    echo "Error: lightning-mcp-server/ directory not found at $MCP_SERVER_DIR" >&2
    exit 1
fi

# Apply mode defaults.
case $MODE in
    production)
        MAILBOX="${MAILBOX:-mailbox.terminal.lightning.today:443}"
        DEV_MODE="${DEV_MODE:-false}"
        INSECURE="${INSECURE:-false}"
        ;;
    dev)
        MAILBOX="${MAILBOX:-mailbox.terminal.lightning.today:443}"
        DEV_MODE="${DEV_MODE:-true}"
        INSECURE="${INSECURE:-true}"
        ;;
    *)
        MAILBOX="${MAILBOX:-mailbox.terminal.lightning.today:443}"
        DEV_MODE="${DEV_MODE:-false}"
        INSECURE="${INSECURE:-false}"
        ;;
esac

TIMEOUT="${TIMEOUT:-30}"

ENV_FILE="$MCP_SERVER_DIR/.env"

echo "=== Configuring MCP LNC Server ==="
echo ""
echo "Mailbox:   $MAILBOX"
echo "Dev mode:  $DEV_MODE"
echo "Insecure:  $INSECURE"
echo "Timeout:   ${TIMEOUT}s"
echo "Output:    $ENV_FILE"
echo ""

cat > "$ENV_FILE" <<EOF
# MCP LNC Server Configuration
# Generated by configure.sh on $(date -u +"%Y-%m-%dT%H:%M:%SZ")

# Mailbox relay server
LNC_MAILBOX_SERVER=$MAILBOX

# Enable dev mode for local/regtest environments
LNC_DEV_MODE=$DEV_MODE

# Skip TLS verification (only for development!)
LNC_INSECURE=$INSECURE

# Connection timeout in seconds
LNC_CONNECT_TIMEOUT=$TIMEOUT
EOF

chmod 600 "$ENV_FILE"

echo "Configuration written to $ENV_FILE (mode 0600)"
echo ""
echo "Next steps:"
echo "  1. Add to Claude Code: skills/lightning-mcp-server/scripts/setup-claude-config.sh"
echo "  2. Restart Claude Code to pick up the new MCP server"
