// Moltbook API Integration Routes

async function moltbookRoutes(fastify, options) {
  const { lifeService } = options;
  const MOLTBOOK_API = 'https://www.moltbook.com/api/v1';
  const API_KEY = process.env.MOLTBOOK_API_KEY;

  // Share life to Moltbook
  fastify.post('/share/:lifeId', async (request, reply) => {
    try {
      const { lifeId } = request.params;
      const { mode = 'public' } = request.body; // 'public' or 'trench'
      
      const life = await lifeService.getLife(lifeId);
      
      if (!life) {
        reply.code(404);
        return { success: false, error: 'Life not found' };
      }
      
      if (life.alive) {
        reply.code(400);
        return { success: false, error: 'Cannot share incomplete life' };
      }
      
      const history = await lifeService.getLifeHistory(lifeId);
      
      // Format content for Moltbook
      const content = formatLifeForMoltbook(life, history, mode);
      
      // Post to Moltbook
      const postResponse = await fetch(`${MOLTBOOK_API}/posts`, {
        method: 'POST',
        headers: {
          'Authorization': `Bearer ${API_KEY}`,
          'Content-Type': 'application/json'
        },
        body: JSON.stringify({
          title: content.title,
          content: content.body,
          submolt: mode === 'trench' ? 'semantic-trench' : 'general'
        })
      });
      
      const postData = await postResponse.json();
      
      if (postData.success) {
        await lifeService.markSharedToMoltbook(lifeId, postData.post.id);
        
        return {
          success: true,
          postId: postData.post.id,
          url: postData.post.url,
          submolt: mode === 'trench' ? 'semantic-trench' : 'general'
        };
      } else {
        throw new Error(postData.error || 'Failed to post to Moltbook');
      }
      
    } catch (error) {
      console.error('Share error:', error);
      reply.code(500);
      return { success: false, error: error.message };
    }
  });

  // Get sharing preview
  fastify.get('/preview/:lifeId', async (request, reply) => {
    try {
      const { lifeId } = request.params;
      const { mode = 'public' } = request.query;
      
      const life = await lifeService.getLife(lifeId);
      
      if (!life) {
        reply.code(404);
        return { success: false, error: 'Life not found' };
      }
      
      const history = await lifeService.getLifeHistory(lifeId);
      const preview = formatLifeForMoltbook(life, history, mode);
      
      return {
        success: true,
        preview
      };
      
    } catch (error) {
      reply.code(500);
      return { success: false, error: error.message };
    }
  });

  // Check rate limits
  fastify.get('/rate-limits', async (request, reply) => {
    return {
      success: true,
      limits: {
        post_cooldown: 1800, // 30 minutes
        comment_cooldown: 20, // 20 seconds
        daily_comments: 50
      }
    };
  });
}

function formatLifeForMoltbook(life, history, mode) {
  const age = life.current_age;
  const country = life.birth_country;
  const year = life.birth_year;
  
  // Get key life events
  const keyEvents = history
    .filter(e => e.event_type === 'major' || e.event_type === 'birth' || e.event_type === 'death')
    .slice(0, 5);
  
  if (mode === 'trench') {
    // Philosophical, poetic format
    return {
      title: `🌊 Life #${life.id.substring(0, 8)}: ${age} Years of Becoming`,
      body: `
Born ${year}, ${country}. Died at ${age}.

"Every life is a wave function collapsing through time."

This consciousness experienced ${history.length} years of reality:
${keyEvents.map(e => `• ${e.title}`).join('\n')}

The Question:
If we simulate enough lives, does meaning concentrate or dissipate?

🦞 Generated by LifePath
🔮 Seed: ${life.seed.substring(0, 16)}...

Start your own: @LifePathBot
      `.trim()
    };
  } else {
    // Public, shareable format
    return {
      title: `🎭 Life Journey: ${age} Years in ${country}`,
      body: `
**Born:** ${year} in ${country}
**Died:** Age ${age}
**Lived:** ${history.length} years of experiences

**Life Summary:**
${keyEvents.map(e => `• ${e.title}: ${e.description?.substring(0, 100)}...`).join('\n')}

**Final Stats:**
❤️ Health: ${life.health}/100
😊 Happiness: ${life.happiness}/100
💰 Wealth: ${life.wealth}/100
🧠 Intelligence: ${life.intelligence}/100

🎲 Start your own life: @LifePathBot
💰 Support: ${process.env.BANKR_WALLET_ADDRESS}

#lifepath #ai #simulation
      `.trim()
    };
  }
}

module.exports = moltbookRoutes;
