---
name: lemlist
description: |
  Lemlist API integration with managed OAuth. Sales automation and cold outreach platform.
  Use this skill when users want to manage campaigns, leads, activities, schedules, or unsubscribes in Lemlist.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
compatibility: Requires network access and valid Maton API key
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    homepage: "https://maton.ai"
    requires:
      env:
        - MATON_API_KEY
---

# Lemlist

Access the Lemlist API with managed OAuth authentication. Manage campaigns, leads, activities, schedules, sequences, and unsubscribes for sales automation and cold outreach.

## Quick Start

```bash
# List campaigns
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/lemlist/api/campaigns')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/lemlist/api/{resource}
```

Replace `{resource}` with the actual Lemlist API endpoint path. The gateway proxies requests to `api.lemlist.com/api` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Lemlist OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=lemlist&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'lemlist'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "3ecf268f-42ad-40cc-b77a-25e020fbf591",
    "status": "ACTIVE",
    "creation_time": "2026-02-12T02:00:53.023887Z",
    "last_updated_time": "2026-02-12T02:01:45.284131Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "lemlist",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Lemlist connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/lemlist/api/campaigns')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', '3ecf268f-42ad-40cc-b77a-25e020fbf591')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### Team

#### Get Team

```bash
GET /lemlist/api/team
```

Returns team information including user IDs and settings.

#### Get Team Credits

```bash
GET /lemlist/api/team/credits
```

Returns remaining credits balance.

#### Get Team Senders

```bash
GET /lemlist/api/team/senders
```

Returns all team members and their associated campaigns.

### Campaigns

#### List Campaigns

```bash
GET /lemlist/api/campaigns
```

#### Create Campaign

```bash
POST /lemlist/api/campaigns
Content-Type: application/json

{
  "name": "My Campaign"
}
```

Creates a new campaign with an empty sequence and default schedule automatically added.

#### Get Campaign

```bash
GET /lemlist/api/campaigns/{campaignId}
```

#### Update Campaign

```bash
PATCH /lemlist/api/campaigns/{campaignId}
Content-Type: application/json

{
  "name": "Updated Campaign Name"
}
```

#### Pause Campaign

```bash
POST /lemlist/api/campaigns/{campaignId}/pause
```

Pauses a running campaign.

### Campaign Sequences

#### Get Campaign Sequences

```bash
GET /lemlist/api/campaigns/{campaignId}/sequences
```

Returns all sequences and steps for a campaign.

### Campaign Schedules

#### Get Campaign Schedules

```bash
GET /lemlist/api/campaigns/{campaignId}/schedules
```

Returns all schedules associated with a campaign.

### Leads

#### Add Lead to Campaign

```bash
POST /lemlist/api/campaigns/{campaignId}/leads
Content-Type: application/json

{
  "email": "lead@example.com",
  "firstName": "John",
  "lastName": "Doe",
  "companyName": "Acme Inc"
}
```

Creates a new lead and adds it to the campaign. If the lead already exists, it will be inserted into the campaign.

#### Get Lead by Email

```bash
GET /lemlist/api/leads/{email}
```

#### Update Lead in Campaign

```bash
PATCH /lemlist/api/campaigns/{campaignId}/leads/{email}
Content-Type: application/json

{
  "firstName": "Jane",
  "lastName": "Smith"
}
```

#### Delete Lead from Campaign

```bash
DELETE /lemlist/api/campaigns/{campaignId}/leads/{email}
```

### Activities

#### List Activities

```bash
GET /lemlist/api/activities
```

Returns the history of campaign activities (last 100 activities).

Query parameters:
- `campaignId` - Filter by campaign
- `type` - Filter by activity type (emailsSent, emailsOpened, emailsClicked, etc.)

### Schedules

#### List Schedules

```bash
GET /lemlist/api/schedules
```

Returns all schedules with pagination.

Response:
```json
{
  "schedules": [...],
  "pagination": {
    "totalRecords": 10,
    "currentPage": 1,
    "nextPage": 2,
    "totalPage": 2
  }
}
```

#### Create Schedule

```bash
POST /lemlist/api/schedules
Content-Type: application/json

{
  "name": "Business Hours",
  "timezone": "America/New_York",
  "start": "09:00",
  "end": "17:00",
  "weekdays": [1, 2, 3, 4, 5]
}
```

Weekdays: 0 = Sunday, 1 = Monday, ..., 6 = Saturday

#### Get Schedule

```bash
GET /lemlist/api/schedules/{scheduleId}
```

#### Update Schedule

```bash
PATCH /lemlist/api/schedules/{scheduleId}
Content-Type: application/json

{
  "name": "Updated Schedule",
  "start": "08:00",
  "end": "18:00"
}
```

#### Delete Schedule

```bash
DELETE /lemlist/api/schedules/{scheduleId}
```

### Companies

#### List Companies

```bash
GET /lemlist/api/companies
```

Returns companies with pagination.

Response:
```json
{
  "data": [...],
  "total": 100
}
```

### Unsubscribes

#### List Unsubscribes

```bash
GET /lemlist/api/unsubscribes
```

Returns all unsubscribed emails and domains.

#### Add Unsubscribe

```bash
POST /lemlist/api/unsubscribes
Content-Type: application/json

{
  "email": "unsubscribe@example.com"
}
```

Can also add domains by using a domain value.

### Inbox Labels

#### List Labels

```bash
GET /lemlist/api/inbox/labels
```

Returns all labels available to the team.

## Pagination

Lemlist uses page-based pagination with different formats depending on the endpoint:

**Schedules format:**
```json
{
  "schedules": [...],
  "pagination": {
    "totalRecords": 100,
    "currentPage": 1,
    "nextPage": 2,
    "totalPage": 10
  }
}
```

**Companies format:**
```json
{
  "data": [...],
  "total": 100
}
```

## Code Examples

### JavaScript - List Campaigns

```javascript
const response = await fetch(
  'https://gateway.maton.ai/lemlist/api/campaigns',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const campaigns = await response.json();
console.log(campaigns);
```

### Python - List Campaigns

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/lemlist/api/campaigns',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
campaigns = response.json()
for campaign in campaigns:
    print(f"{campaign['name']}: {campaign['_id']}")
```

### Python - Create Campaign and Add Lead

```python
import os
import requests

headers = {'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
base_url = 'https://gateway.maton.ai/lemlist/api'

# Create campaign
campaign_response = requests.post(
    f'{base_url}/campaigns',
    headers=headers,
    json={'name': 'Q1 Outreach'}
)
campaign = campaign_response.json()
print(f"Created campaign: {campaign['_id']}")

# Add lead to campaign
lead_response = requests.post(
    f'{base_url}/campaigns/{campaign["_id"]}/leads',
    headers=headers,
    json={
        'email': 'prospect@example.com',
        'firstName': 'John',
        'lastName': 'Doe',
        'companyName': 'Acme Corp'
    }
)
lead = lead_response.json()
print(f"Added lead: {lead['_id']}")
```

## Notes

- Campaign IDs start with `cam_`
- Lead IDs start with `lea_`
- Schedule IDs start with `skd_`
- Sequence IDs start with `seq_`
- Team IDs start with `tea_`
- User IDs start with `usr_`
- Campaigns cannot be deleted via API (only paused)
- When creating a campaign, an empty sequence and default schedule are automatically added
- Lead emails are used as identifiers for lead operations
- IMPORTANT: When using curl commands, use `curl -g` when URLs contain brackets to disable glob parsing
- IMPORTANT: When piping curl output to `jq`, environment variables may not expand correctly. Use Python examples instead.

## Rate Limits

| Operation | Limit |
|-----------|-------|
| API calls | 20 per 2 seconds per API key |

When rate limited, implement exponential backoff for retries.

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Bad request or missing Lemlist connection |
| 401 | Invalid or missing Maton API key |
| 404 | Resource not found |
| 405 | Method not allowed |
| 422 | Validation error |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Lemlist API |

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `lemlist`. For example:

- Correct: `https://gateway.maton.ai/lemlist/api/campaigns`
- Incorrect: `https://gateway.maton.ai/api/campaigns`

## Resources

- [Lemlist API Documentation](https://developer.lemlist.com/)
- [Lemlist API Reference](https://developer.lemlist.com/api-reference)
- [Lemlist Help Center - API](https://help.lemlist.com/en/collections/17109856-api-webhooks)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
