# Email Discovery & Verification

> The most valuable piece of enrichment. A verified email = direct line to your lead.

---

## Email Discovery Methods

### Method 1: Pattern Matching (Free)

Most companies use predictable email patterns:

```yaml
# email-patterns.yaml
common_patterns:
  - "{first}@{domain}"                    # john@company.com
  - "{first}.{last}@{domain}"             # john.smith@company.com
  - "{first}{last}@{domain}"              # johnsmith@company.com
  - "{f}{last}@{domain}"                  # jsmith@company.com
  - "{first}_{last}@{domain}"             # john_smith@company.com
  - "{first}-{last}@{domain}"             # john-smith@company.com
  - "{last}@{domain}"                     # smith@company.com
  - "{first}{l}@{domain}"                 # johns@company.com

# Pattern frequency by company size
pattern_likelihood:
  startups_small:                         # <20 employees
    - "{first}@{domain}": 0.45
    - "{first}.{last}@{domain}": 0.30
    - "{first}{last}@{domain}": 0.15
    
  startups_medium:                        # 20-100 employees
    - "{first}.{last}@{domain}": 0.50
    - "{first}@{domain}": 0.25
    - "{f}{last}@{domain}": 0.15
    
  enterprise:                             # 100+ employees
    - "{first}.{last}@{domain}": 0.60
    - "{f}{last}@{domain}": 0.25
```

### Implementation

```yaml
# pattern-guesser.yaml
input:
  first_name: "John"
  last_name: "Smith"
  company_domain: "acme.com"
  company_size: "small"

process:
  1. Generate all pattern permutations
  2. Order by likelihood for company size
  3. Verify each (see verification below)
  4. Return first verified match

output:
  email: "john@acme.com"
  confidence: 0.95
  pattern_used: "{first}@{domain}"
```

---

### Method 2: Public Sources (Free)

Scrape emails from public sources:

```yaml
# public-email-sources.yaml
sources:
  github:
    - profile_email              # Often public
    - commit_emails              # In git history
    
  twitter:
    - bio_email                  # Sometimes listed
    - linked_website             # Check contact page
    
  personal_website:
    - contact_page
    - about_page
    - footer
    
  company_website:
    - team_page
    - about_page
    - press_releases
    - job_postings               # Often have contact email
    
  google_dorks:
    - '"{first} {last}" email @{domain}'
    - 'site:{domain} "{first} {last}" contact'
    
  wayback_machine:
    - historical_team_pages      # Emails from old pages
```

---

### Method 3: Hunter.io (Paid - Best ROI)

```yaml
# hunter-config.yaml
provider: hunter.io
credentials:
  api_key: ${HUNTER_API_KEY}
  
endpoints:
  # Find email by name + domain
  email_finder:
    input:
      first_name: "John"
      last_name: "Smith"
      domain: "acme.com"
    output:
      email: "john.smith@acme.com"
      confidence: 94
      sources: 3
      
  # Find all emails at domain
  domain_search:
    input:
      domain: "acme.com"
    output:
      pattern: "{first}.{last}"
      emails:
        - email: "jane.doe@acme.com"
          position: "CEO"
        - email: "john.smith@acme.com"
          position: "CTO"
          
  # Verify existing email
  email_verifier:
    input:
      email: "john@acme.com"
    output:
      status: "valid"
      score: 95

pricing:
  free_tier: 25/month
  starter: $49/mo for 500 requests
  growth: $149/mo for 5000 requests
  
rate_limits:
  requests_per_second: 10
```

---

### Method 4: Apollo.io (Paid - Best Database)

```yaml
# apollo-config.yaml
provider: apollo.io
credentials:
  api_key: ${APOLLO_API_KEY}
  
features:
  - 270M+ contact database
  - Email + phone
  - Company data included
  - LinkedIn integration
  
endpoints:
  people_search:
    filters:
      titles: ["CEO", "Founder"]
      company_size: "1-50"
      industries: ["SaaS"]
    output:
      - name
      - email
      - phone
      - linkedin_url
      - company
      
  enrich_person:
    input:
      linkedin_url: "linkedin.com/in/johnsmith"
    output:
      - email
      - phone
      - company_details
      
pricing:
  free_tier: 50 credits/month
  basic: $49/mo for 200 credits
  professional: $99/mo for 400 credits
```

---

### Method 5: Clearbit (Paid - Enterprise)

```yaml
# clearbit-config.yaml
provider: clearbit
credentials:
  api_key: ${CLEARBIT_API_KEY}
  
endpoints:
  prospector:
    input:
      domain: "acme.com"
      role: "ceo"
    output:
      - full profile with email
      
  enrichment:
    input:
      email: "john@acme.com"
    output:
      - 100+ data points
      - social profiles
      - company data
      
pricing:
  starts_at: $99/mo
  enterprise: custom
```

---

## Email Verification

**ALWAYS verify before sending.** Bad emails = destroyed sender reputation.

### Verification Levels

```yaml
# verification-levels.yaml
levels:
  syntax_check:
    validates: "Format is valid email"
    confidence: 0.20
    cost: free
    
  domain_check:
    validates: "Domain exists, has MX records"
    confidence: 0.40
    cost: free
    
  smtp_check:
    validates: "Mailbox exists (SMTP ping)"
    confidence: 0.80
    cost: free (but risky)
    
  api_verification:
    validates: "Third-party verification"
    confidence: 0.95
    cost: $0.001-0.01/email
```

### Free Verification Stack

```yaml
# free-verification.yaml
steps:
  1_syntax:
    regex: '^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,}$'
    fail_action: reject
    
  2_domain:
    check: MX_record_exists
    fail_action: reject
    
  3_disposable:
    check: not_in_disposable_list
    list_source: "github.com/disposable-email-domains"
    fail_action: reject
    
  4_role_based:
    check: not_role_email
    role_emails: [info@, support@, admin@, sales@, contact@]
    fail_action: flag_as_generic
    
  5_smtp_ping:
    check: mailbox_exists
    timeout: 10s
    fail_action: mark_unverified
    warning: "Some servers block SMTP pings"
```

### Paid Verification (Recommended)

```yaml
# paid-verification.yaml
providers:
  zerobounce:
    api_key: ${ZEROBOUNCE_KEY}
    cost: $0.008/email
    accuracy: 98%
    
  neverbounce:
    api_key: ${NEVERBOUNCE_KEY}
    cost: $0.008/email
    accuracy: 97%
    
  debounce:
    api_key: ${DEBOUNCE_KEY}
    cost: $0.005/email
    accuracy: 97%
    
  hunter_verifier:
    api_key: ${HUNTER_KEY}
    cost: included_in_plan
    accuracy: 95%

recommendation:
  budget_conscious: "debounce"
  best_accuracy: "zerobounce"
  already_using_hunter: "hunter_verifier"
```

---

## Verification Pipeline

```yaml
# verification-pipeline.yaml
input:
  email: "john@acme.com"
  
pipeline:
  step_1:
    action: syntax_check
    on_fail: reject
    
  step_2:
    action: domain_check
    on_fail: reject
    
  step_3:
    action: disposable_check
    on_fail: reject
    
  step_4:
    action: role_based_check
    on_fail: flag
    
  step_5:
    action: api_verification
    provider: zerobounce
    on_fail: mark_risky
    
output:
  email: "john@acme.com"
  status: "valid"           # valid | invalid | risky | unknown
  confidence: 0.95
  deliverable: true
  catch_all: false
  disposable: false
  role_based: false
  free_provider: false
```

---

## Handling Results

```yaml
# result-handling.yaml
valid:
  confidence: >= 0.90
  action: ready_for_outreach
  
risky:
  confidence: 0.60-0.89
  action: send_with_caution
  limit: 10/day
  
invalid:
  confidence: < 0.60
  action: do_not_send
  try: find_alternative_email
  
catch_all:
  description: "Domain accepts all emails"
  action: send_but_monitor_bounce
  limit: 5/day
  
role_based:
  examples: [info@, sales@, support@]
  action: deprioritize
  try: find_personal_email
```

---

## Email Discovery Workflow

```
┌─────────────────────────────────────────────────────────────┐
│                    LEAD INPUT                               │
│  Name: John Smith                                           │
│  Company: Acme Inc                                          │
│  Domain: acme.com                                           │
└─────────────────────────┬───────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│              1. CHECK PUBLIC SOURCES                        │
│  GitHub profile? Twitter bio? Company website?              │
│  → Found: No                                                │
└─────────────────────────┬───────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│              2. PATTERN GUESSING                            │
│  Generate: john@acme.com, john.smith@acme.com, etc.         │
│  → Generated: 8 candidates                                  │
└─────────────────────────┬───────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│              3. VERIFICATION (per candidate)                │
│  Syntax → Domain → Disposable → SMTP/API                    │
│  → Verified: john.smith@acme.com (95% confidence)           │
└─────────────────────────┬───────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│              4. OUTPUT                                      │
│  email: john.smith@acme.com                                 │
│  confidence: 0.95                                           │
│  method: pattern_match + api_verify                         │
│  ready_for_outreach: true                                   │
└─────────────────────────────────────────────────────────────┘
```

---

## Cost Optimization

```yaml
# cost-optimization.yaml
strategy:
  1. Always try free methods first
     - Public source scraping
     - Pattern matching with free verification
     
  2. Use paid services for high-value leads only
     - ICP score > 70
     - Company funding > $1M
     
  3. Batch verification
     - Collect leads first
     - Verify in bulk (cheaper)
     
  4. Cache results
     - Don't re-verify same email
     - Domain patterns are reusable

estimated_cost_per_lead:
  free_methods_only: $0
  hybrid_approach: $0.01-0.03
  full_paid_stack: $0.05-0.10
```

---

→ Continue to `/enrichment/company-data.md`
