---
name: lead-enrichment
description: Turn a name into a full dossier in seconds. Feed in a name + company (or email, or LinkedIn URL) and get back a rich profile with social links, bio, company intel, recent activity, and personalized talking points. Aggregates data from multiple public sources — LinkedIn, Twitter, GitHub, company websites, news — so you can skip the manual research and jump straight to personalized outreach. Your agent does the detective work while you close deals. Supports single enrichment, batch processing, and multiple output formats (JSON, Markdown, CRM-ready). Use when researching prospects, preparing for sales calls, personalizing cold outreach, or building lead lists. Pairs perfectly with trawl for autonomous lead gen → enrichment → outreach pipelines.
metadata:
  clawdbot:
    emoji: "🔍"
    requires:
      skills:
        - browser
---

# Lead Enrichment — Research Prospects in Seconds

**Stop spending hours stalking LinkedIn. Let your agent do it.**

Sales teams waste 6+ hours per week manually researching prospects. You Google their name, check LinkedIn, scroll their Twitter, hunt for their email, read their company's About page, search for recent news... and then do it all over again for the next lead.

Lead Enrichment automates all of it. Give your agent a name and company (or email, or LinkedIn URL), and get back a complete dossier: contact info, social profiles, bio, company intel, recent posts, news mentions, and AI-generated talking points.

**The pain:** Generic outreach gets ignored. Personalization takes forever. You're always behind quota.

**The fix:** Your agent researches 10 leads while you grab coffee. Rich profiles ready when you need them. Spend your time selling, not searching.

## What You Get

For each lead, the enrichment pulls:

**Personal Profile:**
- Full name, current title, company
- Professional bio/summary
- Profile photo URL
- Location
- Social media handles (LinkedIn, Twitter, GitHub, personal site)

**Contact Discovery:**
- Likely email addresses (pattern-based + verification attempts)
- Public phone numbers (if available)
- Best channels for outreach

**Company Context:**
- Company description, industry, size
- Funding stage, recent news
- Tech stack (for technical sales)
- Key decision makers

**Intelligence & Timing:**
- Recent posts/articles (last 30 days)
- Job change signals
- Company news mentions
- Shared connections or interests
- Conference/event participation

**AI-Generated Talking Points:**
- 3-5 personalized hooks based on their recent activity
- Common ground opportunities
- Relevant pain points to address
- Recommended opening lines

## Setup

1. Run `scripts/setup.sh` to initialize config
2. Edit `~/.config/lead-enrichment/config.json` with preferences
3. No API keys required for basic enrichment (uses public sources)
4. Optional: Add premium data sources (see config)

## Scripts

| Script | Purpose |
|--------|---------|
| `scripts/setup.sh` | Initialize config and data directories |
| `scripts/enrich.sh` | Enrich a single lead (main script) |
| `scripts/batch.sh` | Process multiple leads from CSV/JSON |
| `scripts/export.sh` | Export enriched leads (JSON/MD/CSV) |

## Usage

### Single Lead

```bash
# By name + company
./scripts/enrich.sh --name "Sarah Chen" --company "Acme Corp"

# By email
./scripts/enrich.sh --email "sarah@acmecorp.com"

# By LinkedIn URL
./scripts/enrich.sh --linkedin "https://linkedin.com/in/sarahchen"

# Output to file
./scripts/enrich.sh --name "Sarah Chen" --company "Acme Corp" --output sarah-chen.json

# With talking points
./scripts/enrich.sh --name "Sarah Chen" --company "Acme Corp" --talking-points
```

### Batch Processing

```bash
# From CSV (columns: name, company, email, linkedin_url)
./scripts/batch.sh --input leads.csv --output enriched/

# From JSON array
./scripts/batch.sh --input leads.json --output enriched/

# Process with concurrency
./scripts/batch.sh --input leads.csv --parallel 3
```

### Export Formats

```bash
# Export as JSON (default)
./scripts/export.sh --format json enriched/*.json > leads.json

# Export as Markdown (readable)
./scripts/export.sh --format markdown enriched/*.json > leads.md

# Export as CSV (CRM import)
./scripts/export.sh --format csv enriched/*.json > leads.csv

# Pipe to your CRM
./scripts/export.sh --format json enriched/*.json | \
  curl -X POST https://your-crm.com/api/leads -d @-
```

## Config

Config lives at `~/.config/lead-enrichment/config.json`. See `config.example.json` for full schema.

Key sections:

**enrichment.sources** — Which data sources to check (all public by default):
- `linkedin` — Public profiles via search
- `twitter` — Social activity and bio
- `github` — For technical leads
- `company_website` — About pages, team directories
- `news` — Recent mentions
- `crunchbase` — Company funding (public data)

**enrichment.depth** — How thorough to be:
- `quick` — Basic profile only (name, title, LinkedIn, company)
- `standard` — Above + social profiles + recent activity (default)
- `deep` — Above + news mentions + talking points + shared connections

**output.format** — Default output format (json/markdown/csv)

**output.include** — What to include in output:
- `contact_info` — Email attempts, phone
- `social_profiles` — All discovered links
- `recent_activity` — Posts, articles (last 30 days)
- `company_intel` — Company description, size, funding
- `talking_points` — AI-generated personalization hooks
- `raw_sources` — Source URLs for verification

**talking_points.enabled** — Generate AI talking points (requires Claude)

**talking_points.style** — Tone for suggestions (professional/friendly/bold)

**privacy.respect_robots** — Skip profiles with clear "no scraping" signals

**privacy.store_locally** — Cache enriched profiles (default: true)

## Data Sources

All sources are **public and free**:

1. **LinkedIn** — Public profiles via search (no API, respects robots.txt)
2. **Twitter/X** — Bio, recent tweets, follower count
3. **GitHub** — For technical roles (repos, activity, README)
4. **Company websites** — Team pages, About sections
5. **Google News** — Recent mentions
6. **Crunchbase** — Public company data (no API key needed for basic info)
7. **Common email patterns** — firstname@company.com, f.lastname@company.com, etc.

**Premium sources** (optional, requires API keys):
- Hunter.io — Email verification
- Clearbit — Enhanced company data
- Apollo — Direct contact info

Add API keys to `~/.clawdbot/secrets.env` if you have them. Enrichment works fine without them.

## Output Schema

Each enriched lead is saved as JSON:

```json
{
  "lead_id": "sarah-chen-acme-corp",
  "enriched_at": "2025-01-29T10:30:00Z",
  "input": {
    "name": "Sarah Chen",
    "company": "Acme Corp"
  },
  "profile": {
    "full_name": "Sarah Chen",
    "title": "VP of Engineering",
    "company": "Acme Corp",
    "location": "San Francisco, CA",
    "bio": "Building the future of...",
    "photo_url": "https://...",
    "social_profiles": {
      "linkedin": "https://linkedin.com/in/sarahchen",
      "twitter": "https://twitter.com/sarahchen",
      "github": "https://github.com/sarahchen",
      "personal_site": "https://sarahchen.com"
    }
  },
  "contact": {
    "emails": [
      { "address": "sarah@acmecorp.com", "confidence": 0.85, "verified": false },
      { "address": "s.chen@acmecorp.com", "confidence": 0.60, "verified": false }
    ],
    "phones": [],
    "preferred_channel": "email"
  },
  "company": {
    "name": "Acme Corp",
    "domain": "acmecorp.com",
    "industry": "SaaS",
    "size": "51-200 employees",
    "description": "AI-powered...",
    "funding": "Series B ($25M)",
    "tech_stack": ["React", "Node.js", "AWS"],
    "recent_news": [
      {
        "title": "Acme Corp raises $25M...",
        "url": "https://...",
        "date": "2025-01-15"
      }
    ]
  },
  "intelligence": {
    "recent_activity": [
      {
        "type": "twitter_post",
        "content": "Excited to announce...",
        "url": "https://...",
        "date": "2025-01-20"
      }
    ],
    "job_change_signal": false,
    "shared_connections": [],
    "interests": ["AI", "startups", "engineering leadership"]
  },
  "talking_points": [
    "Reference their recent Series B — congrats and ask about growth plans",
    "Mention mutual interest in AI/ML engineering",
    "Their tech stack (React/Node) aligns with your solution"
  ],
  "sources": [
    "https://linkedin.com/in/sarahchen",
    "https://twitter.com/sarahchen",
    "https://acmecorp.com/about"
  ],
  "confidence_score": 0.88
}
```

## Integration with Trawl

Lead Enrichment pairs perfectly with Trawl (autonomous lead gen):

```bash
# Trawl finds leads, enrichment researches them
trawl sweep.sh                    # Discover leads
trawl leads.sh list --json |      # Export qualified leads
  jq -r '.[] | "\(.name)|\(.company)"' |
  while IFS='|' read name company; do
    ./enrich.sh --name "$name" --company "$company"
  done

# Or automate it via config:
# trawl config: "post_qualify_action": "enrich"
```

## Tips

**Email Discovery:**
- Works best when you provide company domain
- Tries common patterns (first@company, f.last@company, etc.)
- Marks confidence level (high/medium/low)
- Does NOT spam or verify via email sends (respects privacy)

**Talking Points:**
- Most valuable when enrichment depth = "deep"
- Requires recent activity data (posts, news)
- AI analyzes content for personalization hooks
- Style can be professional, friendly, or bold

**Batch Processing:**
- Use `--parallel` for speed (3-5 concurrent recommended)
- Progress saved (resume if interrupted)
- Failed leads logged to `batch-errors.json`

**Data Freshness:**
- Cached profiles expire after 30 days
- Force refresh with `--refresh` flag
- Social activity always fetched fresh

## Use Cases

**Sales Reps:**
- Research prospects before calls
- Personalize cold email sequences
- Find mutual connections or interests

**Recruiters:**
- Assess candidate backgrounds
- Find contact info for passive candidates
- Check GitHub activity for technical roles

**Partnerships:**
- Research potential partners
- Understand company context
- Find the right contact person

**Investors:**
- Quick founder background checks
- Company traction signals
- Network mapping

## Privacy & Ethics

This skill only uses **publicly available data**. It:
- Respects robots.txt and rate limits
- Does NOT scrape private profiles or paywalled content
- Does NOT send verification emails (won't spam your leads)
- Does NOT store data if privacy.store_locally = false
- Provides source URLs for transparency

**Be a human:** Just because you CAN enrich someone doesn't mean you should spam them. Use this for genuine, personalized outreach.

## Data Storage

Enriched leads are stored at `~/.config/lead-enrichment/data/leads/`:

```
~/.config/lead-enrichment/
├── config.json                 # User configuration
├── data/
│   ├── leads/                  # Enriched profiles (one file per lead)
│   │   ├── sarah-chen-acme.json
│   │   └── john-smith-techco.json
│   ├── cache/                  # Temporary data (30-day expiry)
│   └── batch-runs/             # Batch processing logs
└── exports/                    # Generated exports
```

## FAQ

**Q: Is this legal?**
A: Yes. All data is publicly available. We respect robots.txt and rate limits.

**Q: How accurate are the emails?**
A: Pattern-based = 60-80% accuracy. Verified (if you add Hunter.io key) = 95%+.

**Q: Can I enrich 1000 leads?**
A: Yes via batch.sh. Expect ~30 sec per lead (deep mode). That's 8 hours for 1000. Run overnight.

**Q: Does this work for non-US leads?**
A: Yes. LinkedIn and Twitter are global. Some data sources are US-biased.

**Q: Will this get me blocked by LinkedIn?**
A: No. We use search (public), not scraping. Rate-limited and respectful.

## What's Next

Ideas for future versions:
- Chrome extension (enrich while browsing LinkedIn)
- CRM integrations (auto-enrich on lead create)
- Slack bot (enrich on-demand from Slack)
- Email warmup integration (find + verify + warm sequence)
- Mutual connection finder (via agent networks)
- Real-time alerts (when a lead changes jobs)

---

**Stop researching. Start selling.**

Feed your agent a list of names. Get back a stack of dossiers. Personalize every message. Close more deals.

That's Lead Enrichment.
