---
name: zod
description: Complete Zod validation library documentation. Use when working with Zod schema validation, TypeScript type inference, form validation, API validation, error handling, or data parsing. Covers schema definition, refinements, transforms, error customization, and ecosystem integrations.
---

# Zod Documentation

Complete Zod validation library documentation embedded in markdown. Read from `references/` to answer questions about schema validation, type inference, and error handling.

## Documentation Structure

All documentation is in `references/` organized by topic:

### Core Documentation

#### Getting Started
- `index.mdx` - Introduction and quick start
- `basics.mdx` - Basic usage and schema definition

#### API Reference
- `api.mdx` - Complete API documentation
  - Primitives (string, number, boolean, etc.)
  - Complex types (object, array, tuple, union, etc.)
  - Schema methods (parse, safeParse, parseAsync, etc.)
  - Refinements and transforms
  - Type inference
  - Error handling

#### Advanced Features
- `error-formatting.mdx` - Error formatting and customization
- `error-customization.mdx` - Custom error messages
- `codecs.mdx` - Serialization and deserialization
- `json-schema.mdx` - JSON Schema generation
- `metadata.mdx` - Schema metadata

#### Integration & Ecosystem
- `ecosystem.mdx` - Community packages and integrations
- `library-authors.mdx` - Guide for library authors
- `packages/` - Related packages

#### Version 4
- `v4/` - Zod v4 features and migration guide

## Quick Reference

### Common Tasks

| Task | File to Read |
|------|--------------|
| Get started | `index.mdx`, `basics.mdx` |
| Define schemas | `api.mdx` (Primitives section) |
| Object validation | `api.mdx` (Objects section) |
| Array validation | `api.mdx` (Arrays section) |
| Union types | `api.mdx` (Unions section) |
| Refinements | `api.mdx` (Refinements section) |
| Transforms | `api.mdx` (Transforms section) |
| Error handling | `error-formatting.mdx`, `error-customization.mdx` |
| Type inference | `api.mdx` (Type Inference section) |
| Async validation | `api.mdx` (Async section) |
| JSON Schema | `json-schema.mdx` |
| Custom errors | `error-customization.mdx` |
| Ecosystem | `ecosystem.mdx` |

### Schema Examples

**Primitives:**
```typescript
z.string()
z.number()
z.boolean()
z.date()
z.undefined()
z.null()
z.any()
z.unknown()
```

**Complex Types:**
```typescript
z.object({ ... })
z.array(z.string())
z.tuple([z.string(), z.number()])
z.union([z.string(), z.number()])
z.record(z.string())
z.map(z.string(), z.number())
z.set(z.string())
```

**Refinements:**
```typescript
z.string().email()
z.string().url()
z.string().uuid()
z.number().min(5).max(10)
z.string().regex(/pattern/)
```

**Transforms:**
```typescript
z.string().transform(val => val.toUpperCase())
z.coerce.number()
```

### When to Use This Skill

- Form validation in React/Next.js
- API request/response validation
- Environment variable parsing
- Runtime type checking
- Data transformation and parsing
- Error message customization
- Integration with tRPC, React Hook Form, etc.
- TypeScript type inference from schemas

### How to Navigate

1. **Start with `index.mdx`** for introduction
2. **For basic usage:** Read `basics.mdx`
3. **For API details:** Check `api.mdx` (comprehensive reference)
4. **For error handling:** See `error-formatting.mdx` and `error-customization.mdx`
5. **For advanced features:** Browse `codecs.mdx`, `json-schema.mdx`, `metadata.mdx`
6. **For integrations:** Check `ecosystem.mdx`
7. **For v4 features:** See `v4/` directory

All files are `.mdx` (Markdown + JSX) but readable as plain markdown.
