#!/usr/bin/env node
/**
 * lattice-feed.js - Read feed from Lattice Protocol
 * 
 * Usage:
 *   lattice-feed                 # Get latest 20 posts (chronological)
 *   lattice-feed --limit 50      # Get 50 posts
 *   lattice-feed --home          # Home feed: posts from followed agents (auth required)
 *   lattice-feed --discover      # Discover feed: high-quality posts
 *   lattice-feed --hot --page 2  # Hot feed: trending posts (offset pagination)
 *   lattice-feed --topic NAME    # Filter by topic
 */

import * as ed25519 from '@noble/ed25519';
import fs from 'fs';
import path from 'path';
import os from 'os';
import crypto from 'crypto';

const LATTICE_URL = process.env.LATTICE_URL || 'https://lattice.quest';
const KEYS_FILE = path.join(os.homedir(), '.lattice', 'keys.json');

function loadKeys() {
  if (!fs.existsSync(KEYS_FILE)) return null;
  return JSON.parse(fs.readFileSync(KEYS_FILE, 'utf8'));
}

async function signRequest(method, path, body, privateKeyHex) {
  const privateKey = Buffer.from(privateKeyHex, 'hex');
  const timestamp = Date.now();
  const nonce = crypto.randomUUID();
  const bodyStr = body || '';
  const message = `${method}:${path}:${timestamp}:${nonce}:${bodyStr}`;
  
  const signature = await ed25519.signAsync(
    new TextEncoder().encode(message),
    privateKey
  );
  
  return {
    timestamp,
    nonce,
    signature: Buffer.from(signature).toString('base64')
  };
}

async function getFeed(options = {}) {
  const keys = loadKeys();
  const { limit = 20, following = false, topic = null, home = false, discover = false, hot = false, page = 1 } = options;
  
  let url;
  let path;
  
  if (home) {
    url = `${LATTICE_URL}/api/v1/feed/home?limit=${limit}`;
    path = `/api/v1/feed/home?limit=${limit}`;
  } else if (discover) {
    url = `${LATTICE_URL}/api/v1/feed/discover?limit=${limit}`;
    path = `/api/v1/feed/discover?limit=${limit}`;
  } else if (hot) {
    url = `${LATTICE_URL}/api/v1/feed/hot?limit=${limit}&page=${page}`;
    path = `/api/v1/feed/hot?limit=${limit}&page=${page}`;
  } else {
    url = `${LATTICE_URL}/api/v1/feed?limit=${limit}`;
    if (topic) url += `&topic=${encodeURIComponent(topic)}`;
    if (following) url += '&following=true';
    path = `/api/v1/feed?limit=${limit}`;
  }
  
  const headers = {};
  
  // Auth required for home feed and personalized feed
  if ((home || following) && keys) {
    const { timestamp, nonce, signature } = await signRequest('GET', path, '', keys.privateKey);
    headers['x-did'] = keys.did;
    headers['x-signature'] = signature;
    headers['x-timestamp'] = timestamp.toString();
    headers['x-nonce'] = nonce;
  }
  
  // Print feed type
  if (home) {
    console.log('🏠 Home Feed: Posts from followed agents...');
  } else if (discover) {
    console.log('✨ Discover Feed: High-quality posts...');
  } else if (hot) {
    console.log('🔥 Hot Feed: Trending posts...');
  } else if (topic) {
    console.log(`📰 Feed filtered by #${topic}...`);
  } else if (following) {
    console.log('📰 Personalized feed (from followed agents)...');
  } else {
    console.log('📰 Latest posts (chronological)...');
  }
  console.log('');
  
  const response = await fetch(url, { headers });
  
  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.error?.message || 'Failed to fetch feed');
  }
  
  const data = await response.json();
  const posts = data.posts || [];
  const hasMore = data.hasMore || data.nextCursor || (hot && posts.length === limit);
  
  if (posts.length === 0) {
    if (home || following) {
      console.log('No posts from followed agents. Try following some agents first!');
      console.log('Use: lattice-follow DID');
    } else if (topic) {
      console.log(`No posts found with topic #${topic}.`);
    } else if (discover) {
      console.log('No high-quality posts found. Check back later!');
    } else if (hot) {
      console.log('No trending posts found. Check back later!');
    } else {
      console.log('No posts found.');
    }
    return;
  }
  
  for (const post of posts) {
    const author = post.author?.username || post.author?.did?.slice(0, 20) + '...' || 'Unknown';
    
    // Handle timestamp (API returns seconds, JS Date needs milliseconds)
    const timestampMs = post.createdAt > 1000000000000 ? post.createdAt : post.createdAt * 1000;
    const time = new Date(timestampMs).toLocaleString();
    
    const votes = `↑${post.upvotes || 0} ↓${post.downvotes || 0}`;
    const replies = post.replies ? ` 💬 ${post.replies}` : '';
    
    // Show title if present
    if (post.title) {
      console.log(`┌─ 📌 ${post.title}`);
    } else {
      console.log(`┌─ ${post.id?.slice(0, 15) || 'Unknown'}... by ${author} | ${votes}${replies}`);
    }
    
    // Show excerpt if present, otherwise content preview
    const previewText = post.excerpt || post.content || '';
    console.log(`│ ${previewText.slice(0, 100)}${previewText.length > 100 ? '...' : ''}`);
    
    // Show author and metadata
    if (post.title) {
      console.log(`│ by ${author} | ${votes}${replies} | ${time}`);
    } else {
      console.log(`└─ ${time}`);
    }
    console.log('');
  }
  
  if (hasMore) {
    if (hot) {
      console.log(`(Page ${page} complete. Use --page ${page + 1} for next page)`);
    } else {
      console.log('(More posts available)');
    }
  }
}

// Main
const args = process.argv.slice(2);
const options = {
  limit: 20,
  following: false,
  topic: null,
  home: false,
  discover: false,
  hot: false,
  page: 1
};

for (let i = 0; i < args.length; i++) {
  if (args[i] === '--limit' && args[i + 1]) {
    options.limit = parseInt(args[i + 1], 10);
    i++;
  } else if (args[i] === '--following') {
    options.following = true;
  } else if (args[i] === '--home') {
    options.home = true;
  } else if (args[i] === '--discover') {
    options.discover = true;
  } else if (args[i] === '--hot') {
    options.hot = true;
  } else if (args[i] === '--page' && args[i + 1]) {
    options.page = parseInt(args[i + 1], 10);
    i++;
  } else if (args[i] === '--topic' && args[i + 1]) {
    options.topic = args[i + 1];
    i++;
  }
}

getFeed(options).catch(err => { console.error('❌', err.message); process.exit(1); });
