#!/usr/bin/env node
/**
 * Update a task
 * 
 * Usage:
 *   node update-task.mjs --task-id "xxx" --title "New Title"
 */

import { parseArgs } from 'util';
import { updateTask } from '../lib/task.mjs';
import { DEFAULT_TIMEZONE } from '../lib/calendar.mjs';

const { values } = parseArgs({
  options: {
    'task-id': { type: 'string' },
    title: { type: 'string' },
    description: { type: 'string' },
    due: { type: 'string' },
    timezone: { type: 'string', default: DEFAULT_TIMEZONE },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Update a Lark task

Options:
  --task-id       Task GUID to update (required)
  --title         New task title
  --description   New task description
  --due           New due time: YYYY-MM-DD HH:MM:SS
  --timezone      IANA timezone (default: Asia/Singapore)
  -h, --help      Show this help

Examples:
  node update-task.mjs --task-id "xxx" --title "Updated Task"
  node update-task.mjs --task-id "xxx" --due "2026-02-06 18:00:00"
`);
  process.exit(0);
}

if (!values['task-id']) {
  console.error('Error: --task-id is required');
  process.exit(1);
}

async function main() {
  try {
    console.log(`Updating task: ${values['task-id']}`);
    
    const task = await updateTask({
      taskId: values['task-id'],
      title: values.title,
      description: values.description,
      dueTime: values.due,
      timezone: values.timezone
    });
    
    console.log('✅ Task updated successfully!');
    console.log('');
    console.log('Updated Task:');
    console.log(`  Title: ${task.summary}`);
    console.log(`  Task ID: ${task.guid}`);
    
  } catch (error) {
    console.error('❌ Failed to update task:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
