#!/usr/bin/env node
/**
 * Update a calendar event
 * 
 * Usage:
 *   node update-event.mjs --event-id "xxx" --title "New Title"
 */

import { parseArgs } from 'util';
import { updateEvent, DEFAULT_CALENDAR_ID, DEFAULT_TIMEZONE } from '../lib/calendar.mjs';

const { values } = parseArgs({
  options: {
    'event-id': { type: 'string' },
    title: { type: 'string' },
    description: { type: 'string' },
    start: { type: 'string' },
    end: { type: 'string' },
    location: { type: 'string' },
    timezone: { type: 'string', default: DEFAULT_TIMEZONE },
    calendar: { type: 'string', default: DEFAULT_CALENDAR_ID },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Update a Lark calendar event

Options:
  --event-id      Event ID to update (required)
  --title         New event title
  --description   New event description
  --start         New start time: YYYY-MM-DD HH:MM:SS
  --end           New end time: YYYY-MM-DD HH:MM:SS
  --location      New event location
  --timezone      IANA timezone (default: Asia/Singapore)
  --calendar      Calendar ID
  -h, --help      Show this help

Examples:
  node update-event.mjs --event-id "xxx" --title "Updated Meeting"
  node update-event.mjs --event-id "xxx" --start "2026-02-03 15:00:00" --end "2026-02-03 16:00:00"
`);
  process.exit(0);
}

if (!values['event-id']) {
  console.error('Error: --event-id is required');
  process.exit(1);
}

async function main() {
  try {
    console.log(`Updating event: ${values['event-id']}`);
    
    const event = await updateEvent({
      eventId: values['event-id'],
      title: values.title,
      description: values.description,
      startTime: values.start,
      endTime: values.end,
      location: values.location,
      timezone: values.timezone,
      calendarId: values.calendar
    });
    
    console.log('✅ Event updated successfully!');
    console.log('');
    console.log('Updated Event:');
    console.log(`  Title: ${event.summary}`);
    console.log(`  Event ID: ${event.event_id}`);
    
  } catch (error) {
    console.error('❌ Failed to update event:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
