#!/usr/bin/env node
/**
 * Add or remove members from a task
 * 
 * Usage:
 *   node manage-task-members.mjs --task-id "xxx" --add "RK,jc"
 *   node manage-task-members.mjs --task-id "xxx" --remove "jc"
 */

import { parseArgs } from 'util';
import { addTaskMembers, removeTaskMembers } from '../lib/task.mjs';
import { resolveNames, getDisplayNameSync } from '../lib/employees.mjs';

const { values } = parseArgs({
  options: {
    'task-id': { type: 'string' },
    add: { type: 'string' },
    remove: { type: 'string' },
    'user-ids': { type: 'string' },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Manage task members/assignees

Options:
  --task-id     Task GUID (required)
  --add         Comma-separated names to add
  --remove      Comma-separated names to remove
  --user-ids    Comma-separated user_ids directly (use with --add or --remove)
  -h, --help    Show this help

Examples:
  node manage-task-members.mjs --task-id "xxx" --add "RK,jc"
  node manage-task-members.mjs --task-id "xxx" --remove "jc"
`);
  process.exit(0);
}

if (!values['task-id']) {
  console.error('Error: --task-id is required');
  process.exit(1);
}

if (!values.add && !values.remove) {
  console.error('Error: Either --add or --remove is required');
  process.exit(1);
}

async function main() {
  try {
    const taskId = values['task-id'];
    
    if (values.add) {
      const names = values.add.split(',').map(s => s.trim()).filter(Boolean);
      const { resolved, unresolved } = resolveNames(names);
      
      if (values['user-ids']) {
        resolved.push(...values['user-ids'].split(',').map(s => s.trim()).filter(Boolean));
      }
      
      if (unresolved.length > 0) {
        console.warn(`Warning: Could not resolve names: ${unresolved.join(', ')}`);
      }
      
      if (resolved.length === 0) {
        console.error('No valid members to add');
        process.exit(1);
      }
      
      console.log(`Adding members: ${resolved.map(id => getDisplayNameSync(id)).join(', ')}`);
      await addTaskMembers(taskId, resolved);
      console.log('✅ Members added successfully!');
    }
    
    if (values.remove) {
      const names = values.remove.split(',').map(s => s.trim()).filter(Boolean);
      const { resolved, unresolved } = resolveNames(names);
      
      if (unresolved.length > 0) {
        console.warn(`Warning: Could not resolve names: ${unresolved.join(', ')}`);
      }
      
      if (resolved.length === 0) {
        console.error('No valid members to remove');
        process.exit(1);
      }
      
      console.log(`Removing members: ${resolved.map(id => getDisplayNameSync(id)).join(', ')}`);
      await removeTaskMembers(taskId, resolved);
      console.log('✅ Members removed successfully!');
    }
    
  } catch (error) {
    console.error('❌ Failed:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
