#!/usr/bin/env node
/**
 * Add or remove attendees from a calendar event
 * 
 * Usage:
 *   node manage-attendees.mjs --event-id "xxx" --add "RK,jc"
 *   node manage-attendees.mjs --event-id "xxx" --remove "jc"
 */

import { parseArgs } from 'util';
import { addEventAttendees, removeEventAttendees, DEFAULT_CALENDAR_ID } from '../lib/calendar.mjs';
import { resolveNames, getDisplayNameSync } from '../lib/employees.mjs';

const { values } = parseArgs({
  options: {
    'event-id': { type: 'string' },
    add: { type: 'string' },
    remove: { type: 'string' },
    'user-ids': { type: 'string' },
    calendar: { type: 'string', default: DEFAULT_CALENDAR_ID },
    help: { type: 'boolean', short: 'h' }
  }
});

if (values.help) {
  console.log(`
Manage calendar event attendees

Options:
  --event-id    Event ID (required)
  --add         Comma-separated names to add
  --remove      Comma-separated names to remove
  --user-ids    Comma-separated user_ids directly (use with --add or --remove)
  --calendar    Calendar ID
  -h, --help    Show this help

Examples:
  node manage-attendees.mjs --event-id "xxx" --add "RK,jc"
  node manage-attendees.mjs --event-id "xxx" --remove "jc"
`);
  process.exit(0);
}

if (!values['event-id']) {
  console.error('Error: --event-id is required');
  process.exit(1);
}

if (!values.add && !values.remove) {
  console.error('Error: Either --add or --remove is required');
  process.exit(1);
}

async function main() {
  try {
    const calendarId = values.calendar;
    const eventId = values['event-id'];
    
    if (values.add) {
      const names = values.add.split(',').map(s => s.trim()).filter(Boolean);
      const { resolved, unresolved } = resolveNames(names);
      
      if (values['user-ids']) {
        resolved.push(...values['user-ids'].split(',').map(s => s.trim()).filter(Boolean));
      }
      
      if (unresolved.length > 0) {
        console.warn(`Warning: Could not resolve names: ${unresolved.join(', ')}`);
      }
      
      if (resolved.length === 0) {
        console.error('No valid attendees to add');
        process.exit(1);
      }
      
      console.log(`Adding attendees: ${resolved.map(id => getDisplayNameSync(id)).join(', ')}`);
      const result = await addEventAttendees(calendarId, eventId, resolved);
      console.log('✅ Attendees added successfully!');
      console.log(`Updated attendees: ${(result.attendees || []).map(a => a.display_name).join(', ')}`);
    }
    
    if (values.remove) {
      const names = values.remove.split(',').map(s => s.trim()).filter(Boolean);
      const { resolved, unresolved } = resolveNames(names);
      
      if (unresolved.length > 0) {
        console.warn(`Warning: Could not resolve names: ${unresolved.join(', ')}`);
      }
      
      if (resolved.length === 0) {
        console.error('No valid attendees to remove');
        process.exit(1);
      }
      
      console.log(`Removing attendees: ${resolved.map(id => getDisplayNameSync(id)).join(', ')}`);
      await removeEventAttendees(calendarId, eventId, resolved);
      console.log('✅ Attendees removed successfully!');
    }
    
  } catch (error) {
    console.error('❌ Failed:', error.message);
    if (error.larkResponse) {
      console.error('Lark response:', JSON.stringify(error.larkResponse, null, 2));
    }
    process.exit(1);
  }
}

main();
